(* File: state.ml *)
(* Author: Chad E Brown *)
(* Created: October 2010 *)

open String
open Flags
open Syntax
open Refut
open Log
open Error
open Minisat

  (*** BEGIN mizartypeinfo ***)
let miztypedebug = ref false
let print_soft_typing_info = false

let unknown_name_sort_p p frl =
  match neg_body p with
  | Some(q) ->
      begin
	match head_spine q with
	| (_,(a::_)) ->
	    let cl = consts_of_trm [] a in
	    begin
	      try
		ignore (List.find
			  (fun (y,_) -> List.mem_assoc y frl)
			  cl);
		true
	      with Not_found -> false
	    end
	| _ -> false
      end
  | None ->
      begin
	match head_spine p with
	| (_,(a::_)) ->
	    let cl = consts_of_trm [] a in
	    begin
	      try
		ignore (List.find
			  (fun (y,_) -> List.mem_assoc y frl)
			  cl);
		true
	      with Not_found -> false
	    end
	| _ -> false
      end

let hounif1negflex = ref false

(*** copied from match.ml to modify for cong clos BEGIN ***)
(*** For Terms with Meta Vars and Meta Type Vars ***)
exception NotGround
exception PatternMatchFailed

let emptyd : (string,trm) Hashtbl.t = Hashtbl.create 1

type metatrm =
  | MGround of trm (*** Assume this is DB, Name, or a logical constant ***)
  | MVar of (int * metatrm option ref) * metatrm list
  | MLam of stp * metatrm
  | MAp of metatrm * metatrm

let mneg m = MAp(MAp(MGround(Imp),m),MGround(False))

let mneg_p m =
  match m with
  | MAp(MAp(MGround(Imp),_),MGround(False)) -> true
  | _ -> false

type ctx = stp list
type dpair = ctx * metatrm * trm * stp
type evar = int * metatrm option ref

let rec id_subst i gamma =
  match gamma with
  | (a::gammar) -> ((MGround(DB(i,a)))::(id_subst (i + 1) gammar))
  | [] -> []

let evarcount = ref 0

let string_of_evar (e,_) =
  "?" ^ (string_of_int e)

let rec new_evar v gamma a : evar * metatrm =
  match a with
  | Ar(a1,a2) ->
      let (x,xs) = new_evar v (a1::gamma) a2 in
      (x,MLam(a1,xs))
  | _ ->
      let x = (!evarcount,ref None) in
      incr evarcount;
      let s = id_subst 0 gamma in
      (x,MVar(x,s))

let rec copy_evar v x : evar =
  let y = (!evarcount,ref None) in
  incr evarcount;
  y

let lpar p = if p then "(" else ""

let rpar p = if p then ")" else ""

let rec metatrm_str_rec m p =
  match m with
  | MGround(m0) -> "<" ^ (trm_str m0) ^ ">"
  | MVar((x,r),s) ->
      begin
	match !r with
	| None ->
	    (lpar p) ^ "?" ^ (string_of_int x) ^ "[" ^ (String.concat "," (List.map (fun m -> metatrm_str_rec m false) s)) ^ "]" ^ (rpar p)
	| Some(m) ->
	    (lpar p) ^ "?" ^ (string_of_int x) ^ "{" ^ (metatrm_str_rec m false) ^ "}[" ^ (String.concat "," (List.map (fun m -> metatrm_str_rec m false) s)) ^ "]" ^ (rpar p)
      end
  | MLam(a,m1) -> (lpar p) ^ "\\_:" ^ (stp_str a) ^ "." ^ (metatrm_str_rec m1 false) ^ (rpar p)
  | MAp(m1,m2) ->
      begin
	match m1 with
	| MLam _ -> (lpar p) ^ (metatrm_str_rec m1 true) ^ " " ^ (metatrm_str_rec m2 true) ^ (rpar p)
	| _ -> (lpar p) ^ (metatrm_str_rec m1 false) ^ " " ^ (metatrm_str_rec m2 true) ^ (rpar p)
      end

let metatrm_str m = metatrm_str_rec m false
  
let rec to_meta m =
  match m with
  | Ap(m1,m2) -> MAp(to_meta m1,to_meta m2)
  | Lam(a,m1) -> MLam(a,to_meta m1)
  | _ -> MGround(m)

let rec meta_to_ground_rec m tau =
  match m with
  | MGround(m1) -> simulsubst m1 tau
  | MVar((e,x),sigma) ->
      begin
	match (!x) with
	| None -> raise NotGround
	| Some n -> meta_to_ground_rec n (List.map (fun p -> meta_to_ground_rec p tau) sigma)
      end
  | MLam(a,m1) -> Lam(a,meta_to_ground_rec m1 ((DB(0,a))::(List.map (fun n -> shift n 0 1) tau)))
  | MAp(m1,m2) -> Ap(meta_to_ground_rec m1 tau,meta_to_ground_rec m2 tau)
	
let meta_to_ground d m =
   norm d (meta_to_ground_rec m [])

let rec metashift m i j =
  match m with
    MGround(DB(k,_)) when k < i -> m
  | MGround(DB(k,a)) -> MGround(DB(k + j,a))
  | MLam(a1,m2) -> MLam(a1,metashift m2 (i + 1) j)
  | MAp(m1,m2) -> MAp(metashift m1 i j,metashift m2 i j)
  | _ -> m

let rec metasubst m i n =
  match m with
    MGround(DB(k,_)) when k < i -> m
  | MGround(DB(k,_)) when k = i -> n
  | MGround(DB(k,a)) -> MGround(DB(k - 1,a))
  | MGround(_) -> m
  | MLam(a1,m2) -> MLam(a1,metasubst m2 (i + 1) (metashift n 0 1))
  | MAp(m1,m2) -> MAp(metasubst m1 i n,metasubst m2 i n)
  | MVar(x,sigma) -> MVar(x,List.map (fun m0 -> metasubst m0 i n) sigma)

let gen_mlam_body a m =
  match m with
  | MLam(_,m1) -> m1
  | MVar(x,sigma) -> raise (Failure ("Metavar of function type? " ^ (string_of_evar x)))
  | _ -> MAp(metashift m 0 1,MGround(DB(0,a)))

let rec metatermNotFreeIn m i =
  match m with
  | MGround(DB(j,_)) when i = j -> false
  | MVar(x,ml) -> metatermNotFreeInL ml i
  | MAp(m1,m2) -> (metatermNotFreeIn m1 i) && (metatermNotFreeIn m2 i)
  | MLam(a,m1) -> metatermNotFreeIn m1 (i + 1)
  | _ -> true
and metatermNotFreeInL ml i =
  match ml with
  | (m::mr) -> (metatermNotFreeIn m i) && (metatermNotFreeInL mr i)
  | [] -> true

let rec meta_simulsubst_meta m tau =
   match m with
   | MGround(DB(k,_)) -> List.nth tau k
   | MGround(m1) -> m
   | MVar((e,x),sigma) ->
       begin
	 match (!x) with
	 | None ->
	     MVar((e,x),List.map (fun p -> meta_simulsubst_meta p tau) sigma)
	 | Some n ->
	     meta_simulsubst_meta n (List.map (fun p -> meta_simulsubst_meta p tau) sigma)
       end
   | MLam(a,m1) -> MLam(a,meta_simulsubst_meta m1 ((MGround(DB(0,a)))::(List.map (fun n -> metashift n 0 1) tau)))
   | MAp(m1,m2) -> MAp(meta_simulsubst_meta m1 tau,meta_simulsubst_meta m2 tau)

let rec metanorm1 m =
  match m with
  | MGround(m1) ->
      let (n1,b) = onlybetanorm1 m1 in
      (MGround(n1),b)
  | MAp(MLam(a,m1),m2) -> (* beta *)
      let (n1,_) = metanorm1 m1 in
      let (n2,_) = metanorm1 m2 in
      (metasubst n1 0 n2,true)
(***  | MLam(_,MAp(m1,MGround(DB(0,_)))) when (metatermNotFreeIn m1 0) -> (* eta *) (*** Removed eta here.  The way dpairs at functional type are handled, they will be more or less eta expanded on the fly anyway.  Hence there is no point in wasting time doing this. - Chad, Jan 10, 2011 ***)
      (metashift m1 0 (- 1),true) ***)
	(*** dneg ***)
  | MAp(MAp(MGround(Imp),MAp(MAp(MGround(Imp),m1),MGround(False))),MGround(False)) -> (* double negation reduce *)
      let (n1,_) = metanorm1 m1 in
      (n1,true)
  | MAp(m1,m2) ->
      let (n1,b1) = metanorm1 m1 in
      let (n2,b2) = metanorm1 m2 in
      if (b1 || b2) then
	(MAp(n1,n2),true)
      else
	(m,false)
  | MLam(a1,m1) ->
      let (n1,b1) = metanorm1 m1 in
      if b1 then
	(MLam(a1,n1),true)
      else
	(m,false)
  | MVar((e,x),sigma) ->
      begin
	match (!x) with
	| None ->
	    let (sigmar,sigmab) = metasubstnorm1 sigma in
	    if sigmab then
	      (MVar((e,x),sigmar),true)
	    else
	      (m,false)
	| Some m1 ->
	    (meta_simulsubst_meta m1 sigma,true)
      end
and metasubstnorm1 sigma =
  match sigma with
  | (m1::sigmar) ->
      let (n1,b1) = metanorm1 m1 in
      let (sigma2,b2) = metasubstnorm1 sigmar in
      if (b1 || b2) then
	(n1::sigma2,true)
      else
	(sigma,false)
  | [] -> ([],false)

(* beta-dneg *)
let rec metanorm m =
  let (m1,reduced) = metanorm1 m in
  if reduced then (metanorm m1) else m

let metanormneg m =
  match m with
  | MAp(MAp(MGround(Imp),m1),MGround(False)) -> m1
  | _ -> MAp(MAp(MGround(Imp),m),MGround(False))

let rec meta_copy m evarassoc =
  begin
    match m with
    | MGround(m1) -> m
    | MVar((e,x),sigma) ->
	let sigma1 = List.map (fun m -> meta_copy m evarassoc) sigma in
	begin
	  match (!x) with
	  | None ->
	      begin
		try
		  let (_,y) = List.find (fun (e1,_) -> (e == e1)) evarassoc in
		  MVar(y,sigma1)
		with
		| Not_found ->
		    MVar((e,x),sigma1)
	      end
	  | Some m1 ->
	      let m1c = meta_copy m1 evarassoc in
	      metanorm (meta_simulsubst_meta m1c sigma1)
	end
    | MLam(a1,m1) ->
	let m1c = meta_copy m1 evarassoc in
	MLam(a1,m1c)
    | MAp(m1,m2) ->
	let m1c = meta_copy m1 evarassoc in
	let m2c = meta_copy m2 evarassoc in
	MAp(m1c,m2c)
  end

let rec distinct_bvar_list_p_rec ml dl =
  match ml with
  | [] -> true
  | ((MGround(DB(i,_)))::mr) when (not (List.mem i dl)) -> distinct_bvar_list_p_rec mr (i::dl)
  | _ -> false

let distinct_bvar_list_p ml =
  distinct_bvar_list_p_rec ml []

let mvar_p m =
  match m with
  | MVar _ -> true
  | _ -> false

let rec pattern_p m =
  match m with
  | MGround(m1) -> true
  | MVar((e,x),ml) ->
      begin
	match (!x) with
	| None -> distinct_bvar_list_p ml
	| Some n ->
	    pattern_p (meta_simulsubst_meta n ml) (*** Mar 2012 - finally wrote this case ***)
      end
  | MLam(a,m1) -> pattern_p m1
  | MAp(m1,m2) -> pattern_p m1 && pattern_p m2

let rec meta_head_spine_rec m args =
  match m with
  | MAp(m1,m2) -> meta_head_spine_rec m1 (m2::args)
  | MGround(m1) -> (m1,tpof m1,args)
  | _ -> raise (Failure "Unexpected case in meta_head_spine_rec")

let meta_head_spine m =
  meta_head_spine_rec m []

let rec occurs_check_p (e,x) m =
  match m with
  | MGround(_) -> false
  | MVar((e',y),_) when (e = e') -> true
  | MVar((e',y),ml) -> occurs_check_list_p (e,x) ml
  | MLam(a,m1) -> occurs_check_p (e,x) m1
  | MAp(m1,m2) -> occurs_check_p (e,x) m1 || occurs_check_p (e,x) m2
and occurs_check_list_p x ml =
  match ml with
  | (m::mr) -> occurs_check_p x m || occurs_check_list_p x mr
  | [] -> false

let rec pattern_invert_db ml n j i k =
  match ml with
  | ((MGround(DB(i2,a)))::mr) when i = i2 -> MGround(DB(j+k,a))
  | (_::mr) -> pattern_invert_db mr n j i (k + 1)
  | [] -> raise PatternMatchFailed

let rec pattern_invert x ml n j =
  match n with
  | DB(i,_) ->
      if (i < j) then
	MGround(n)
      else
	pattern_invert_db ml n j (i - j) 0
  | Lam(a,n1) -> MLam(a,pattern_invert x ml n1 (j + 1))
  | Ap(n1,n2) -> MAp(pattern_invert x ml n1 j,pattern_invert x ml n2 j)
  | _ -> MGround(n)
and pattern_list_invert x ml nl j =
  match nl with
  | (n::nr) -> (pattern_invert x ml n j::pattern_list_invert x ml nr j)
  | [] -> []

let rec pattern_match_rec congforms dl cl =
  match dl with
  | [] -> cl
  | (d::dr) -> pattern_match_rec_1 congforms d dr cl
and pattern_match_rec_1 congforms d dl cl =
  match d with
  | (gamma,m1,n1,Ar(a,b)) ->
      pattern_match_rec_1 congforms (a::gamma,gen_mlam_body a m1,gen_lam_body a n1,b) dl cl
  | (gamma,MVar((e,x),ml),n,b) ->
      begin
	match (!x) with
	| None ->
	    begin
	      if (distinct_bvar_list_p ml) then
		let ni = pattern_invert (e,x) ml n 0 in
		begin
		  x := Some ni;
		  let clr = ref [] in
		  let dlr = ref dl in
		  List.iter
		    (fun ((gammac,mc,nc,b) as c) ->
		      let (mc1,mb) = metanorm1 mc in
		      if mb then
			let mcn = metanorm mc1 in
			dlr := ((gammac,mcn,nc,b)::!dlr)
		      else
			clr := (c::(!clr))
		    )
		    cl;
		  pattern_match_rec congforms (!dlr) (!clr)
		end
	      else
		begin
		  pattern_match_rec congforms dl (d::cl)
		end
	    end
	| Some(mx) -> pattern_match_rec_1 congforms (gamma,metanorm (meta_simulsubst_meta mx ml),n,b) dl cl
      end
  | (gamma,m,n,b) ->
      let (mh,mhtp,ml) = meta_head_spine m in
      try
        let (nh,nl) = head_spine n in
	if (mh = nh) then
	  pattern_match_rec_spine congforms gamma mhtp ml nl dl cl
	else
	  raise PatternMatchFailed
      with PatternMatchFailed -> (*** try harder using congruent forms of n ***)
	pattern_match_rec_2 congforms gamma mh mhtp ml (congforms n) b dl cl
and pattern_match_rec_2 congforms gamma mh mhtp ml ncl b dl cl =
  match ncl with
  | [] -> raise PatternMatchFailed
  | (n::ncr) ->
      try
	let (nh,nl) = head_spine n in
	if (mh = nh) then
	  pattern_match_rec_spine congforms gamma mhtp ml nl dl cl
	else
	  raise PatternMatchFailed
      with PatternMatchFailed -> pattern_match_rec_2 congforms gamma mh mhtp ml ncr b dl cl
and pattern_match_rec_spine congforms gamma tp ml nl dl cl =
  match (tp,ml,nl) with
  | (_,[],[]) -> pattern_match_rec congforms dl cl
  | (Ar(a,b),(m::ml),(n::nl)) -> pattern_match_rec_spine congforms gamma b ml nl ((gamma,m,n,a)::dl) cl
  | _ -> raise PatternMatchFailed
      
let pattern_match congforms dl = pattern_match_rec congforms dl []

let mem_evar (d,a) l =
  try
    ignore(List.find (fun (e,x) -> e = d) l);
    true
  with Not_found -> false

let rec update_strict xl m =
  match m with
  | MGround(_) -> xl
  | MVar(x,sigma) ->
      if (mem_evar x xl) then
	xl
      else if (distinct_bvar_list_p sigma) then
	(x::xl)
      else
	xl
  | MAp(m1,m2) -> update_strict (update_strict xl m1) m2
  | MLam(a1,m1) -> update_strict xl m1

(** HOUNIF1 - Mar 2012 - Chad **)
type udpair = ctx * metatrm * metatrm * stp
type cdpairs = int * evar list * metatrm list * (ctx * metatrm * metatrm * stp) list
type frpairstp = (ctx * (metatrm * (int * metatrm option ref) * metatrm list) *
         (metatrm * Syntax.trm * Syntax.stp * metatrm list) * stp) list
type ffpairstp = (ctx * (metatrm * (int * metatrm option ref) * metatrm list) *
		    (metatrm * (int * metatrm option ref) * metatrm list) * stp) list

exception SimplFailed
exception SimplElim of evar * metatrm * udpair list * frpairstp * ffpairstp

let rec id_subst_2 i gamma =
  match gamma with
  | (a::gammar) -> ((DB(i,a))::(id_subst_2 (i + 1) gammar))
  | [] -> []

let meta_to_ground_2 m tau =
   norm emptyd (meta_to_ground_rec m tau)

let rec spine_dpairs gam a args1 args2 dpairs =
  match (a,args1,args2) with
  | (Ar(a,b),(m1::args1),(m2::args2)) -> spine_dpairs gam b args1 args2 ((gam,m1,m2,a)::dpairs)
  | (_,[],[]) -> dpairs
  | _ -> raise (Failure "Unexpected case in spine_dpairs")

let rec metapattern_invert_db ml n j i k =
  match ml with
  | ((MGround(DB(i2,a)))::mr) when i = i2 -> MGround(DB(j+k,a))
  | (_::mr) -> pattern_invert_db mr n j i (k + 1)
  | [] -> raise PatternMatchFailed

let rec metapattern_invert (e,x) ml n j =
  match n with
  | MGround(DB(i,_)) ->
      if (i < j) then
	n
      else
	metapattern_invert_db ml n j (i - j) 0
  | MGround(_) -> n
  | MLam(a,n1) -> MLam(a,metapattern_invert (e,x) ml n1 (j + 1))
  | MAp(n1,n2) -> MAp(metapattern_invert (e,x) ml n1 j,metapattern_invert (e,x) ml n2 j)
  | MVar((d,y),sl) when e = d -> raise PatternMatchFailed (*** occurs check ***)
  | MVar((d,y),sl) -> MVar((d,y),metapattern_list_invert (e,x) ml sl j)
and metapattern_list_invert x ml nl j =
  match nl with
  | (n::nr) -> (metapattern_invert x ml n j::metapattern_list_invert x ml nr j)
  | [] -> []

let pack_into_dpairs dpairs frpairs ffpairs =
  (List.append (List.map (fun (gam,m1,m2,a) -> (gam,metanorm m1,metanorm m2,a)) dpairs)
     (List.append
	(List.map (fun (gam,(m1,_,_),(m2,_,_,_),a) -> (gam,metanorm m1,metanorm m2,a)) frpairs)
	(List.map (fun (gam,(m1,_,_),(m2,_,_),a) -> (gam,metanorm m1,metanorm m2,a)) ffpairs)))

let rec simpl_dpairs (dpairs:udpair list) xl frpairs ffpairs =
  if (!verbosity > 5) then
    begin
      Printf.printf "simpl_dpairs\n";
      List.iter (fun (gam,m1,m2,b) -> Printf.printf ". %s\nm1 = %s\nm2 = %s\n" (stp_str b) (metatrm_str m1) (metatrm_str m2)) dpairs;
    end;
  match dpairs with
  | (gam,m1,m2,Ar(a,b))::dpairs ->
      if (!verbosity > 5) then begin Printf.printf "simpl_dpairs 0\n" end;
      simpl_dpairs ((a::gam,gen_mlam_body a m1,gen_mlam_body a m2,b)::dpairs) xl frpairs ffpairs
  | (gam,((MVar(x,sl1)) as m1),m2,b)::dpairs when distinct_bvar_list_p sl1 -> (*** pattern ***)
      begin
	try
	  let m2' = metapattern_invert x sl1 m2 0 in
	  raise (SimplElim(x,m2',dpairs,frpairs,ffpairs))
	with PatternMatchFailed ->
	  simpl_dpairs_2 dpairs xl frpairs ffpairs gam m1 m2 b
      end
  | (gam,m1,((MVar(x,sl2)) as m2),b)::dpairs when distinct_bvar_list_p sl2 -> (*** pattern ***)
      begin
	try
	  let m1' = metapattern_invert x sl2 m1 0 in
	  raise (SimplElim(x,m1',dpairs,frpairs,ffpairs))
	with PatternMatchFailed ->
	  simpl_dpairs_2 dpairs xl frpairs ffpairs gam m1 m2 b
      end
  | (gam,m1,m2,b)::dpairs ->
      simpl_dpairs_2 dpairs xl frpairs ffpairs gam m1 m2 b
  | [] -> (frpairs,ffpairs)
and simpl_dpairs_2 dpairs xl frpairs ffpairs gam m1 m2 b =
  if (!verbosity > 5) then begin Printf.printf "simpl_dpairs_2\n" end;
  begin
    match (m1,m2) with
    | (MVar(x1,sigma1),MVar(x2,sigma2)) ->
	simpl_dpairs dpairs xl frpairs ((gam,(m1,x1,sigma1),(m2,x2,sigma2),b)::ffpairs)
    | (_,MVar(x2,sigma2)) ->
	begin
	  match (meta_head_spine m1) with
	  | (h1,h1tp,args1) ->
	      simpl_dpairs dpairs xl ((gam,(m2,x2,sigma2),(m1,h1,h1tp,args1),b)::frpairs) ffpairs
	end
    | (MVar(x1,sigma1),_) ->
	begin
	  match (meta_head_spine m2) with
	  | (h2,h2tp,args2) ->
	      simpl_dpairs dpairs xl ((gam,(m1,x1,sigma1),(m2,h2,h2tp,args2),b)::frpairs) ffpairs
	end
    | (_,MAp(MAp(MGround(Imp),MVar(x2,sigma2)),MGround(False))) when not (mneg_p m1) && !hounif1negflex ->
	simpl_dpairs dpairs xl ((gam,(MVar(x2,sigma2),x2,sigma2),(mneg m1,Imp,Ar(Prop,Ar(Prop,Prop)),[m1;MGround(False)]),b)::frpairs) ffpairs
    | (MAp(MAp(MGround(Imp),MVar(x1,sigma1)),MGround(False)),_) when not (mneg_p m2) && !hounif1negflex ->
	simpl_dpairs dpairs xl ((gam,(MVar(x1,sigma1),x1,sigma1),(mneg m2,Imp,Ar(Prop,Ar(Prop,Prop)),[m2;MGround(False)]),b)::frpairs) ffpairs
    | _ ->
	begin
	  match (meta_head_spine m1,meta_head_spine m2) with
	  | ((h1,h1tp,args1),(h2,h2tp,args2)) ->
	      if (h1 = h2) then
		simpl_dpairs
		  (spine_dpairs gam h1tp args1 args2 dpairs)
		  xl
		  frpairs ffpairs
	      else
		raise SimplFailed
	end
  end

let defaultinsts : (int,metatrm) Hashtbl.t = Hashtbl.create 100
let evargamma : (int,ctx) Hashtbl.t = Hashtbl.create 100
let evarprojs : (int,(int * stp * stp list * stp) list) Hashtbl.t = Hashtbl.create 100

let defaultmeta a =
  match a with
  | Prop -> MGround(False)
  | _ -> MAp(MGround(Choice(a)),MLam(a,MGround(False)))

let rec new_evar_w_default gamma a g =
  match a with
  | Ar(a1,a2) ->
      let (x,xs) = new_evar_w_default (a1::gamma) a2 (fun n -> MLam(a1,g n)) in
      (x,MLam(a1,xs))
  | _ ->
      let x = (!evarcount,ref None) in
      let s = id_subst 0 gamma in
      Hashtbl.add defaultinsts !evarcount (g (defaultmeta a));
      Hashtbl.add evargamma !evarcount gamma;
      let i = ref 0 in
      let projl = ref [] in
      List.iter (fun b ->
	let (argtps,rtp) = argtps_rtp b in
	if (rtp = a) then
	  begin
	    if (!verbosity > 20) then Printf.printf "%d is a possible projection for %s\n" !i (string_of_evar x);
	    projl := (!i,b,argtps,a)::!projl;
	  end;
	incr i;
	)
	gamma;
      Hashtbl.add evarprojs !evarcount !projl;
      incr evarcount;
      (x,MVar(x,s))

let rec force_meta_to_ground_rec m tau =
  if (!verbosity > 5) then begin Printf.printf "force_meta_to_ground_rec m %s\n" (metatrm_str m) end;
   match m with
   | MGround(m1) -> simulsubst m1 tau
   | MVar((e,x),sigma) ->
       begin
	 match (!x) with
	 | None ->
	     let n = Hashtbl.find defaultinsts e in
	     x := Some n;
	     force_meta_to_ground_rec n (List.map (fun p -> force_meta_to_ground_rec p tau) sigma)
	 | Some n ->
	     force_meta_to_ground_rec n (List.map (fun p -> force_meta_to_ground_rec p tau) sigma)
       end
   | MLam(a,m1) -> Lam(a,force_meta_to_ground_rec m1 ((DB(0,a))::(List.map (fun n -> shift n 0 1) tau)))
   | MAp(m1,m2) -> Ap(force_meta_to_ground_rec m1 tau,force_meta_to_ground_rec m2 tau)

let rec imitate gam gam' sigma h a args =
  match (a,args) with
  | (Ar(a1,a2),(arg1::argr)) ->
      let (z,zsub) = new_evar_w_default gam a1 (fun x -> x) in
      let (g,zl,dpairs2) = imitate gam gam' sigma (MAp(h,zsub)) a2 argr in
      (g,(z::zl),((gam',(meta_simulsubst_meta zsub sigma),arg1,a1)::dpairs2))
  | (_,[]) -> (h,[],[])
  | _ -> raise (Failure "Unexpected case in imitate")

let rec project gam h a =
  match a with
  | Ar(a1,a2) ->
      let (z,zsub) = new_evar_w_default gam a1 (fun x -> x) in
      let (g,zl) = project gam (MAp(h,zsub)) a2 in
      (g,(z::zl))
  | _ -> (h,[])

let rec union_evars xl yl =
  match xl with
  | (x::xr) ->
      begin
	try
	  ignore (List.find (fun x' -> x' == x) yl);
	  union_evars xr yl
	with
	  Not_found -> union_evars xr (x::yl)
      end
  | [] -> yl

let rec union_metatrms ml nl =
  match ml with
  | (m::mr) ->
      begin
	if (List.mem m nl) then
	  union_metatrms mr nl
	else
	  union_metatrms mr (m::nl)
      end
  | [] -> nl

let mate_dpairs : (int * cdpairs) list ref = ref [];;
let metaflexposatoms : (evar list * metatrm list * metatrm) list ref = ref [];;
let metaflexnegatoms : (evar list * metatrm list * metatrm) list ref = ref [];;
let metaposatoms : (evar list * metatrm list * metatrm) list ref = ref [];;
let metanegatoms : (evar list * metatrm list * metatrm) list ref = ref [];;

(*** copied from match.ml to modify for cong clos END ***)

type mizartypingaxiom =
  | MizarTypeHierarchyStrict of string * trm * string list * int * trm list * trm list
  | MizarTypeHierarchy of string * trm * string list * int * trm list * trm list
  | MizarFunctionType of string * trm * string list * int * trm list * trm list
  | MizarTypingGeneric of string * trm * int * trm list * trm list
  | MizarRedefinition of string * trm * int * trm list * string * trm list * trm
  | MizarClusterNonempty of string * trm * int * trm list * trm list
let typingaxiomslist : mizartypingaxiom list ref = ref []

let mizar_typing_assumed_lits_h : (int,unit) Hashtbl.t = Hashtbl.create 10
let mizar_pos_typing_predicates : (string,unit) Hashtbl.t = Hashtbl.create 100
let mizar_neg_typing_predicates : (string,unit) Hashtbl.t = Hashtbl.create 100
let mizar_both_typing_predicates : (string,unit) Hashtbl.t = Hashtbl.create 100

let mizar_pattern_table : (string,int * trm list * stp * int list * trm * trm) Hashtbl.t = Hashtbl.create 1000
let mizar_pattern_list : (string * int * trm list * stp * int list * trm * trm) list ref = ref []

let mizar_typing_axiom_lits : int list ref = ref []

let mizar_symbol_order_pred : (string, string) Hashtbl.t ref = ref (Hashtbl.create 1000)
let mizar_symbol_order_succ : (string, string) Hashtbl.t = Hashtbl.create 1000
let mizar_symbol_order_mem : (string * string,unit) Hashtbl.t = Hashtbl.create 1000

let mizar_name_tp : (string,stp) Hashtbl.t ref = ref (Hashtbl.create 1000)

let mizar_func_types : (string,string * int * int * trm list * trm list) Hashtbl.t = Hashtbl.create 20000
let mizar_typing : (string,string * int * int * trm * trm list * trm list) Hashtbl.t = Hashtbl.create 5000
let mizar_typing_gen_forward : (string,string * int * int * trm * trm list * trm list) Hashtbl.t = Hashtbl.create 5000
let mizar_redef : (string,string * int * int * trm list * trm list * trm) Hashtbl.t = Hashtbl.create 5000
let mizar_nonempty_clusters : (string * trm * int * trm list * trm list) list ref = ref []

let add_mizar_symbol_order x y =
  if not (Hashtbl.mem mizar_symbol_order_mem (x,y)) then
    let a u v =
      Hashtbl.add mizar_symbol_order_mem (u,v) ();
      Hashtbl.add mizar_symbol_order_succ u v;
      Hashtbl.add !mizar_symbol_order_pred v u;
    in
    let b u v =
      if not (Hashtbl.mem mizar_symbol_order_mem (x,y)) then
	begin
	  try
	    a u v
	  with Not_found ->
	    a u v
	end
    in
    let zl = Hashtbl.find_all mizar_symbol_order_succ y in
    let wl = Hashtbl.find_all !mizar_symbol_order_pred x in
    a x y;
    List.iter (fun z -> b x z) zl;
    List.iter
      (fun w ->
	b w y;
	List.iter (fun z -> b w z) zl)
      wl

let rec extract_conjuncts m pol =
  match m with
  | Ap(Ap(And,m1),m2) when pol -> extract_conjuncts m1 true @ extract_conjuncts m2 true
  | Ap(Ap(Or,m1),m2) when not pol -> extract_conjuncts m1 false @ extract_conjuncts m2 false
  | Ap(Neg,m1) -> extract_conjuncts m1 (not pol)
  | Ap(Ap(Imp,m1),False) -> extract_conjuncts m1 (not pol)
  | Ap(Ap(Imp,m1),m2) when not pol -> extract_conjuncts m1 true @ extract_conjuncts m2 false
  | False -> if pol then raise Not_found else []
  | True -> if pol then [] else raise Not_found
  | _ -> if pol then [m] else [neg m]

let rec andlist f l =
  match l with
  | x::r -> andlist f r && f x
  | [] -> true

let names_in_mizar_type_assumptions : string list ref = ref []
let mizar_hook_names : string list ref = ref []

let rec fofunstp_p a =
  match a with
  | Base("$i") -> true
  | Ar(Base("$i"),b) -> fofunstp_p b
  | _ -> false

let rec foterm_p m =
  let (h,s) = head_spine m in
  match h with
  | DB(i,Base("$i")) when s = [] -> true
  | Name(f,a) -> fofunstp_p a && foterm_spine_p s
  | _ -> false
and foterm_spine_p s =
  match s with
  | [] -> true
  | m::r -> foterm_p m && foterm_spine_p r

let rec mizar_type_lit_genoa_p c m =
  match m with
  | Ap(Ap(Imp,m1),False) -> mizar_type_lit_genoa_p c m1
  | _ ->
      let (h,s) = head_spine m in
      match h with
      | Name(p,_) ->
	  if foterm_spine_p s then
	    begin
	      names_in_mizar_type_assumptions := p::!names_in_mizar_type_assumptions;
	      c s
	    end
	  else
	    false
      | _ -> false

let rec mizar_type_lit_genoc_p c m =
  match m with
  | Ap(Ap(Imp,m1),False) -> mizar_type_lit_genoc_p c m1
  | _ ->
      let (h,s) = head_spine m in
      match h with
      | Name(p,_) ->
	  if foterm_spine_p s then
	    if c s then
	      if List.exists
		  (fun x -> x = p || Hashtbl.mem mizar_symbol_order_mem (p,x))
		  !names_in_mizar_type_assumptions
	      then
		begin
		  false
		end
	      else
		begin
		  if not (List.mem p !mizar_hook_names) then mizar_hook_names := p::!mizar_hook_names;
		  List.iter (fun x -> add_mizar_symbol_order x p) !names_in_mizar_type_assumptions;
		  true
		end
	    else
	      false
	  else
	    false
      | _ -> false

let rec mizar_type_lit_gen_p c m =
  match m with
  | Ap(Ap(Imp,m1),False) -> mizar_type_lit_gen_p c m1
  | _ ->
      let (h,s) = head_spine m in
      match h with
      | Name(p,_) ->
	  if foterm_spine_p s then
	    c s
	  else
	    false
      | _ -> false

let mizar_type_of_var_p m =
  mizar_type_lit_gen_p
    (fun s ->
      match s with
      | (DB(_,_)::_) -> true
      | _ -> false)
   m

let mizar_type_of_varoa_p m =
  mizar_type_lit_genoa_p
    (fun s ->
      match s with
      | (DB(_,_)::_) -> true
      | _ -> false)
   m

let mizar_type_of_varoc_p m =
  mizar_type_lit_genoc_p
    (fun s ->
      match s with
      | (DB(_,_)::_) -> true
      | _ -> false)
   m

let mizar_type_of_vars_p m =
  mizar_type_lit_gen_p
    (fun s -> andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s)
    m

let mizar_type_of_varsoa_p m =
  mizar_type_lit_genoa_p
    (fun s -> andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s)
    m

let mizar_type_of_result_p m =
  mizar_type_lit_gen_p
    (fun s ->
      match s with
      | (a::_) ->
	  begin
	    match head_spine a with
	    | (Name(_,_),l) -> andlist (fun x -> match x with DB(_,_) -> true | _ -> false) l
	    | _ -> false
	  end
      | [] -> false)
    m

let rec spine_countdown_p l i =
  match l with
  | DB(j,Base("$i"))::r when j = i -> spine_countdown_p r (i-1)
  | [] -> i = (-1)
  | _ -> false

let mizar_type_of_resultoc_p i m =
  if mizar_type_lit_genoa_p (fun _ -> true) m then (*** just to collect the name of type conclusion type as if it's an assumption for dependency checking ***)
    mizar_type_lit_gen_p (*** call gen_ and not genoc_ here because we're not checking that the type in the conclusion comes later than the types in the assumption; but are checking if the type in the conclusion comes before the function ***)
      (fun s ->
	match s with
	| (a::_) ->
	    begin
	      match head_spine a with
	      | (Name(p,_),l) -> (*** insist that l is of the form [DB(i-1,Base("$i"));...;DB(0,Base("$i")] ***)
		  if spine_countdown_p l (i-1) then
		    begin
		      if List.exists
			  (fun x -> x = p || Hashtbl.mem mizar_symbol_order_mem (p,x))
			  !names_in_mizar_type_assumptions
		      then
			false
		      else
			begin
			  if not (List.mem p !mizar_hook_names) then mizar_hook_names := p::!mizar_hook_names;
			  List.iter (fun x -> add_mizar_symbol_order x p) !names_in_mizar_type_assumptions;
			  true
			end
		    end
		  else
		    false
		  (*** More general variant that would require permuting / matching
		  let vo = Array.create i false in (*** check that every var occurs as an arg of the func ***)
		  if andlist (fun x -> match x with DB(j,_) when j < i && j >= 0 -> (vo.(j) <- true; true) | _ -> false) l then
		    let missingvar = ref false in
		    begin
		      for j = 0 to i-1 do
			if not vo.(j) then missingvar := true
		      done;
		      if !missingvar then
			false
		      else if List.exists
			  (fun x -> x = p || Hashtbl.mem mizar_symbol_order_mem (p,x))
			  !names_in_mizar_type_assumptions
		      then
			false
		      else
			begin
			  if not (List.mem p !mizar_hook_names) then mizar_hook_names := p::!mizar_hook_names;
			  List.iter (fun x -> add_mizar_symbol_order x p) !names_in_mizar_type_assumptions;
			  true
			end
		    end
		  else
		     false
		     ***)
	      | _ -> false
	    end
	| [] -> false)
      m
  else
    false
      
let rec mizar_type_fact1_p m i pl =
  if !miztypedebug then Printf.printf "fact1_p %s %d %d\n" (trm_str m) i (List.length pl);
  match m with
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact1b_p m1 (i+1) (List.map (fun p -> shift p 0 1) pl)
  | _ ->
      let ml = extract_conjuncts m true in
      if !miztypedebug then (Printf.printf "fact1_p ml:\n"; List.iter (fun q -> Printf.printf "%s\n" (trm_str q)) ml);
      if andlist (mizar_type_lit_gen_p (fun _ -> true)) ml then
	(i,pl,ml)
      else
	raise Not_found
and mizar_type_fact1b_p m i pl =
  if !miztypedebug then Printf.printf "fact1b_p %s %d %d\n" (trm_str m) i (List.length pl);
  match m with
  | Ap(Ap(Imp,m1),False) ->
      let ml = extract_conjuncts m1 false in
      if !miztypedebug then (Printf.printf "fact1b_p case 1 ml:\n"; List.iter (fun q -> Printf.printf "%s\n" (trm_str q)) ml);
      if andlist (mizar_type_lit_gen_p (fun _ -> true)) ml then
	(i,pl,ml)
      else
	raise Not_found
  | Ap(Ap(Imp,m1),m2) ->
      let ql = extract_conjuncts m1 true in
      if !miztypedebug then (Printf.printf "fact1b_p case 2 ml:\n"; List.iter (fun q -> Printf.printf "%s\n" (trm_str q)) ql);
      if andlist mizar_type_of_var_p ql then
	mizar_type_fact1b_p m2 i (ql @ pl)
      else
	raise Not_found
  | _ -> mizar_type_fact1_p m i pl

let rec radixof_n_p m n =
  match m with
  | Ap(Name(x,_),DB(0,_)) -> true
  | Ap(m1,DB(k,_)) when k = n -> radixof_n_p m1 (n+1)
  | _ -> false

let radixof_p m =
  match m with
  | Ap(Name(x,_),DB(0,_)) -> true
  | Ap(m1,DB(n,_)) when n = 1 -> radixof_n_p m1 (n+1)
  | _ -> false

let rec moreforalls_p m =
  match m with
  | Ap(Forall(_),_) -> true
  | Ap(Ap(Imp,m1),m2) -> moreforalls_p m2
  | _ -> false

let rec mizar_type_fact2_p m =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,Ap(Ap(Imp,m1),m2))) when radixof_p m1 && not (moreforalls_p m2) ->
      let ml = extract_conjuncts m2 true in
      andlist mizar_type_of_var_p ml
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact2b_p m1
  | _ -> false
and mizar_type_fact2b_p m =
  match m with
  | Ap(Ap(Imp,m1),m2) -> andlist mizar_type_of_var_p (extract_conjuncts m1 true) && mizar_type_fact2b_p m2
  | _ -> mizar_type_fact2_p m

let rec mizar_type_fact2o_p i m =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,Ap(Ap(Imp,m1),m2))) when radixof_p m1 && not (moreforalls_p m2) ->
      let ml = extract_conjuncts m2 true in
      if andlist mizar_type_of_varoa_p ml then
	begin
	  let (h,s) = head_spine m1 in (*** since radixof_p m1 holds, we know s is all db's in the order 0,k,...,1 so to know all vars occur it is enough to check that k = i (or no k if i = 0) ***)
	  begin
	    try
	      begin
		match s with
		| [_] when i > 0 -> raise Exit
		| (_::DB(k,_)::_) when not (i = k) -> raise Exit
		| _ ->
		    match h with
		    | Name(p,_) ->
			if List.exists
			    (fun x -> x = p || Hashtbl.mem mizar_symbol_order_mem (p,x))
			    !names_in_mizar_type_assumptions
			then
			  begin
			    false
			  end
			else
			  begin
			    if not (List.mem p !mizar_hook_names) then mizar_hook_names := p::!mizar_hook_names;
			    List.iter (fun x -> add_mizar_symbol_order x p) !names_in_mizar_type_assumptions;
			    true
			  end
		    | _ -> false
	      end
	    with Exit -> false
	  end
	end
      else
	false
  | Ap(Forall(Base("$i")),Lam(_,m1)) ->
      mizar_type_fact2ob_p (i+1) m1
  | _ -> false
and mizar_type_fact2ob_p i m =
  match m with
  | Ap(Ap(Imp,m1),m2) -> andlist mizar_type_of_varoa_p (extract_conjuncts m1 true) && mizar_type_fact2ob_p i m2
  | _ -> mizar_type_fact2o_p i m
	
let rec mizar_type_fact3_p m =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,Ap(Ap(Imp,m1),m2))) when radixof_p m1 && not (moreforalls_p m2) ->
      let ml = extract_conjuncts m2 true in
      andlist mizar_type_of_var_p ml
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact3b_p m1
  | _ -> false
and mizar_type_fact3b_p m =
  match m with
  | Ap(Ap(Imp,m1),m2) -> andlist mizar_type_of_vars_p (extract_conjuncts m1 true) && mizar_type_fact3b_p m2
  | _ -> mizar_type_fact3_p m

let rec mizar_type_fact3o_p m =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,Ap(Ap(Imp,m1),m2))) when radixof_p m1 && not (moreforalls_p m2) ->
      let ml = extract_conjuncts m2 true in
      if andlist mizar_type_of_varoa_p ml then
	begin
	  let (h,_) = head_spine m1 in (*** TODO: I should also check that the m1 has all the DB's of the fact so that matching will instantiate all vars. ***)
	  match h with
	  | Name(p,_) ->
	      if List.exists
		  (fun x -> x = p || Hashtbl.mem mizar_symbol_order_mem (p,x))
		  !names_in_mizar_type_assumptions
	      then
		begin
		  false
		end
	      else
		begin
		  if not (List.mem p !mizar_hook_names) then mizar_hook_names := p::!mizar_hook_names;
		  List.iter (fun x -> add_mizar_symbol_order x p) !names_in_mizar_type_assumptions;
		  true
		end
	  | _ -> false
	end
      else
	false
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact3ob_p m1
  | _ -> false
and mizar_type_fact3ob_p m =
  match m with
  | Ap(Ap(Imp,m1),m2) -> andlist mizar_type_of_varsoa_p (extract_conjuncts m1 true) && mizar_type_fact3ob_p m2
  | _ -> mizar_type_fact3o_p m
	
let rec mizar_type_fact4_p m =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact4b_p m1
  | _ ->
      let ml = extract_conjuncts m true in
      andlist mizar_type_of_result_p ml
and mizar_type_fact4b_p m =
  match m with
  | Ap(Ap(Imp,m1),False) ->
      let ml = extract_conjuncts m1 false in
      andlist mizar_type_of_result_p ml
  | Ap(Ap(Imp,m1),m2) -> andlist mizar_type_of_var_p (extract_conjuncts m1 true) && mizar_type_fact4b_p m2
  | _ -> mizar_type_fact4_p m

let rec mizar_type_fact4oa_p i m =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact4ob_p (i+1) m1
  | _ ->
      let ml = extract_conjuncts m true in
      andlist (mizar_type_of_resultoc_p i) ml
and mizar_type_fact4ob_p i m =
  match m with
  | Ap(Ap(Imp,m1),False) ->
      let ml = extract_conjuncts m1 false in
      andlist (mizar_type_of_resultoc_p i) ml
  | Ap(Ap(Imp,m1),m2) -> andlist mizar_type_of_varoa_p (extract_conjuncts m1 true) && mizar_type_fact4ob_p i m2
  | _ -> mizar_type_fact4oa_p i m

let mizar_type_fact4o_p m = mizar_type_fact4oa_p 0 m
	
let mizar_abbrev_eqn m =
  match m with
  | Ap(Ap(Eq(_),m1),m2) ->
      begin
	let (h1,s1) = head_spine m1 in
	let (h2,s2) = head_spine m2 in
	match (h1,h2) with
	| (Name(f1,_),Name(f2,_)) when not (f1 = f2) ->
	    if andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s1
		&&
	      andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s2
		&&
	      List.for_all (fun x -> List.mem x s1) s2
	    then
	      (f1,s1,m2)
	    else
	      raise Not_found
	| (Name(f1,_),Eq(_)) ->
	    if andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s1
		&&
	      andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s2
		&&
	      List.for_all (fun x -> List.mem x s1) s2
	    then
	      (f1,s1,m2)
	    else
	      raise Not_found
	| _ -> raise Not_found
      end
  | _ -> raise Not_found

let mizar_abbrev_eqn_o m =
  match m with
  | Ap(Ap(Eq(_),m1),m2) ->
      begin
	let (h1,s1) = head_spine m1 in
	let (h2,s2) = head_spine m2 in
	match (h1,h2) with
	| (Name(f1,_),Name(f2,_)) when not (f1 = f2) ->
	    if andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s1
		&&
	      andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s2
		&&
	      List.for_all (fun x -> List.mem x s1) s2
	    then
	      begin
		names_in_mizar_type_assumptions := f2::!names_in_mizar_type_assumptions;
		if List.exists
		    (fun x -> x = f1 || Hashtbl.mem mizar_symbol_order_mem (f1,x))
		    !names_in_mizar_type_assumptions
		then
		  raise Not_found
		else
		  begin
		    List.iter (fun x -> add_mizar_symbol_order x f1) !names_in_mizar_type_assumptions;
		    (f1,s1,m2)
		  end
	      end
	    else
	      raise Not_found
	| (Name(f1,_),Eq(_)) ->
	    if andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s1
		&&
	      andlist (fun x -> match x with DB(_,_) -> true | _ -> false) s2
		&&
	      List.for_all (fun x -> List.mem x s1) s2
	    then
	      if List.exists
		  (fun x -> x = f1 || Hashtbl.mem mizar_symbol_order_mem (f1,x))
		  !names_in_mizar_type_assumptions
	      then
		raise Not_found
	      else
		begin
		  List.iter (fun x -> add_mizar_symbol_order x f1) !names_in_mizar_type_assumptions;
		  (f1,s1,m2)
		end
	    else
	      raise Not_found
	| _ -> raise Not_found
      end
  | _ -> raise Not_found

let rec mizar_type_fact5_p m i pl =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact5b_p m1 (i+1) (List.map (fun p -> shift p 0 1) pl)
  | _ ->
      let (f1,s1,m2) = mizar_abbrev_eqn m in
      (i,pl,f1,s1,m2)
and mizar_type_fact5b_p m i pl =
  match m with
  | Ap(Ap(Imp,m1),False) -> raise Not_found
  | Ap(Ap(Imp,m1),m2) ->
      let ql = extract_conjuncts m1 true in
      if andlist mizar_type_of_var_p ql then
	mizar_type_fact5b_p m2 i (pl @ ql)
      else
	raise Not_found
  | _ -> mizar_type_fact5_p m i pl

let rec mizar_type_fact5o_p m i pl =
  match m with
  | Ap(Forall(Base("$i")),Lam(_,m1)) -> mizar_type_fact5ob_p m1 (i+1) (List.map (fun p -> shift p 0 1) pl)
  | _ ->
      let (f1,s1,m2) = mizar_abbrev_eqn_o m in
      (i,pl,f1,s1,m2)
and mizar_type_fact5ob_p m i pl =
  match m with
  | Ap(Ap(Imp,m1),False) -> raise Not_found
  | Ap(Ap(Imp,m1),m2) ->
      let ql = extract_conjuncts m1 true in
      if andlist mizar_type_of_varoa_p ql then
	mizar_type_fact5ob_p m2 i (pl @ ql)
      else
	raise Not_found
  | _ -> mizar_type_fact5o_p m i pl

let rec mizar_pattern m i pl =
(*  Printf.printf "mizar_pattern %d %s\n" i (trm_str m); flush stdout; *)
  match m with
  | Ap(Forall(Base("$i")),Lam(_,m1)) ->
      mizar_pattern m1 (i+1) (List.map (fun p -> shift p 0 1) pl)
  | Ap(Ap(Imp,m1),m2) ->
      let ql = extract_conjuncts m1 true in
      if andlist mizar_type_of_var_p ql then
	mizar_pattern m2 i (pl @ ql)
      else
	raise Not_found
  | Ap(Ap(Eq(rtp),m1),m2) ->
      begin
	let (h,al) = head_spine m1 in
	match h with
	| Name(h1,_) ->
	    let al2 = List.map (fun n -> match n with DB(j,a) when a = Base("$i") -> j | _ -> raise Not_found) al in
	    (h1,i,pl,rtp,al2,m1,m2)
	| _ -> raise Not_found
      end
  | _ -> raise Not_found

	(*** elaboration ***)
let istp = Base("$i")
let oo = Ar(Prop,Prop)
let ooo = Ar(Prop,oo)
let predisj x y = Ap(Ap(Or,x),y)
let preconj x y = Ap(Ap(And,x),y)
let preiff x y = Ap(Ap(Iff,x),y)

let premneg x = MAp(MGround(Neg),x)
let meq a x y = MAp(MAp(MGround(Eq(a)),x),y)

let nxtfreshvar a st =
  let (_,_,sh,_,_,(_,_,varcnt),_) = st in
  incr varcnt;
  let en = "#" ^ string_of_int !varcnt in
  let e = Name(en,a) in
  if !miztypedebug then Printf.printf "#+#++ Created fresh var %s\n" en;
  e

let rec abstract_subterms i x m =
  if x = m then
    DB(i,tpof x)
  else
    match m with
    | Ap(m1,m2) -> Ap(abstract_subterms i x m1,abstract_subterms i x m2)
    | Lam(a,m1) -> Lam(a,abstract_subterms (i+1) x m1)
    | _ -> m

exception ElaborationSimpleTypeError of trm * stp * stp
exception ElaborationPatternArityError of string * int * int
exception ElaborationMizarTypeConflict of trm * trm
exception ElaborationFailure of string

let rec pre_head_spine m ml =
  match m with
  | PAp(m1,m2) -> pre_head_spine m1 (m2::ml)
  | _ -> (m,ml)

let emptydelta : (string,trm) Hashtbl.t = Hashtbl.create 1

let mhead_spine m =
  let rec mhead_spine_r m s =
    match m with
      MAp(m1,m2) -> mhead_spine_r m1 (m2::s)
    | MGround(n) -> (n,s)
    | _ -> raise Not_found
  in
  mhead_spine_r m []

let mneg_body m =
  match m with
  | MAp(MGround(Neg),n) -> Some(m)
  | MAp(MAp(MGround(Imp),n),MGround(False)) -> Some(m)
  | _ -> None

let extract_mizar_type_mprop_atom p =
  match mneg_body p with
  | Some(n) ->
      begin
	let (h,s) = mhead_spine n in
	match s with
        | (a::r) -> (false,a,h,r)
        | _ -> raise Not_found
      end
  | None ->
      begin
	let (h,s) = mhead_spine p in
	match s with
        | (a::r) -> (true,a,h,r)
        | _ -> raise Not_found
      end

let extract_mizar_type_prop_atom p =
  match neg_body p with
  | Some(n) ->
      begin
	let (h,s) = head_spine n in
	match s with
        | (a::r) -> (false,a,h,r)
        | _ -> raise Not_found
      end
  | None ->
      begin
	let (h,s) = head_spine p in
	match s with
        | (a::r) -> (true,a,h,r)
        | _ -> raise Not_found
      end

let rec mizar_type_fact6_p p i pl =
  match p with
  | Ap(Forall(Base(a)),Lam(_,p)) when a = "$i" ->
      mizar_type_fact6_p p (i+1) (List.map (fun p -> shift p 0 1) pl)
  | Ap(Ap(Imp,Ap(Forall(Base(a)),Lam(_,q))),False) when a = "$i" ->
      begin
	let ql = extract_conjuncts q false in
	try
	  ignore (List.find
		    (fun c ->
		      let (pol,a,m,r) = extract_mizar_type_prop_atom c in (*** if it's marked as a variable, then just assign it the type ***)
		      not (a = DB(0,Base("$i"))))
		    ql);
	  raise Exit
	with
	| Not_found -> (i,pl,ql)
	| Exit -> raise Not_found
      end
  | Ap(Ap(Imp,Ap(Forall(Base(a)),Name(x,b))),False) when a = "$i" && b = Ar(Base("$i"),Prop) -> (** eta short **)
      begin
	let ql = [neg (Ap(Name(x,b),DB(0,Base("$i"))))] in
	(i,pl,ql)
      end
  | Ap(Ap(Imp,p1),p2) ->
      begin
	let pl1 = extract_conjuncts p1 false in
	if andlist mizar_type_of_var_p pl1 then
	  mizar_type_fact6_p p2 i (pl @ pl1)
	else
	  raise Not_found
      end
  | _ -> raise Not_found

exception MizarRuleDoesNotApply

    (*** Following Nelson Oppen 1978 Congruence Closure Paper ***)
type cctables = (trm,unit) Hashtbl.t * (trm,trm) Hashtbl.t * (trm,trm) Hashtbl.t * (trm,trm) Hashtbl.t

let print_cctables cc =
  let (ccv,ccp,ccb,ccf) = cc in
  Printf.printf "===== CC ======\n";
  Printf.printf "===== handled ======\n";
  Hashtbl.iter (fun k () -> Printf.printf "%s\n" (trm_str k)) ccv;
  Printf.printf "===== pred ======\n";
  Hashtbl.iter (fun k v -> Printf.printf "%s  ~~>  %s\n" (trm_str k) (trm_str v)) ccp;
  Printf.printf "===== back ======\n";
  Hashtbl.iter (fun k v -> Printf.printf "%s  ~~>  %s\n" (trm_str k) (trm_str v)) ccb;
  Printf.printf "===== find ======\n";
  Hashtbl.iter (fun k v -> Printf.printf "%s  ~~>  %s\n" (trm_str k) (trm_str v)) ccf


let rec mizar_cong_find ccf m =
  try
    let r = mizar_cong_find ccf (Hashtbl.find ccf m) in
    Hashtbl.replace ccf m r;
    r
  with Not_found ->
    m

let mizar_cong_union (ccb,ccf) m n =
  let m1 = mizar_cong_find ccf m in
  let n1 = mizar_cong_find ccf n in
  Hashtbl.add ccf m1 n1;
  Hashtbl.add ccb n1 m1

let mizar_cong_class ccb m =
  let mh : (trm,unit) Hashtbl.t = Hashtbl.create 10 in
  let rec mizar_cong_class_1 m =
    if not (Hashtbl.mem mh m) then
      begin
	Hashtbl.add mh m ();
	List.iter (fun n -> mizar_cong_class_1 n) (Hashtbl.find_all ccb m)
      end
  in
  mizar_cong_class_1 m;
  let ml = ref [] in
  Hashtbl.iter (fun m () -> ml := m::!ml) mh;
  !ml

let mizar_cong_preds (ccp,ccb) m =
  let mh : (trm,unit) Hashtbl.t = Hashtbl.create 10 in
  let ph : (trm,unit) Hashtbl.t = Hashtbl.create 10 in
  let rec mizar_cong_preds_1 m =
    if not (Hashtbl.mem mh m) then
      begin
	List.iter (fun p -> if not (Hashtbl.mem ph p) then Hashtbl.add ph p ()) (Hashtbl.find_all ccp m);
	Hashtbl.add mh m ();
	List.iter (fun n -> mizar_cong_preds_1 n) (Hashtbl.find_all ccb m)
      end
  in
  mizar_cong_preds_1 m;
  let pl = ref [] in
  Hashtbl.iter (fun p () -> pl := p::!pl) ph;
  !pl

let rec mizar_cong_merge (ccp,ccb,ccf) m n =
  if !miztypedebug then Printf.printf "merging %s with %s\n" (trm_str m) (trm_str n);
  let m1 = mizar_cong_find ccf m in
  let n1 = mizar_cong_find ccf n in
  if m1 = n1 then
    ()
  else
    let pml = mizar_cong_preds (ccp,ccb) m1 in
    let pnl = mizar_cong_preds (ccp,ccb) n1 in
    mizar_cong_union (ccb,ccf) m n;
    List.iter
      (fun p ->
	let p1 = mizar_cong_find ccf p in
	(List.iter
	   (fun q ->
	     let q1 = mizar_cong_find ccf q in
	     if not (p1 = q1) && mizar_cong_congruent (ccp,ccb,ccf) p q then
	       mizar_cong_merge (ccp,ccb,ccf) p q)
	   pnl))
      pml
and mizar_cong_congruent (ccp,ccb,ccf) m n =
  let (mh,ms) = head_spine m in
  let (nh,ns) = head_spine n in
  if mh = nh then
    begin
      try
	if not (List.length ms = List.length ns) then
	  begin
	    Printf.printf "#+#++ mizar_cong_congruent iter2 |ms| = %d ; |ns| = %d ;\nm = %s\nn = %s\n" (List.length ms) (List.length ns) (trm_str m) (trm_str n);
	    raise Exit
	  end
	else
	  List.iter2
	    (fun mi ni ->
	      let mi1 = mizar_cong_find ccf mi in
	      let ni1 = mizar_cong_find ccf ni in
	      if not (mi1 = ni1) then raise Exit)
	    ms ns;
	true
      with Exit -> false
    end
  else
    false

let rec extend_cong_graph (ccv,ccp,ccb,ccf) m =
  if not (Hashtbl.mem ccv m) then
    begin
      if !miztypedebug then Printf.printf "extend_cong_graph %s\n" (trm_str m);
      let (mh,s) = head_spine m in
      List.iter
	(fun n ->
	  extend_cong_graph (ccv,ccp,ccb,ccf) n;
	  if !miztypedebug then Printf.printf "back from handling arg %s to extend_cong_graph %s\n" (trm_str n) (trm_str m);
	  let pnl = mizar_cong_preds (ccp,ccb) (mizar_cong_find ccf n) in
	  List.iter
	    (fun p ->
	      if !miztypedebug then Printf.printf "pred %s in extend_cong_graph %s\n" (trm_str p) (trm_str m);
	      let p1 = mizar_cong_find ccf p in
	      if !miztypedebug then Printf.printf "pred find %s\n" (trm_str p1);
	      let m1 = mizar_cong_find ccf m in
	      if !miztypedebug then Printf.printf "m find %s\n" (trm_str m1);
	      if not (p1 = m1) && mizar_cong_congruent (ccp,ccb,ccf) p m then
		mizar_cong_merge (ccp,ccb,ccf) p m)
	    pnl;
	  Hashtbl.add ccp n m)
	s;
      Hashtbl.add ccv m ()
    end

let mizar_cong cc m1 m2 =
  extend_cong_graph cc m1;
  extend_cong_graph cc m2;
  let (_,_,_,ccf) = cc in
  let n1 = mizar_cong_find ccf m1 in
  let n2 = mizar_cong_find ccf m2 in
  if !miztypedebug then
    begin
      if not (m1 = n1) then Printf.printf "cong_find\n   %s\n ~> %s\n" (trm_str m1) (trm_str n1);
      if not (m2 = n2) then Printf.printf "cong_find\n   %s\n ~> %s\n" (trm_str m2) (trm_str n2);
    end;
  n1 = n2
	
type mizstate = (pretrm,trm * stp * ((string * (trm list)) list)) Hashtbl.t * (trm,trm) Hashtbl.t * (trm,trm) Hashtbl.t * (trm,unit) Hashtbl.t * cctables * ((string,unit) Hashtbl.t * bool ref * int ref) * ((trm -> unit) ref)

let rec append_consis ml nl =
  match ml with
  | m::mr ->
      let nm = normneg m in
      if List.mem nm nl then
	raise (ElaborationFailure(Printf.sprintf "assigned + and - sort %s" (trm_str m)))
      else
	append_consis mr (m::nl)
  | [] -> nl

let rec adjoin_frees x ml frl =
  match frl with
  | [] -> [(x,ml)]
  | ((y,nl)::frr) when x = y -> ((x,append_consis ml nl)::frr)
  | ((y,nl)::frr) -> (y,nl)::adjoin_frees x ml frr

let rec union_frees frl1 frl2 =
  match frl1 with
  | [] -> frl2
  | ((x,ml)::frr1) -> union_frees frr1 (adjoin_frees x ml frl2)
  (*** END mizartypeinfo ***)

exception CoqProofTooBig of int

let clauses : clause list ref = ref []
let clausesTable : (clause,unit) Hashtbl.t = Hashtbl.create 10

let slaveargs = ref [Sys.argv.(0)];;
let mode : string list ref = ref []
let timeout : float option ref = ref None
let hardtimeout : float option ref = ref None
let nontheorem : bool ref = ref false (*** March 2012 - Know if you're explicitly trying to determine Satisfiable/CounterSatisfiable ***)
let coq = ref false
let coq2 = ref false
let problemfile = ref ""
let coqlocalfile = ref false
let coqglobalfile = ref false
let coqinchannel : in_channel ref = ref stdin
let coqoutchannel : out_channel ref = ref stdout
let coqinticks : ((out_channel -> unit) option * int) list ref = ref []
let coqoutfn1 = ref (fun c -> ())
let coqctx : (string option * pretrm option * pretrm option) list ref = ref []
let coqglobalsectionstack : (string * (out_channel -> unit) * (string option * pretrm option * pretrm option) list) list ref = ref []
let recognize_hashroots = ref false

let training_file : string option ref = ref None

type probitem =
  | ProbSoftTyping of mizartypingaxiom
  | ProbDef of string * stp * trm * (string * string) list * float
  | ProbAx of string * string * trm * (string * string) list * float
  | ProbConj of string * trm * (string * string) list * float
let probsig : probitem list ref = ref []

let rec updatecoqglobalsectionstack cx cgss co =
  match cgss with
  | ((secname,cfn,lcx)::cgss') -> ((secname,co cx cfn,lcx)::(updatecoqglobalsectionstack (List.append cx lcx) cgss' co))
  | [] -> []

let conjecturename : string ref = ref "claim"
let conjecture : (trm * trm) option ref = ref None
type proofkind = TSTP | CoqScript | CoqSPfTerm | HOCore | Model | ModelTrue | IsarScript | PfInfo | PfUseful | PfFormdeps
let mkproofterm = ref None
let mkprooftermp () = Utils.bool_of_option !mkproofterm || Utils.bool_of_option !training_file
let pfusefulout = ref None
let pfformdepsout = ref None
let slave = ref false
let coqsig_base : string list ref = ref []
let coqsig_const : (string * stp) list ref = ref []
let coqsig_def : (string * pretrm) list ref = ref []
let coqsig_def_trm : (string * trm) list ref = ref []
let coqsig_hyp : (string * pretrm) list ref = ref []
let coqsig_hyp_trm : (string * trm) list ref = ref []
let name_base : (string,unit) Hashtbl.t = Hashtbl.create 10
let name_base_list : string list ref = ref []
let name_tp : (string,stp) Hashtbl.t = Hashtbl.create 100
let name_trm : (string,(trm * stp) * bool ref) Hashtbl.t = Hashtbl.create 100
let name_trm_list : (string * trm * stp) list ref = ref []
let translucent_defns : bool ref = ref false
let name_def_empty : (string,trm) Hashtbl.t = Hashtbl.create 1
let name_def : (string,trm) Hashtbl.t = Hashtbl.create 100
let name_def_all : (string,trm) Hashtbl.t = Hashtbl.create 100
let name_def_prenorm : (string,trm) Hashtbl.t = Hashtbl.create 100
let name_hyp : (string,trm) Hashtbl.t = Hashtbl.create 100
let name_hyp_inv : (trm,string * trm) Hashtbl.t = Hashtbl.create 100 (*** associate a normalized/logic normalized term with the name of the assumption it came from and its original prenormalized form ***)
let assumption_lit : (int,trm * trm) Hashtbl.t = Hashtbl.create 100 (*** associate assumption literals with their term after preprocessing and before preprocessing ***)

(** Replaces all occurences of 'Neg' by 'implies False' **)
let rec negnorm1 m =
  match m with
  | Ap(Neg,m1) ->
      let (n1,_) = negnorm1 m1 in
      (imp n1 False,true)
  | Neg -> (Lam(Prop,imp (DB(0,Prop)) False),true)
  | Ap(m1,m2) ->
      let (n1,b1) = negnorm1 m1 in
      let (n2,b2) = negnorm1 m2 in
      if (b1 || b2) then
	(Ap(n1,n2),true)
      else
	(m,false)
  | Lam(a1,m1) ->
      let (n1,b1) = negnorm1 m1 in
      if b1 then
	(Lam(a1,n1),true)
      else
	(m,false)
  | _ -> (m,false)
(** applies neg- and betanormalization**)
let onlynegnorm m =
  let (n,_) = negnorm1 m in onlybetanorm n
(** applies neg-, beta- and delta-normalization**)
let coqnorm m =
  let m = betanorm name_def_prenorm m in
  let (n,_) = negnorm1 m in n
(** partially normalizes, without expanding certain defns (e.g., is_of) **)
let partialnormalize pt = norm name_def (logicnorm pt)
(** applies full satallax normalization**)
let normalize pt = norm name_def_all (logicnorm pt)
let belnorm pt = norm name_def_empty (logicnorm pt)

let coqknown (x,y) =
  if (!coq2) then y else x

let mult_timeout f =
  match (!timeout) with
  | Some tm -> timeout := Some (tm *. f)
  | None -> ()

let requireSet0a () =
  let a = Base "set" in
  let b = PName "set" in
  Hashtbl.add coq_used_names "In" ();
  Hashtbl.add coq_used_names "Subq" ();
  Hashtbl.add coq_names "In" "In";
  Hashtbl.add coq_names "Subq" "Subq";
  coqsig_const := ("In",Ar(a,Ar(a,Prop)))::!coqsig_const;
  coqsig_const := ("Subq",Ar(a,Ar(a,Prop)))::!coqsig_const;
  coqsig_def := ("Subq",PLam([("x",b);("y",b)],PAll(["z",b],PAp(PAp(PImplies,PAp(PAp(PName "In",PName "z"),PName "x")),PAp(PAp(PName "In",PName "z"),PName "y")))))::!coqsig_def;
  coqsig_def_trm := ("Subq",Lam(a,Lam(a,Ap(Forall(a),Lam(a,Ap(Ap(Imp,Ap(Ap(Name("In",Ar(a,Ar(a,Prop))),DB(0,a)),DB(2,a))),Ap(Ap(Name("In",Ar(a,Ar(a,Prop))),DB(0,a)),DB(1,a))))))))::!coqsig_def_trm;
  ()

let required : string ref = ref ""

let require x =
  if (!verbosity > 5) then Printf.printf "Requiring %s\n" x;
  required := x;
  let f = !coqoutfn1 in
  begin
    coqoutfn1 := (fun c -> f c; Printf.fprintf c "Require Export %s.\n" x);
    match x with
    | "set0a" -> requireSet0a()
    | "set0" -> raise (GenericError "set0 is not yet supported.")
    | "set1" -> raise (GenericError "set1 is not yet supported.")
    | _ -> ()
  end

(*** March 31, 2011 - Chad - THF policy regarding definitions. (See comments before declare_definition_real below. ***)
exception TreatAsAssumption

let next_fresh_name : int ref = ref 0

let rec get_fresh_name a =
  let x = make_fresh_name !next_fresh_name in
  incr next_fresh_name;
  if !coq then ignore (coqify_name x coq_names coq_used_names);
  if ((Hashtbl.mem name_base x) || (Hashtbl.mem name_tp x)) then
    get_fresh_name a
  else
    let xa = Name(x,a) in
    begin
      Hashtbl.add name_tp x a;
      Hashtbl.add name_trm x ((xa,a), ref false);
      name_trm_list := (x,xa,a)::!name_trm_list;
      (x,xa)
    end


let initial_branch : trm list ref = ref []
let initial_branch_prenorm : trm list ref = ref []

let processed : (trm,int) Hashtbl.t = Hashtbl.create 100
let clause_ruleinfo : (clause,ruleinfo) Hashtbl.t ref = ref (Hashtbl.create 100)

let allclauses : clause list ref = ref [] (* allclauses is only used if formdeps is activated, in which case someone wants useless information. it contains all the clauses that were used in all searches across different subgoals; doesn't get reset. Aug 2016 *)
let allclause_ruleinfo : (clause,ruleinfo) Hashtbl.t ref = ref (Hashtbl.create 100)

exception DuplicateClause

let insert_clause c r =
(***  (Printf.printf "inserting search clause %s: %s\n" (List.fold_left (fun x y -> if ((String.length x) == 0) then (string_of_int y) else (x ^ " " ^ (string_of_int y))) "" c) (match r with (Some r) -> (ruleinfo_str r) | _ -> "None"); flush stdout); ***)
  if (Hashtbl.mem clausesTable c) then
    (*** Chad: April 4, 2011: Check if the clause has already been added before.  If it has, raise DuplicateClause. (Otherwise, we may end up with bad rule info -- e.g., a later rule which will violate freshness) ***)
(***    Printf.printf "duplicate clause %s: %s\n" (List.fold_left (fun x y -> if ((String.length x) == 0) then (string_of_int y) else (x ^ " " ^ (string_of_int y))) "" c) (match r with (Some r) -> (ruleinfo_str r) | _ -> "None"); flush stdout; ***)
     raise DuplicateClause
  else
    begin
      Hashtbl.add clausesTable c ();
      clauses := c::!clauses;
      if !mkproofterm = Some(PfFormdeps) then allclauses := c::!allclauses;
      match r with
      | Some(r) ->
	  Hashtbl.add (!clause_ruleinfo) c r;
	  if !mkproofterm = Some(PfFormdeps) then
	    Hashtbl.add (!allclause_ruleinfo) c r;
      | None -> ()
    end



let print_subproof_info minclauses cri =
  begin
    if (!verbosity > 0) then
      begin
	List.iter
	  (fun c ->
	    try
	      match (Hashtbl.find cri c) with
	      | InstRule(a,m,n) ->
		  begin
(***
		    match a with
		    | Ar(_,_) -> Printf.printf "HO Instantiation of type %s for\n%s:\n* %s\n" (stp_str a) (trm_str m) (trm_str n)
		    | _ -> () (*** Printf.printf "FO Instantiation of type %s for\n%s:\n* %s\n" (stp_str a) (trm_str m) (trm_str n) ***)
***)
		  end
	      | _ -> ()
	    with
	    | Not_found -> ())
	  minclauses;
      end
  end


let print_proof_statistics minclauses =
  let numassumptionclauses = ref 0 in
  let numsearchclauses = ref 0 in
  let assumptionlits = ref [] in
  let searchlits = Hashtbl.create 100 in
  List.iter
    (fun c ->
       if Hashtbl.mem (!clause_ruleinfo) c then
       begin
         incr numsearchclauses;
         List.iter (fun x -> if (not (Hashtbl.mem searchlits (abs x))) then Hashtbl.add searchlits (abs x) ()) c
       end
       else
       begin
         incr numassumptionclauses;
         match c with
         | [x] -> assumptionlits := ((abs x)::(!assumptionlits))
         | _ -> raise (GenericError "Assumption clause is not a unit clause?")
       end
    )
    minclauses;
  List.iter (Hashtbl.remove searchlits) (!assumptionlits);
  if (!verbosity > 3) then
  begin
    Printf.printf "Refutation Statistics:\n";
    Printf.printf "Number of assumptions %d\n" (!numassumptionclauses);
    Printf.printf "Number of search clauses %d\n" (!numsearchclauses);
    Printf.printf "Number of search literals %d\n" (Hashtbl.length searchlits);
  end


let new_assumption_lit l =
  try
    let c = [l] in
    insert_clause c None;
    if not (add_clauses c) || not (minisat_search_period ()) then
      raise (Unsatisfiable None)
  with
  | DuplicateClause -> ()
  | (Unsatisfiable _) ->
      if (!verbosity > 20) then Printf.printf "Proof found with assumption clauses only (no search)...%d clauses\n%!" (List.length (!clauses));
      if (mkprooftermp ()) then
	raise (Unsatisfiable (Some (AssumptionConflictR(l))))
      else
	raise (Unsatisfiable None)

let new_search_clause c r =
  try
    insert_clause c r;
    if not (add_clauses c) || not (minisat_search_period ()) then
      raise (Unsatisfiable None);
  with
  | DuplicateClause -> ()
  | (Unsatisfiable _) ->
      if (!verbosity > 3) then Printf.printf "Proof found for a goal...%d clauses\n%!" (List.length !clauses);
      if (mkprooftermp ()) then
	let cri = !clause_ruleinfo in (*** remember the clauses and this hashtable for them ***)
(*** To test with minimal unsatisfiable core: ***)
	let minclauses = Unsatcore.minimal_unsatisfiable_core (!clauses) in
	if (!verbosity > 3) then Printf.printf "Reduced to %d clauses\n%!" (List.length minclauses);
(*** To test all clauses: ***)
(***	    let minclauses = !clauses in ***)
	print_subproof_info minclauses cri;
        if (!verbosity > 0) then print_proof_statistics minclauses;
	raise (Unsatisfiable (Some (SearchR (minclauses, Hashtbl.find cri))))
      else
	raise (Unsatisfiable None)

let patoms : (string,int * (trm list)) Hashtbl.t = Hashtbl.create 10
let natoms : (string,int * (trm list)) Hashtbl.t = Hashtbl.create 10

let pchoiceatoms : (stp,int * (trm list)) Hashtbl.t = Hashtbl.create 10
let nchoiceatoms : (stp,int * (trm list)) Hashtbl.t = Hashtbl.create 10

let peqns : (string,int * trm * trm) Hashtbl.t = Hashtbl.create 10
let neqns : (string,int * trm * trm) Hashtbl.t = Hashtbl.create 10

let univpreds : (stp,(int * trm)) Hashtbl.t = Hashtbl.create 10
let instantiations : (stp,(trm,unit) Hashtbl.t) Hashtbl.t = Hashtbl.create 10
let instantiationslist : (stp,trm list) Hashtbl.t = Hashtbl.create 10
let default_elts : (string,trm) Hashtbl.t = Hashtbl.create 10

let set_default_elt aname x = Hashtbl.add default_elts aname x

let set_default_elt_if_needed a x =
  match a with
  | Base(aname) ->
      if (not (Hashtbl.mem default_elts aname)) then set_default_elt aname x
  | _ -> ()

let default_elt aname =
  try
    Hashtbl.find default_elts aname
  with
  | Not_found ->
      let a = Base aname in
      let (_,x) = get_fresh_name a in
      begin
	set_default_elt aname x;
	x
      end

let default_elt_p aname =
  Hashtbl.mem default_elts aname

let get_instantiations a =
  try
    Hashtbl.find instantiationslist a
  with Not_found -> []

let known_instantiation a m =
  try
    let h = Hashtbl.find instantiations a in
    Hashtbl.find h m;
    true
  with Not_found -> false

let cons_instantiation m ml =
    let iordcyc = get_int_flag "INSTANTIATION_ORDER_CYCLE" in
    let iordmask = get_int_flag "INSTANTIATION_ORDER_MASK" in
    if (iordcyc < 2) then
      begin
	if (iordmask mod 2 = 0) then
	  (m::ml)
	else
	  (ml @ [m])
      end
    else
      let j = List.length ml mod iordcyc in
      begin
	if ((iordmask lsr j) mod 2 = 0) then
	  (m::ml)
	else
	  (ml @ [m])
      end

let add_instantiation_2 a m =
  try
    let ml = Hashtbl.find instantiationslist a in
    Hashtbl.replace instantiationslist a (cons_instantiation m ml)
  with Not_found ->
    Hashtbl.add instantiationslist a [m]

let add_instantiation a m =
  try
    let h = Hashtbl.find instantiations a in
    Hashtbl.add h m ();
    add_instantiation_2 a m;
    set_default_elt_if_needed a m
  with Not_found ->
    let h : (trm,unit) Hashtbl.t = Hashtbl.create 10 in
    Hashtbl.add instantiations a h;
    Hashtbl.add h m ();
    add_instantiation_2 a m;
    set_default_elt_if_needed a m

let choiceopnames : (string,(stp * trm * trm)) Hashtbl.t = Hashtbl.create 10

let choiceop_axiom m =
  match m with
  | Ap (Forall (Ar (a, Prop)),
	Lam (Ar (_, Prop),
	     Ap (Forall _,
		 Lam (_,
		      Ap (Ap (Imp, Ap (DB (1, Ar (_, Prop)), DB (0, _))),
			  Ap (DB (1, Ar (_, Prop)),
			      Ap (Name (x, Ar (Ar (_, Prop), _)),
				  DB (1, Ar (_, Prop)))))))))
    -> Some(x,a)
  | Ap (Forall (Ar (a, Prop)),
	Lam (Ar (_, Prop),
	     Ap
	       (Ap (Imp,
		    Ap
		      (Ap (Imp,
			   Ap (Forall _,
			       Lam (_,
				    Ap (Ap (Imp, Ap (DB (1, Ar (_, Prop)), DB (0, _))),
					False)))),
		       False)),
		Ap (DB (0, Ar (_, Prop)),
		    Ap (Name (x, Ar (Ar (_, Prop), _)),
			DB (0, Ar (_, Prop)))))))
    -> Some(x,a)
  | _ -> None

let declare_choiceop x a (m,mb) = Hashtbl.add choiceopnames x (a,m,mb)

let choiceop m =
  match m with
  | Choice(a) -> Some(a)
  | Name(x,_) ->
      begin
	try
	  let (a,_,_) = Hashtbl.find choiceopnames x in
	  Some(a)
	with
	| Not_found -> None
      end
  | _ -> None

let filtered : (int,unit) Hashtbl.t = Hashtbl.create 10

let part_of_conjecture : (trm,unit) Hashtbl.t = Hashtbl.create 10

type namecategory =
    ChoiceOp of int * int * stp list * trm list (*** (i,n,sigmal,tl) where length of sigmal and tl are n, 0 <= i < n, Name has type (sigmal -> o) -> sigmal[i], and for each j:{0,...,n-1} tl[j] is the jth component of the n-ary choice operator (in particular, tl[i] is this one) ***)
   | DescrOp of int * int * stp list * trm list (*** (i,n,sigmal,tl) where length of sigmal and tl are n, 0 <= i < n, Name has type (sigmal -> o) -> sigmal[i], and for each j:{0,...,n-1} tl[j] is the jth component of the n-ary description operator (in particular, tl[i] is this one) ***)
   | IfThenElse of int * int * stp list (*** (i,n,sigmal) where length of sigmal is n, 0 <= i < n, Name has type o -> sigmal -> sigmal -> sigmal[i] ***)
   | ReflexiveBinary
   | IrreflexiveBinary
   | SymmetricBinary
   | ReflexiveSymmetricBinary
   | IrreflexiveSymmetricBinary

let constrainedName : (string,namecategory) Hashtbl.t = Hashtbl.create 10

let decomposable x =
   try
   let c = Hashtbl.find constrainedName x in (*** Some categorized names are decomposable and some are not ***)
   match c with
   | IfThenElse _ -> false (*** TO DO: Check that n-ary if-then-else need not be decomposable ***)
   (*** TO DO: Should *Binary cases be decomposable? ***)
   | _ -> true
   with Not_found -> true (*** A name is decomposable by default ***)

(*** Set completep to false if I use a mode that makes the search incomplete, so that failure does not imply satisfiability. ***)
let completep = ref true

let get_timeout_default x = match (!timeout) with Some y -> y | None -> x

let st_include_fun : (string -> unit) ref = ref (fun x -> raise (GenericError("Bug when including file " ^ x)))
let st_find_read_thf_fun : (string -> string -> unit) ref = ref (fun d x -> raise (GenericError("Bug when including file " ^ x)))

let coq_init () =
  begin
    Hashtbl.add coq_used_names "as" ();
    Hashtbl.add coq_used_names "at" ();
    Hashtbl.add coq_used_names "cofix" ();
    Hashtbl.add coq_used_names "else" ();
    Hashtbl.add coq_used_names "end" ();
    Hashtbl.add coq_used_names "exists2" ();
    Hashtbl.add coq_used_names "fix" ();
    Hashtbl.add coq_used_names "for" ();
    Hashtbl.add coq_used_names "forall" ();
    Hashtbl.add coq_used_names "fun" ();
    Hashtbl.add coq_used_names "if" ();
    Hashtbl.add coq_used_names "IF" ();
    Hashtbl.add coq_used_names "in" ();
    Hashtbl.add coq_used_names "let" ();
    Hashtbl.add coq_used_names "match" ();
    Hashtbl.add coq_used_names "mod" ();
    Hashtbl.add coq_used_names "Prop" ();
    Hashtbl.add coq_used_names "return" ();
    Hashtbl.add coq_used_names "Set" ();
    Hashtbl.add coq_used_names "then" ();
    Hashtbl.add coq_used_names "Type" ();
    Hashtbl.add coq_used_names "using" ();
    Hashtbl.add coq_used_names "where" ();
    Hashtbl.add coq_used_names "with" ();
    (*** Avoid certain names used by stt in Coq ***)
    Hashtbl.add coq_used_names "SType" ();
    Hashtbl.add coq_used_names "Sar" ();
    Hashtbl.add coq_used_names "o" ();
    Hashtbl.add coq_used_names "prop" ();
    Hashtbl.add coq_used_names "Sepsilon" ();
    Hashtbl.add coq_used_names "forall" ();
    Hashtbl.add coq_used_names "exists" ();
    Hashtbl.add coq_used_names "False" ();
    Hashtbl.add coq_used_names "True" ();
    Hashtbl.add coq_used_names "not" ();
    Hashtbl.add coq_used_names "Snot" ();
    Hashtbl.add coq_used_names "and" ();
    Hashtbl.add coq_used_names "Sand" ();
    Hashtbl.add coq_used_names "or" ();
    Hashtbl.add coq_used_names "Sor" ();
    Hashtbl.add coq_used_names "iff" ();
    Hashtbl.add coq_used_names "Siff" ();
    Hashtbl.add coq_used_names "ex" ();
    Hashtbl.add coq_used_names "SSigma" ();
    Hashtbl.add coq_used_names "SPi" ();
    Hashtbl.add coq_used_names "eq" ();
    Hashtbl.add coq_used_names "Seq" ();
    Hashtbl.add coq_used_names "I" ();
    Hashtbl.add coq_used_names "FalseE" ();
    Hashtbl.add coq_used_names "conj" ();
    Hashtbl.add coq_used_names "proj1" ();
    Hashtbl.add coq_used_names "proj2" ();
    Hashtbl.add coq_used_names "and_ind" ();
    Hashtbl.add coq_used_names "or_introl" ();
    Hashtbl.add coq_used_names "or_intror" ();
    Hashtbl.add coq_used_names "or_ind" ();
    Hashtbl.add coq_used_names "ex_intro" ();
    Hashtbl.add coq_used_names "ex_ind" ();
    Hashtbl.add coq_used_names "refl_equal" ();
    Hashtbl.add coq_used_names "eq_ind" ();
    Hashtbl.add coq_used_names "SinhE" ();
    Hashtbl.add coq_used_names "classic" ();
    Hashtbl.add coq_used_names "NNPP" ();
    Hashtbl.add coq_used_names "prop_ext" ();
    Hashtbl.add coq_used_names "functional_extensionality" ();
    Hashtbl.add coq_used_names "Sepsilon_spec" ();
    (*** Other names to avoid ***)
    Hashtbl.add coq_used_names "claim" ();

    (*FIXME add isar keywords*)
    Hashtbl.add coq_used_names "thm" ();
    Hashtbl.add coq_used_names "lemma" ();
  end

let print_coqsig c =
  let rec print_coqsig_base l =
    match l with
        (x :: r) ->
	        print_coqsig_base r;
          if !mkproofterm = Some IsarScript then
            if x <> "i" (*FIXME "$i"="i"*) then
              (*there's no need to redeclare $i*)
              (*FIXME to avoid name clashes, could suffix names with something*)
              Printf.fprintf c "typedecl %s\n" x
            else ()
	  else if !mkproofterm = Some TSTP then (* Don't print Coq declarations when tstp was requested. Chad, May 3 2016 *)
	    ()
          else
	    if (not (!coq2)) then Printf.fprintf c "Variable %s:SType.\n" x
      | [] -> ()
  in
  let rec print_coqsig_const l =
    match l with
        ((x, a) :: r) ->
	        begin
	          print_coqsig_const r;
	          try
	            ignore (List.assoc x (!coqsig_def));
	          with
	            | Not_found ->
	                begin
		          try
			    if !mkproofterm = Some IsarScript then
                              begin
				Printf.fprintf c "consts %s :: \"" x;
				print_stp_isar c a (* coq_names *) false;
		                Printf.fprintf c "\"\n"
                              end
			    else if !mkproofterm = Some TSTP then (* Don't print Coq declarations when tstp was requested. Chad, May 3 2016 *)
			      ()
			    else
                              begin
				Printf.fprintf c "Variable %s : " x;
				if (!coq2) then print_stp_coq2 c a false else print_stp_coq c a coq_names false;
				Printf.fprintf c ".\n"
                              end
		          with
		          | Not_found ->
		              begin
		                if (c != stdout) then close_out c;
		                raise (GenericError("A Satallax bug caused a problem creating the Coq/Isar file."))
		              end
	                end
	        end
      | [] -> ()
  in
  let rec print_coqsig_def l =
    match l with
        ((x,a)::r) ->
	        begin
	          print_coqsig_def r;
	          try
	            let m = List.assoc x (!coqsig_def) in
                    if !mkproofterm = Some IsarScript then
	              begin
	                Printf.fprintf c "definition %s :: \"" x;
	                print_stp_isar c a (* coq_names *) false;
	                Printf.fprintf c "\"\n where \"%s == (" x;
	                print_pretrm_isar c m coq_names coq_used_names (-1) (-1);
	                Printf.fprintf c ")\"\n"
	              end
                    else if !mkproofterm = Some TSTP then (* Don't print Coq declarations when tstp was requested. Chad, May 3 2016 *)
		      ()
                    else
	              begin
	                Printf.fprintf c "Definition %s : " x;
	                if (!coq2) then print_stp_coq2 c a false else print_stp_coq c a coq_names false;
	                Printf.fprintf c "\n := ";
	                if (!coq2) then print_pretrm_coq2 c m (-1) (-1) else print_pretrm_coq c m coq_names coq_used_names (-1) (-1);
	                Printf.fprintf c ".\n"
	              end
	          with
	          | Not_found -> ()
	        end
    | [] -> ()
  in
  let rec print_coqsig_hyp l =
    match l with
      ((x, t)::r) ->
	begin
	  try
            if !mkproofterm = Some IsarScript then
	      begin
	        print_coqsig_hyp r;
	        Printf.fprintf c "assumes %s : \"" x;
	                (* print_pretrm_isar c m coq_names coq_used_names (-1) (-1); *)
                trm_to_isar c (coqnorm t) (Syntax.Variables.make ());
	        Printf.fprintf c "\"\n"
              end
            else if !mkproofterm = Some CoqScript then
                (*have Syntax.trm but need Syntax.pretrm, so look it up*)
              let pt = List.assoc x !coqsig_hyp in
	      begin
	        Printf.fprintf c "Hypothesis %s : " x;
	        if (!coq2) then print_pretrm_coq2 c pt (-1) (-1) else print_pretrm_coq c pt coq_names coq_used_names (-1) (-1);
	        Printf.fprintf c ".\n"
              end
	    else if !mkproofterm = Some TSTP then (* I have no idea why I was printing Coq Hypotheses when tstp was requested. - Chad, May 3 2016 *)
	      ()
            else
              failwith "Printing of hypotheses: Unrecognised proof-output format."
	  with
	  | Not_found ->
	      begin
		if (c != stdout) then close_out c;
		raise (GenericError("A Satallax bug caused a problem creating the Coq file."))
	      end
	end
    | [] -> ()
  in
    if (not (!coqlocalfile)) then
      begin
        begin
          match (!mkproofterm) with
            |	Some CoqSPfTerm ->
	              begin
	                Printf.fprintf c "Require Export stt3.\n";
	              end
            |	Some CoqScript ->
	              begin
	                Printf.fprintf c "Add LoadPath \"%s/itp/coq\".\n" Config.satallaxdir;
	                Printf.fprintf c "Require Import stttab.\n";
	                Printf.fprintf c "Section SatallaxProblem.\n"
	              end
            |	Some IsarScript ->
	              begin
	                Printf.fprintf c "theory SatallaxProblem\n";
	                Printf.fprintf c "imports Satallax\n";
	                Printf.fprintf c "begin\n"
	              end
            |	_ -> ()
        end;
        print_coqsig_base !coqsig_base;
        print_coqsig_const !coqsig_const;
        print_coqsig_def !coqsig_const;
        if !mkproofterm = Some IsarScript then
	  Printf.fprintf c "\nlemma\n";
        print_coqsig_hyp !coqsig_hyp_trm;
        match (!conjecture) with
          | Some (t,_) ->
              if !mkproofterm = Some IsarScript then
	        begin
	          Printf.fprintf c "shows %s : \"" (Hashtbl.find coq_hyp_names (!conjecturename));
	                (* print_pretrm_isar c m coq_names coq_used_names (-1) (-1); *)
	          trm_to_isar c (coqnorm t) (Syntax.Variables.make ());
	          Printf.fprintf c "\"\n";
                  (*FIXME currently all definitions are unfolded, irrespective of whether when they're used. This seems to reflect Satallax's usage anyway.*)
                  if List.length !coqsig_def > 0 then
	            Printf.fprintf c "unfolding %s\n" (String.concat " " (List.map (fun (s, _) -> s ^ "_def") !coqsig_def))
	        end
              else if !mkproofterm = Some TSTP then (* Don't print Coq declarations when tstp was requested. Chad, May 3 2016 *)
		()
		  
              else
	        begin
	          Printf.fprintf c "Theorem %s : " (Hashtbl.find coq_hyp_names (!conjecturename));
(*	          if (!coq2) then print_pretrm_coq2 c m (-1) (-1) else print_pretrm_coq c m coq_names coq_used_names (-1) (-1); *) (*** No longer supported to get rid of dependence on pretrm ***)
	          Printf.fprintf c ".\n"
	        end
          | None ->
              if !mkproofterm = Some IsarScript then
                Printf.fprintf c "shows claim : \"False\"\n"
              else
                Printf.fprintf c "Theorem claim : False.\n"
      end
	
let declare_base_type a =
  if (!coq) then
    begin 
      let y = coqify_name a coq_names coq_used_names in
      coqsig_base := (y::!coqsig_base)
    end;
  Hashtbl.add name_base a ();
  name_base_list := (a::!name_base_list);
  if (get_bool_flag "CHOICE_AS_DEFAULT") then
    set_default_elt a (norm name_def (ap(Choice (Base a),Lam(Base a,False))))

let st_to_stp m =
  begin
    try
      to_stp m
    with
    | DeclareInd ->
	begin (*** Declare the base type i the first time it's seen.  Use a different name if an 'i' has already been used. ***)
	  declare_base_type "$i";
	  to_stp m
	end
  end

let st_to_trm_given_stp m tp =
  begin
    try
      let (n,_) = to_trm name_trm [] m (Some tp) in n
    with
    | DeclareInd ->
	begin (*** Declare the base type i the first time it's seen.  Use a different name if an 'i' has already been used. ***)
	  declare_base_type "$i";
	  let (n,_) = to_trm name_trm [] m (Some tp) in n
	end
  end

let st_to_trm m =
  begin
    try
      to_trm name_trm [] m None
    with
    | DeclareInd ->
	begin (*** Declare the base type i the first time it's seen.  Use a different name if an 'i' has already been used. ***)
	  declare_base_type "$i";
	  to_trm name_trm [] m None
	end
  end

let is_of_names : string list ref = ref []
let all_of_names : string list ref = ref []
let is_of_name : (string,unit) Hashtbl.t = Hashtbl.create 1
let all_of_name : (string,unit) Hashtbl.t = Hashtbl.create 1

let is_of_p m =
  match m with
  | Name(x,Ar(Base("$i"),Ar(Ar(Base("$i"),Prop),Prop))) -> Hashtbl.mem is_of_name x
  | Lam(Base("$i"),Lam(Ar(Base("$i"),Prop),Ap(DB(0,Ar(Base("$i"),Prop)),DB(1,Base("$i"))))) -> true
  | _ -> false
	
let all_of_p m =
  match m with
  | Name(x,Ar(Ar(Base("$i"),Prop),Ar(Ar(Base("$i"),Prop),Prop))) -> Hashtbl.mem all_of_name x
  | Lam(Ar(Base("$i"),Prop),Lam(Ar(Base("$i"),Prop),Ap(Forall(Base("$i")),Lam(Ar(Base("$i"),Prop),Ap(Ap(Imp,Ap(Ap(m1,DB(0,Base("$i"))),DB(2,Ar(Base("$i"),Prop)))),DB(1,Ar(Base("$i"),Prop))))))) ->
      is_of_p m1
  | _ -> false

let rec ontology_prop_p m =
  match m with
  | Ap(Ap(Name(x,Ar(Ar(Base("$i"),Prop),Ar(Ar(Base("$i"),Prop),Prop))),m1),m2) when Hashtbl.mem all_of_name x -> ontology_prop_p (gen_lam_body (Base("$i")) m2)
  | Ap(Ap(Name(x,Ar(Base("$i"),Ar(Ar(Base("$i"),Prop),Prop))),m1),m2) when Hashtbl.mem is_of_name x -> true
  | Ap(Forall(a),m1) -> ontology_prop_p (gen_lam_body a m1)
  | Ap(Ap(Imp,m1),m2) -> ontology_prop_p m2
  | _ -> false

let declare_typed_constant (name:string) (role:string) (m:pretrm) (al:(string * string) list) =
  begin
    if (!verbosity > 4) then (Printf.printf "declare_typed_constant %s %s\n%s\n" name role (pretrm_str m); flush stdout);
    if (!verbosity > 20) then (Printf.printf "annotations:\n"; List.iter (fun (a,b) -> Printf.printf "%s: %s\n" a b) al; flush stdout);
    match m with
      POf(PName(x),m) ->
	begin
	  match m with
	    PType -> (*** Actually unused. - April 20 2012 ***)
	      if (Hashtbl.mem name_base x) then raise (Redeclaration x);
	      if (Hashtbl.mem name_tp x) then raise (Redeclaration x);
	      declare_base_type x
	  | PName "$type" -> (*** The case that's used. Added April 20 2012 ***)
	      if (Hashtbl.mem name_base x) then raise (Redeclaration x);
	      if (Hashtbl.mem name_tp x) then raise (Redeclaration x);
	      declare_base_type x
	  | _ ->
	      let tp = st_to_stp m in
	      if !recognize_hashroots && tp = Ar(Base("$i"),Ar(Ar(Base("$i"),Prop),Prop)) && List.mem ("hashroot","01b8315391fd465f550f1a3788664028f3a6126b6bded775854e3b7d33765cd9") al then (is_of_names := x::!is_of_names; Hashtbl.add is_of_name x ()); (*** this annotation signals the prover to treat the name is the ontological "of" relation ***)
	      if !recognize_hashroots && tp = Ar(Ar(Base("$i"),Prop),Ar(Ar(Base("$i"),Prop),Prop)) && List.mem ("hashroot","67f26991e531d6c57a2161d295d3cdf1f416713aa711d23ccb703b2431633300") al then (all_of_names := x::!all_of_names; Hashtbl.add all_of_name x ()); (*** this annotation signals the prover to treat the name is the ontological "all_of" quantifier ***)
	      if (Hashtbl.mem name_base x) then raise (Redeclaration x);
	      if (Hashtbl.mem name_tp x) then raise (Redeclaration x);
	      if (!coq) then
		begin
		  let y = coqify_name x coq_names coq_used_names in
		  coqsig_const := (y,tp)::!coqsig_const
		end;
	      Hashtbl.add name_tp x tp;
	      Hashtbl.add name_trm x ((Name(x,tp),tp),ref false);
	      name_trm_list := (x,Name(x,tp),tp)::!name_trm_list
	end
    | _ -> raise (GenericError ("Incorrect format for type declaration " ^ name))
  end

let translucent_defn_p m =
  get_bool_flag "ONTOLOGY_DEFS_TRANSLUCENT" && is_of_p m


  (*** BEGIN mizartypeinfo ***)
let mizar_activate_prop st p =
  let (_,_,_,_,_,_,f) = st in
  !f p
    
let known_mizar_type_fact st p =
  if !miztypedebug then Printf.printf "known_mizar_type_fact %s\n" (trm_str p);
  let (_,_,sh,_,_,_,_) = st in
  try
    let (pol,a,m,r) = extract_mizar_type_prop_atom p in
    if !miztypedebug then
      begin
	Printf.printf "find_all %s gives\n" (trm_str a);
	List.iter (fun q -> Printf.printf ". %s\n" (trm_str a)) (Hashtbl.find_all sh a)
      end;
    List.mem p (Hashtbl.find_all sh a)
  with Not_found -> false (*** this shouldn't happen, but if it does just say it's not a fact ***)

let mizar_rule_applies st ax i pl gen spec a =
  let tau = ref [] in
  for j = 0 to i-1 do
    let (_,n) = new_evar 0 [] istp in
    tau := n::!tau
  done;
  try
    let genx = metanorm (meta_simulsubst_meta (to_meta gen) !tau) in
    if !miztypedebug then Printf.printf "About to match\n%s\nagainst\n%s\n" (metatrm_str genx) (trm_str spec);
    let (_,_,sh,_,cc,_,_) = st in
    let (ccv,ccp,ccb,ccf) = cc in
    let dpairs = pattern_match (fun m -> extend_cong_graph cc m; mizar_cong_class ccb (mizar_cong_find ccf m)) [([],genx,spec,a)] in
    if dpairs = [] then (*** nothing should be leftover ***)
      begin
	List.iter
	  (fun p1 ->
	    let p2 = metanorm (meta_simulsubst_meta (to_meta p1) !tau) in
	    try
	      let p3 = meta_to_ground emptydelta p2 in
	      if not (known_mizar_type_fact st p3) then
		begin
		  if !miztypedebug then Printf.printf "Rule not applied since premise not known: %s\n" (trm_str p3);
		  raise MizarRuleDoesNotApply
		end
	    with NotGround ->
	      if !miztypedebug then Printf.printf "Rule not applied since premise not ground: %s\nThis probably shouldn't be a rule, since the primary lit apparently didn't contain all the vars.\n" (metatrm_str p2);
	      raise MizarRuleDoesNotApply)
	  pl;
	    (*** if here, then all premises were known, so the rule applies ***)
	!tau
      end
    else
      begin
	if !miztypedebug then Printf.printf "leftover dpairs; not applying rule\n";
	raise MizarRuleDoesNotApply
      end
  with PatternMatchFailed ->
    if !miztypedebug then Printf.printf "pattern match failed; not applying rule\n";
    raise MizarRuleDoesNotApply
      
let rec mizar_widen st p =
  let (_,_,sh,_,_,_,_) = st in
  if !miztypedebug then Printf.printf "====== mizar_widen\n";
  let (pol,a,m,r) = extract_mizar_type_prop_atom p in
  if not (List.mem p (Hashtbl.find_all sh a)) then (*** only widen if it's a new type for a ***)
    begin
      if !miztypedebug then Printf.printf "New Computed Mizar Sort for %s: %s\n" (trm_str a) (trm_str p);
      Hashtbl.add sh a p;
      begin (*** climb the hierarchy ***)
	match m with
	| Name(mx,_) ->
	    List.iter
	      (fun (ax,axl,i,prim,pl,bl) ->
		if !miztypedebug then
		  begin
		    Printf.printf "Climbing hierarchy using %s\n" ax;
		    Printf.printf "Primary prem: %s\n" (trm_str prim);
		    List.iter (fun p2 -> Printf.printf "Secondary prem: %s\n" (trm_str p2)) pl;
		  end;
		ignore (mizar_apply_typing_rule ax axl st i pl bl prim p "typehier"))
	      (Hashtbl.find_all mizar_typing mx)
	| _ -> ()
      end
    end
and mizar_rule_apply axl st bl tau =
  List.iter
    (fun b ->
      let b2 = metanorm (meta_simulsubst_meta (to_meta b) tau) in
      try
	let b3 = meta_to_ground emptydelta b2 in
	mizar_activate_prop st b3;
	mizar_widen st b3
      with NotGround ->
	if !miztypedebug then Printf.printf "Conclusion of rule was not ground: %s\nThis probably shouldn't be a rule, since the primary lit apparently didn't contain all the vars.\n" (metatrm_str b2);
	raise Exit)
    bl
and mizar_apply_typing_rule ax axl st i pl bl prim p ruleclass = (*** in this function assume there are no missing args -- it's for typing not elaboration ***)
  try
    if !miztypedebug then Printf.printf "Testing if %s applies\n" ax;
    let tau = mizar_rule_applies st ax i pl prim p Prop in
    if !miztypedebug then Printf.printf "%s applies; applying it\n" ax;
	    (*** if here, then all premises were known, so the rule applies ***)
    mizar_rule_apply axl st bl tau;
    let prim2 = metanorm (meta_simulsubst_meta (to_meta prim) tau) in
    let prim3 = meta_to_ground emptydelta prim2 in
    let priml = Atom.get_literal prim3 in
    let pll =
      List.map
	(fun p ->
	  let p2 = metanorm (meta_simulsubst_meta (to_meta p) tau) in
	  let p3 = meta_to_ground emptydelta p2 in
	  (-(Atom.get_literal p3)))
	pl
    in
    List.iter
      (fun b ->
	let b2 = metanorm (meta_simulsubst_meta (to_meta b) tau) in
	try
	  let b3 = meta_to_ground emptydelta b2 in
	  let l = Atom.get_literal b3 in
	  if !verbosity > 4 then Printf.printf "soft typing rule instance %s: %s\n" ax (String.concat " " (List.map string_of_int (-axl::-priml::l::pll)));
	  new_search_clause (-axl::-priml::l::pll) (if (mkprooftermp ()) then (Some(MizarSoftTypingRule(ax,axl,ruleclass))) else None)
	with
	| Unsatisfiable(_) as e -> raise e
	| Timeout -> raise Timeout
	| _ -> ())
      bl;
    true
  with
  | Exit ->
      if !miztypedebug then Printf.printf "Exit? %s does not apply\n" ax;
      false
  | MizarRuleDoesNotApply ->
      if !miztypedebug then Printf.printf "%s does not apply\n" ax;
      false

let func_mizar_sorts st a h s =
  match h with
  | Name(f,_) ->
      begin
	let ls = List.length s in
	let ftl = Hashtbl.find_all mizar_func_types f in
	if !miztypedebug then Printf.printf "trying to find types for %s (|ftl|=%d)\n" f (List.length ftl);
	List.iter
	  (fun (ax,axl,n,pl,ml) ->
	    if not (n = ls) then
	      (Printf.printf "*** arity mismatch while trying to use %s (%d) for %s, rethink code for func_mizar_sorts\n" ax n f)
	    else
	      begin
		if !miztypedebug then Printf.printf "trying to use %s (%d) for %s\n" ax n f;
		try
		  ignore (List.find
			    (fun p ->
			      let p1 = simulsubst p s in
			      let b1 = known_mizar_type_fact st p1 in
			      not b1)
			    pl);
	      (*** if an unmatched premise was found, don't apply the rule ***)
		  ()
		with Not_found ->
	      (*** all premises were matched, apply the rule ***)
		  if !miztypedebug then Printf.printf "applying %s for %s\n" ax f;
		  let pll =
		    List.map
		      (fun p ->
			let p1 = simulsubst p s in
			let p1l = Atom.get_literal p1 in
			-p1l)
		      pl
		  in
		  List.iter
		    (fun m1 ->
		      let m2 = simulsubst m1 s in
		      let m2l = Atom.get_literal m2 in
		      if !verbosity > 4 then Printf.printf "soft typing rule (ft) instance %s: %s\n" ax (String.concat " " (List.map string_of_int (-axl::m2l::pll)));
		      new_search_clause (-axl::m2l::pll) (if (mkprooftermp ()) then (Some(MizarSoftTypingRule(ax,axl,"ft"))) else None);
		      mizar_activate_prop st m2;
		      mizar_widen st m2)
		    ml
	      end)
	  ftl;
      end
  | _ -> ()

let rec all_mizar_sorts st m =
  let (_,_,sh,sc,_,_,_) = st in
  if not (Hashtbl.mem sc m) then 
    begin
      if !miztypedebug then Printf.printf "About to compute all Mizar sorts for %s\n" (trm_str m);
      Hashtbl.add sc m ();
      let (h,s) = head_spine m in
      let rs = List.rev s in
      List.iter
	(fun n -> all_mizar_sorts st n)
	s;
      func_mizar_sorts st m h rs;
      if !miztypedebug then Printf.printf "Finished computing all Mizar sorts for %s\n" (trm_str m);
    end

let assert_mizar_sorts st m =
  if (!verbosity > 4) then Printf.printf "assert_mizar_sorts %s\n" (trm_str m);
  try
    let ql = extract_conjuncts m true in
    List.iter
      (fun q ->
	if (!verbosity > 4) then Printf.printf "assert_mizar_sorts one conjunct q %s\n" (trm_str q);
	let (pol,a,mh,r) = extract_mizar_type_prop_atom q in
	if tpof a = Base("$i") then
	  begin
	    all_mizar_sorts st a;
	    match mh with
	    | Name(_,_) ->
		if !miztypedebug then Printf.printf "assert sorting fact for %s: %s\n" (trm_str a) (trm_str m);
		mizar_widen st q
	    | _ -> ()
	  end)
      ql
  with
  | Not_found -> ()
  | ElaborationSimpleTypeError(_,_,_) -> ()
  | ElaborationPatternArityError(_,_,_) -> ()
  | ElaborationMizarTypeConflict(_,_) -> ()
  | ElaborationFailure(_) -> ()

let alldigits_p n =
  try
    for i = 0 to (String.length n) - 1 do
      let c = Char.code n.[i] in
      if (c < 48) || (c > 57) then raise Exit
    done;
    true
  with Exit -> false
      
let num_p n =
  String.length n > 0 && alldigits_p n

let unknown_name_p st g x =
  not (num_p x)
    &&
  not (Hashtbl.mem !mizar_name_tp x)
    &&
  not (try ignore (List.find (fun (_,n) -> match n with Name(xx,_) when xx = x -> true | _ -> false) g); true with Not_found -> false)

let fake_var_sort_p p g =
  match neg_body p with
  | Some(q) ->
      begin
	match head_spine q with
	| (_,(y::_)) ->
	    begin
	      try
		let (x,_) = List.find (fun (x,z) -> z = y) g in
		x = ""
	      with Not_found ->
		false
	    end
	| _ -> false
      end
  | None ->
      begin
	match head_spine p with
	| (_,(y::_)) ->
	    begin
	      try
		let (x,_) = List.find (fun (x,z) -> z = y) g in
		x = ""
	      with Not_found ->
		false
	    end
	| _ -> false
      end

let init_mizar_knowledge msop mntp mpl mtal =
  let mark_mizar_pred p =
    try
      let (pol,a,h,r) = extract_mizar_type_prop_atom p in
      match h with
      | Name(x,_) ->
	  if pol then
	    begin
	      if not (Hashtbl.mem mizar_pos_typing_predicates x) then
		begin
		  Hashtbl.add mizar_pos_typing_predicates x ();
		  if (Hashtbl.mem mizar_neg_typing_predicates x) then
		    Hashtbl.add mizar_both_typing_predicates x ();
		  if (!verbosity > 4) then Printf.printf "New + sorting predicate %s\n" x;
		end
	    end
	  else
	    begin
	      if not (Hashtbl.mem mizar_neg_typing_predicates x) then
		begin
		  Hashtbl.add mizar_neg_typing_predicates x ();
		  if (Hashtbl.mem mizar_pos_typing_predicates x) then
		    Hashtbl.add mizar_both_typing_predicates x ();
		  if (!verbosity > 4) then Printf.printf "New - sorting predicate %s\n" x;
		end
	    end
      | _ -> ()
    with _ ->
      ()
  in
  mizar_symbol_order_pred := msop;
  mizar_name_tp := mntp;
  mizar_pattern_list := mpl;
  typingaxiomslist := mtal;
  List.iter
    (fun mta ->
      match mta with
      | MizarFunctionType(ax,axm,fl,i,pl,bl) ->
	  if !verbosity > 4 then Printf.printf "Soft typing axiom (ft) %s\n" ax;
	  let axl = Atom.get_literal axm in
	  mizar_typing_axiom_lits := axl::!mizar_typing_axiom_lits;
	  List.iter mark_mizar_pred pl;
	  List.iter mark_mizar_pred bl;
	  List.iter (fun f -> Hashtbl.add mizar_func_types f (ax,axl,i,pl,bl)) fl
      | MizarTypeHierarchyStrict(ax,axm,fl,i,(p::pl),bl) ->
	  if !verbosity > 4 then Printf.printf "Soft typing axiom (ths) %s\n" ax;
	  let bl2 = List.filter (fun m -> not (List.mem m (p::pl))) bl in
	  let axl = Atom.get_literal axm in
	  mizar_typing_axiom_lits := axl::!mizar_typing_axiom_lits;
	  List.iter mark_mizar_pred (p::pl);
	  List.iter mark_mizar_pred bl2;
	  if not (bl2 = []) then
	    begin
	      List.iter (fun f -> Hashtbl.add mizar_typing f (ax,axl,i,p,pl,bl2)) fl
	    end
      | MizarTypeHierarchy(ax,axm,fl,i,(p::pl),bl) ->
	  if !verbosity > 4 then Printf.printf "Soft typing axiom (th) %s\n" ax;
	  let bl2 = List.filter (fun m -> not (List.mem m (p::pl))) bl in
	  let axl = Atom.get_literal axm in
	  mizar_typing_axiom_lits := axl::!mizar_typing_axiom_lits;
	  List.iter mark_mizar_pred (p::pl);
	  List.iter mark_mizar_pred bl2;
	  if not (bl2 = []) then
	    begin
	      List.iter (fun f -> Hashtbl.add mizar_typing f (ax,axl,i,p,pl,bl2)) fl
	    end
      | MizarTypingGeneric(ax,axm,i,pl,ml) ->
	  if !verbosity > 4 then Printf.printf "Soft typing axiom (gen) %s |pl| = %d\n" ax (List.length pl);
	  let axl = Atom.get_literal axm in
	  List.iter mark_mizar_pred pl;
	  mizar_typing_axiom_lits := axl::!mizar_typing_axiom_lits;
	  begin
	   let rec db_occur l i a m =
	     match m with
	     | DB(j,_) when j >= l ->
		let k = j - l in
		if k < i then
		  a.(k) <- true
		else
		  raise Exit
	     | Ap(m1,m2) ->
		db_occur l i a m1;
	       db_occur l i a m2
	     | Lam(_,m1) ->
		db_occur (l+1) i a m1
	     | _ -> ()
	   in
	   let mlsimp =
	     List.filter
	       (fun m -> not (List.mem m pl))
	       (*** I used to ensure that all arguments of the conclusion were variables, but I have no idea why.
	       &&
		 begin
		   match neg_body m with
		   | Some(nm) ->
		       let (h,s) = head_spine nm in
		       (try ignore (List.find (fun n -> match n with DB(_,_) -> false | _ -> true) s); false with Not_found -> true)
		   | None ->
		       let (h,s) = head_spine m in
		       (try ignore (List.find (fun n -> match n with DB(_,_) -> false | _ -> true) s); false with Not_found -> true)
		 end)  ***)
	       ml
	   in
	   if not (mlsimp = []) then
	     begin
	       let pl1 = ref pl in
	       let pr1 = ref [] in
	       while not (!pl1 = []) do
		 match !pl1 with
		 | [] -> () (*** impossible ***)
		 | (prim::plr) ->
		    pl1 := plr;
		   let alldboccur =
		     begin
		       let a = Array.create i false in
		       try
			 db_occur 0 i a prim;
			 for j = 0 to i-1 do
			   if not a.(j) then raise Exit
			 done;
			 true
		       with Exit -> false
		     end
		   in
		   if alldboccur then
		     begin
		       match neg_body prim with
		       | Some(nprim) ->
			  begin
			    let (h,s) = head_spine nprim in
			    match h with
			    | Name(f,_) ->
				if !miztypedebug then Printf.printf "Adding genforward rule %s for %s\n" ax f;
				Hashtbl.add mizar_typing_gen_forward f (ax,axl,i,prim,(!pr1 @ !pl1),mlsimp)
			    | _ -> ()
			  end
		       | None ->
			   begin
			    let (h,s) = head_spine prim in
			    match h with
			    | Name(f,_) ->
				if !miztypedebug then Printf.printf "Adding genforward rule %s for %s\n" ax f;
				Hashtbl.add mizar_typing_gen_forward f (ax,axl,i,prim,(!pr1 @ !pl1),mlsimp)
			    | _ -> ()
			  end
		     end;
		   pr1 := prim::!pr1
	       done;
	     end;
	  end
      | MizarRedefinition(ax,axm,i,pl,f,s,d) ->
	  if !verbosity > 4 then Printf.printf "Soft typing axiom (redef) %s\n" ax;
	  let axl = Atom.get_literal axm in
	  List.iter mark_mizar_pred pl;
	  mizar_typing_axiom_lits := axl::!mizar_typing_axiom_lits;
	  Hashtbl.add mizar_redef f (ax,Atom.get_literal axm,i,pl,s,d)
      | MizarClusterNonempty(ax,axm,i,pl,ql) ->
	  if !verbosity > 4 then Printf.printf "Soft typing axiom (nonempty cluster) %s\n" ax;
	  mizar_nonempty_clusters := (ax,axm,i,pl,ql)::!mizar_nonempty_clusters
      | _ ->
	  if !verbosity > 4 then Printf.printf "Surprisingly skipping a typing axiom\n";
	  ())
    mtal;
  List.iter
    (fun (nname,i,pl,rtp,al,pd,d) ->
      Hashtbl.add mizar_pattern_table nname (i,pl,rtp,al,pd,d))
    !mizar_pattern_list
  (*** END mizartypeinfo ***)

let init_mizstate () : mizstate =
  let mizvarcnt = ref 0 in
  let u = ref false in
  let metab : (pretrm,trm * stp * ((string * (trm list)) list)) Hashtbl.t = Hashtbl.create 100 in
  let melabh : (trm,trm) Hashtbl.t = Hashtbl.create 100 in
  let mtph : (trm,trm) Hashtbl.t = Hashtbl.create 100 in
  let mtpc : (trm,unit) Hashtbl.t = Hashtbl.create 100 in
  let mccv : (trm,unit) Hashtbl.t = Hashtbl.create 100 in
  let mccp : (trm,trm) Hashtbl.t = Hashtbl.create 100 in
  let mccb : (trm,trm) Hashtbl.t = Hashtbl.create 100 in
  let mccf : (trm,trm) Hashtbl.t = Hashtbl.create 100 in
  let mcc = (mccv,mccp,mccb,mccf) in
  let varh : (string,unit) Hashtbl.t = Hashtbl.create 100 in
  let activate_prop_fn : (trm -> unit) ref = ref (fun _ -> ()) in
  let mizst = (metab,melabh,mtph,mtpc,mcc,(varh,u,mizvarcnt),activate_prop_fn) in
  mizst

let reset_mizstate st =
  let (metab,melabh,mtph,mtpc,(mccv,mccp,mccb,mccf),(varh,u,mizvarcnt),activate_prop_fn) = st in
  mizvarcnt := 0;
  u := false;
  Hashtbl.clear metab;
  Hashtbl.clear melabh;
  Hashtbl.clear mtpc;
  Hashtbl.clear mccv;
  Hashtbl.clear mccp;
  Hashtbl.clear mccb;
  Hashtbl.clear mccf;
  Hashtbl.clear varh
    
let mizst : mizstate = init_mizstate()
    
let rec init_probitem m =
  match m with
  | ProbSoftTyping(mi) ->
      begin
	if not (get_bool_flag "USE_MIZAR_SOFT_TYPES") then
	  begin
	    match mi with
	    | MizarFunctionType(ax,axm,_,_,_,_) ->
		let al = [] in
		let w = 1.0 in
		init_probitem (ProbAx(ax,"hypothesis",axm,al,w))
	    | MizarTypeHierarchyStrict(ax,axm,_,_,_,_) ->
		let al = [] in
		let w = 1.0 in
		init_probitem (ProbAx(ax,"hypothesis",axm,al,w))
	    | MizarTypeHierarchy(ax,axm,_,_,_,_) ->
		let al = [] in
		let w = 1.0 in
		init_probitem (ProbAx(ax,"hypothesis",axm,al,w))
	    | MizarTypingGeneric(ax,axm,_,_,_) -> (*** included as ProbAx in probsig by default already ***)
		()
	    | MizarRedefinition(ax,axm,_,_,_,_,_) -> (*** included as ProbAx in probsig by default already ***)
		()
	    | MizarClusterNonempty(ax,axm,_,_,_) ->
		let al = [] in
		let w = 1.0 in
		init_probitem (ProbAx(ax,"hypothesis",axm,al,w))
	  end
      end
  | ProbDef(x,a,m,al,w) ->
      begin
	try
	  if (get_bool_flag "ALL_DEFS_AS_EQNS") then
	    raise TreatAsAssumption
	  else
	    let m0 = logicnorm m in
	    let m2 = norm name_def m0 in
	    if (!verbosity > 20) then Printf.printf "name_def %s %s\n" x (trm_str m2);
	    Hashtbl.add name_def_all x (norm name_def_all m2);
	    if not (translucent_defn_p m) then
	      begin
		translucent_defns := true;
		Hashtbl.add name_def x m2
	      end
	with TreatAsAssumption ->
	  init_probitem (ProbAx(x,"hypothesis",Ap(Ap(Eq(a),Name(x,a)),m),al,w))
      end
  | ProbAx(name,role,tm,al,w) ->
      begin
	let m0 = logicnorm tm in
	let tmn = norm name_def m0 in
	Hashtbl.add name_hyp name tmn;
	Hashtbl.add name_hyp_inv tmn (name,tm);
	if (!verbosity > 20) then Printf.printf "name_hyp %s %s\n" name (trm_str tmn);
	initial_branch_prenorm := tm::(!initial_branch_prenorm);
	initial_branch := tmn::(!initial_branch)
      end
  | ProbConj(name,tm,al,w) ->
      begin
	conjecturename := name;
	if (name != "claim") then
	  ignore (coqify_name name coq_hyp_names coq_used_names);
	let ntm = Ap(Neg,tm) in
	let ntmn = norm name_def (logicnorm ntm) in
	initial_branch_prenorm := (ntm::(!initial_branch_prenorm));
	if (!verbosity > 20) then Printf.printf "name_conj negated %s %s\n" name (trm_str ntmn);
	initial_branch := (ntmn::(!initial_branch));
	conjecture := Some (tm,ntmn);
	Hashtbl.add part_of_conjecture ntmn ()
      end

let defpreweight m al = 2.0

let axpreweight m al =
  if ontology_prop_p m then 1.0 else 3.0

let conjpreweight m al = 0.0

(***
 THF official policy requires that all names are given a type before use,
 that definitions can be circular (in which case they must be treated as equations),
 and that definitions may appear *after* it's already been used in an assumption.
 In order to comply with this policy *and* do something reasonable in the 'usual' case,
 I do the following:

 1. I keep a hashtable (occurred_names) of typed names that have been used in an assumption.
 2. When I encounter a definition, if it's of the form (x = t), I first parse t.  If x has been used,
    then I include the equation as an axiom.  Otherwise, x is treated as a definition and expanded away.

 NOTE: If I wanted to be strict, I would exit if the definition is given without the name being given a type,
 but I will not require the type to be given.  I will print a warning if the verbosity is > 0.
 ***)
let declare_definition_real x m al =
  if (Hashtbl.mem name_base x) then raise (Redeclaration x);
  if (Hashtbl.mem name_def x) then raise TreatAsAssumption; (*** treat it as an assumption per THF policy. - Mar 31, 2011 - Chad ***)
(*** raise (Redeclaration x); ***)
  try
    let tp = Hashtbl.find name_tp x in
    let tm = belnorm (st_to_trm_given_stp m tp) in
    try 
      let (_,r) = Hashtbl.find name_trm x in
      if (!r) then
	raise TreatAsAssumption (*** It's already been used, treat 'definition' as an assumption. ***)
      else
	raise Not_found
    with
    | Not_found ->
	begin
	  if (!coq) then (*** Coq proofs now only work for non-slaves ***)
	    begin (*** The name was coqified when the type was declared. ***)
	      try
		let y = Hashtbl.find coq_names x in
		coqsig_def := (y,m)::!coqsig_def;
		coqsig_def_trm := (y,tm)::!coqsig_def_trm
	      with
	      | Not_found -> raise (GenericError("Could not find Coq version of name " ^ x))
	    end;
	  Hashtbl.add name_def_prenorm x tm;
	  if is_of_p tm then
	    (is_of_names := x::!is_of_names; Hashtbl.add is_of_name x ())
	  else if all_of_p tm then
	    (all_of_names := x::!is_of_names; Hashtbl.add all_of_name x ());
	  let w = defpreweight tm al in
	  probsig := ProbDef(x,tp,tm,al,w)::!probsig
	end
  with
  | Not_found ->
      begin (*** Giving a definition without giving it's type isn't allowed in THF anymore.  I'm still allowing it.  ***)
	if ((!verbosity > 0) && (not (!coqglobalfile)) && (not (!coqlocalfile))) then Printf.printf "WARNING: %s defined without giving type first.\n" x;
	let (tm,tp) = st_to_trm m in
	let tm = belnorm tm in
	if (!coq) then
	  begin
	    let y = coqify_name x coq_names coq_used_names in
	    coqsig_const := (y,tp)::!coqsig_const;
	    coqsig_def := (y,m)::!coqsig_def;
	    coqsig_def_trm := (y,tm)::!coqsig_def_trm
	  end;
	Hashtbl.add name_tp x tp;
	Hashtbl.add name_trm x ((Name(x,tp),tp),ref false);
	name_trm_list := (x,Name(x,tp),tp)::!name_trm_list;
	Hashtbl.add name_def_prenorm x tm;
	let w = defpreweight tm al in
	probsig := ProbDef(x,tp,tm,al,w)::!probsig
      end

let rec declare_thf_logic_formula (name:string) (role:string) (m:pretrm) (al:(string * string) list) =
  begin
    if !verbosity > 4 then (Printf.printf "declare_thf_logic_formula %s %s\n" name role; flush stdout);
    if (!verbosity > 20) then (Printf.printf "annotations:\n"; List.iter (fun (a,b) -> Printf.printf "%s: %s\n" a b) al; flush stdout);
    if ((role = "axiom") || (role = "hypothesis") || (role = "assumption") || (role = "lemma") || (role = "theorem") || (role = "corollary")) then
      begin
	if (Hashtbl.mem name_hyp name) then raise (Redeclaration name);
	let tm = st_to_trm_given_stp m Prop in
	let tm = belnorm tm in
	if (!coq) then (** Giving Coq proofs will now only work for non-slaves since the pretrm is used **)
	  begin
	    let y = coqify_name name coq_hyp_names coq_used_names in
	    coqsig_hyp := ((y,m)::!coqsig_hyp);
	    coqsig_hyp_trm := ((y,tm)::!coqsig_hyp_trm)
	  end;
	let w = axpreweight tm al in
	if true then
	  begin
	    try
	      let (i,pl,ml) = mizar_type_fact1_p tm 0 [] in
	      if ml = [] then raise Not_found; (*** ignore those which have no conclusion (just 'true') ***)
	      names_in_mizar_type_assumptions := [];
	      mizar_hook_names := [];
	      if mizar_type_fact2o_p 0 tm then
		let hooknames2 = !mizar_hook_names in
		begin
		  names_in_mizar_type_assumptions := [];
		  mizar_hook_names := [];
		  let m =
		    if mizar_type_fact3o_p tm then
		      begin
			if print_soft_typing_info then Printf.printf "(THS \"%s\" \"%s\")\n" !problemfile name;
			MizarTypeHierarchyStrict(name,tm,!mizar_hook_names,i,pl,ml)
		      end
		    else
		      begin
			if print_soft_typing_info then Printf.printf "(TH \"%s\" \"%s\")\n" !problemfile name;
			MizarTypeHierarchy(name,tm,hooknames2,i,pl,ml)
		      end
		  in
		  typingaxiomslist := m::!typingaxiomslist;
		  probsig := ProbSoftTyping(m)::!probsig;
		  completep := false;
(*		  probsig := ProbAx(name,role,tm,al,w)::!probsig *)
		end
	      else
		begin
		  names_in_mizar_type_assumptions := [];
		  mizar_hook_names := [];
		  let m =
		    if mizar_type_fact4o_p tm then
		      begin
			if print_soft_typing_info then Printf.printf "(FT \"%s\" \"%s\")\n" !problemfile name;
			MizarFunctionType(name,tm,!mizar_hook_names,i,pl,ml)
		      end
		    else
		      begin
			if print_soft_typing_info then Printf.printf "(GEN \"%s\" \"%s\")\n" !problemfile name;
			MizarTypingGeneric(name,tm,i,pl,ml)
		      end
		  in
		  typingaxiomslist := m::!typingaxiomslist;
		  probsig := ProbSoftTyping(m)::!probsig;
		  match m with
		  | MizarTypingGeneric(_,_,_,_,_) ->
		      probsig := ProbAx(name,role,tm,al,w)::!probsig (* 'generic' mizar soft typing axioms may not even be used by the soft typing, and if they are it is only for limited 'clustering' style reasoning, so include as an ordinary axiom as well *)
		  | _ ->
		      completep := false
		end
	    with Not_found ->
	      try
		names_in_mizar_type_assumptions := [];
		let (i,pl,f,s,def) = mizar_type_fact5o_p tm 0 [] in
		let m = MizarRedefinition(name,tm,i,pl,f,s,def) in
		if print_soft_typing_info then Printf.printf "(REDEF \"%s\" \"%s\")\n" !problemfile name;
		if !miztypedebug then Printf.printf "identified redef %s %s\n" name f;
		typingaxiomslist := m::!typingaxiomslist;
		probsig := ProbSoftTyping(m)::!probsig;
		probsig := ProbAx(name,role,tm,al,w)::!probsig (* include redefinitions as ordinary axioms too *)
	      with Not_found ->
		try
		  let (i,pl,ql) = mizar_type_fact6_p tm 0 [] in
		  let m = MizarClusterNonempty(name,tm,i,pl,ql) in
		  if print_soft_typing_info then Printf.printf "(NONEMP \"%s\" \"%s\")\n" !problemfile name;
		  typingaxiomslist := m::!typingaxiomslist;
		  probsig := ProbSoftTyping(m)::!probsig; (* currently [Jan 2018] nonemptiness of clusters is not used anywhere in the soft typing inference *)
		  completep := false
(*  		  probsig := ProbAx(name,role,tm,al,w)::!probsig *) (* do not include nonemptiness as ordinary axioms just because they are usually not helpful; almost certainly leads to loss of completeness though *)
		with Not_found ->
		  probsig := ProbAx(name,role,tm,al,w)::!probsig
	  end
	else
	  probsig := ProbAx(name,role,tm,al,w)::!probsig
      end
    else if (role = "conjecture") then
      begin
	match (!conjecture) with
	| Some _ -> raise (GenericError "Problem file has more than one conjecture.")
	| None ->
	    let tm = st_to_trm_given_stp m Prop in
	    let tm = belnorm tm in
	    let w = conjpreweight tm al in
	    probsig := ProbConj(name,tm,al,w)::!probsig
      end
    else
      raise (GenericError ("Unknown role " ^ role))
  end
and declare_definition (name:string) (role:string) (m:pretrm) (al:(string * string) list) =
  try
    begin
      if (get_bool_flag "ALL_DEFS_AS_EQNS") then raise TreatAsAssumption;
      if !verbosity > 4 then (Printf.printf "declare_definition %s %s\n" name role; flush stdout);
      if (!verbosity > 20) then (Printf.printf "annotations:\n"; List.iter (fun (a,b) -> Printf.printf "%s: %s\n" a b) al; flush stdout);
      match m with
	PDef(PName(x),m) -> (*** No longer THF syntax. ***)
	  declare_definition_real x m al
      | PAp(PAp(PEq,PName(x)),m) ->
	  declare_definition_real x m al
      | _ -> (*** Treat as an assumption, no matter how it looks. Odd, but OK. This may be too liberal; we haven't decided yet. ***)
	  raise TreatAsAssumption
(*** raise (GenericError ("Incorrect format for definition " ^ name)) ***)
    end
  with
  | TreatAsAssumption ->
      declare_thf_logic_formula name "hypothesis" m al

(*** Code for enumeration of types and terms - Dec 10, 2010 - Chad ***)
let enum_started = ref false
let enum_of_started_ : (stp,unit) Hashtbl.t = Hashtbl.create 5
let enum_of_started a =
  Hashtbl.mem enum_of_started_ a
let enum_of_start a =
  Hashtbl.add enum_of_started_ a ()
let type_continuations_rtp : (stp option,(stp -> int -> unit)) Hashtbl.t = Hashtbl.create 5
let term_continuations_rtp : (stp,(stp list * trm * int -> unit)) Hashtbl.t = Hashtbl.create 5
let usableTypes_rtp : (stp,(stp * int)) Hashtbl.t = Hashtbl.create 5
let usableTypes : (stp * stp * int) list ref = ref []
let usableHeads_rtp : (stp,(stp list * trm * int)) Hashtbl.t = Hashtbl.create 5

let new_type_continuation_rtp ar f =
  Hashtbl.add type_continuations_rtp (Some ar) f

let new_type_continuation f =
  Hashtbl.add type_continuations_rtp None f

let iter_type_continuations_rtp ar a d =
  List.iter (fun f -> f a d) (Hashtbl.find_all type_continuations_rtp (Some ar))

let iter_type_continuations a d =
  List.iter (fun f -> f a d) (Hashtbl.find_all type_continuations_rtp None)

let new_term_continuation_rtp ar f =
  Hashtbl.add term_continuations_rtp ar f

let iter_term_continuations_rtp ar sigmal m p =
  List.iter (fun f -> f (sigmal,m,p)) (Hashtbl.find_all term_continuations_rtp ar)

let new_usable_type_rtp ar a d =
  Hashtbl.add usableTypes_rtp ar (a,d);
  usableTypes := ((ar,a,d)::(!usableTypes))

let usable_types_rtp ar = Hashtbl.find_all usableTypes_rtp ar

let usable_types () = !usableTypes

let new_usable_head_rtp ar sigmal m n = Hashtbl.add usableHeads_rtp ar (sigmal,m,n)

let usable_heads_rtp ar = Hashtbl.find_all usableHeads_rtp ar

(*** search init ***)
let search_init () =
  (*** Add initial instantiations: true and false for o ***)
  add_instantiation Prop False;
  add_instantiation Prop (neg False)
  
(*** reset search ***)
let reset_search () =
  begin
    Hashtbl.clear clausesTable;
    clauses := [];
    clause_ruleinfo := Hashtbl.create 100;
    Searchoption.reset_pqueues ();
    Hashtbl.clear patoms;
    Hashtbl.clear natoms;
    Hashtbl.clear pchoiceatoms;
    Hashtbl.clear nchoiceatoms;
    Hashtbl.clear peqns;
    Hashtbl.clear neqns;
    Hashtbl.clear univpreds;
    Hashtbl.clear instantiations;
    Hashtbl.clear processed;
    enum_started := false;
    Hashtbl.clear enum_of_started_;
    Hashtbl.clear type_continuations_rtp;
    Hashtbl.clear term_continuations_rtp;
    Hashtbl.clear usableTypes_rtp;
    usableTypes := [];
    Hashtbl.clear usableHeads_rtp;
    Hashtbl.clear choiceopnames;
    Hashtbl.clear filtered;
    Eproverstate.reset_eprover_state ();
    search_init()
  end

let print_branch =
  List.iteri (fun i m -> Printf.printf "%d %s\n" (i+1) (trm_str m))

(* select_list l [i1; ...; in] = [nth l in; ...; nth l i1] *)
let select_list l = List.rev_map (List.nth l)

let select_axioms_list : int list ref = ref [];;

let select_axioms () =
  (*** I should select the prenorm branch too, but that's only needed for proof reconstruction. ***)
  initial_branch := select_list !initial_branch !select_axioms_list;
  if (!verbosity > 4) then
    begin
      print_endline "Initial branch after selection:";
      print_branch !initial_branch
    end

let num_axioms () = List.length (!initial_branch)

    
