(* File: satallaxmain.ml *)
(* Author: Chad E Brown *)
(* Created: September 2010 - moved most of this from satallax.ml to satallaxmain.ml in September 2011 *)

open Flags
open Syntax
open State
open Log
open Search
open Error

exception InputHelpError of string
exception InputError of string

let coqfile = ref ""
let flag_overrides = ref []
let schedule_files : string list ref = ref []
let preparsed = ref false
let starttime = ref 0.0
let endtime = ref 0.0

let printnumaxiomsflag : bool ref = ref false
let selectaxiomsflag : bool ref = ref false

let fclassifier_file : string option ref = ref None
let tclassifier_file : string option ref = ref None
let simple_file : string option ref = ref None

let sine_tolerance_vary = ref true
let sine_generality_threshold_vary = ref true
let sine_tolerance = ref 1.2 (*** can be set by command line --sine_tolerance t where t should >= 1.0; if none is set then slice over different values. ***)
let sine_depth = ref None (*** can be set by command line --sine_depth d where d should be an >= 0.0; if none is set then slice over different depths (more or less) ***)
let sine_generality_threshold = ref 0 (*** can be set by command line --sine_generality_threshold g; where g should be an integer >= 0 ***)

let help_lines =
[ "Usage: satallax [-[VvNhi]] [-verb <int>] [-P <PicoMus>] [-M <modedir>] [-s <schedule>] [-m <mode>] [-flag <name> <value>] [-t <timeout in seconds>] [-inferences <int>] [-p <pfformat>] <problemfile>"
; "-M <dir> : Set the directory in which the mode/schedule files are stored."
; "       The default mode directory is the 'modes' subdirectory of the Satallax directory."
; "-s : Schedule of modes to try (previously called the 'strategy schedule')"
; "-m : Mode"
; "-P <file> : PicoMus executable file"
; "-E <file> : E prover executable file"
; "-V : Print version number and quit"
; "-v : Verbose"
; "-h : recognize hashroots for special treatment"
; "-i : given problem is incomplete version of the real problem, so do not report Satisfiable or CounterSatisfiable"
; "-N : Try to determine if the problem is a non-theorem (Satisfiable or CounterSatisfiable)"
; "-verb <int> : Verbosity of given level, -verb 0 means silent"
; "-c [<file>|-] : Create a Coq version of problem, with a proof script if -p is included and proof search succeeds"
; "-C : The problem is given as a Coq file instead of as a THF file."
; "-G : A Coq file containing multiple conjectures is given. Try to prove each of them independently."
; "-p [tstp|coqscript|coqspfterm|hocore|modeltrue|model|isar|info|useful|formdeps]: Output proof/proof object/proof info"
; "-pfusefulout <file>: File in which to save info about what was useful in the proof"
; "-pfformdepsout <file>: File in which to save info about what formulas and dependencies were in the proof"
; "--sine_tolerance <float>: Set the sine_tolerance (t value) to some value >= 1.0"
; "--sine_depth <float>: Set the sine_depth (d value) to some value >= 0.0"
; "--sine_generality_threshold <int>: Set the sine_generality_threshold (g value) to some value >= 0"
]

let comment_line_p l =
  if String.length l = 0 then true
  else l.[0] = ';'

let rec get_input_lines c =
  try 
    let line = input_line c in
    line :: get_input_lines c
  with End_of_file -> []

let get_content_lines c =
  List.filter (fun line -> not (comment_line_p line)) (get_input_lines c)

let rec set_flags = function
  flag :: value :: rest -> set_flag flag value; set_flags rest
| flag :: [] -> raise (InputError ("Value after flag " ^ flag ^ " expected"))
| [] -> ()

let read_mode_file c = set_flags (get_content_lines c)

let read_schedule_line l =
  Scanf.sscanf l "%s %f" (fun mode time -> (mode, time))

let rec read_sine_schedule_lines ll (t,g) =
  match ll with
  | [] -> ([],[])
  | l::lr ->
      try
	Scanf.sscanf l "SINE %f %d"
	  (fun t g ->
	    let (r1,r2) = read_sine_schedule_lines lr (t,g) in
	    ([],(t,g,r1)::r2))
      with Scanf.Scan_failure(_) ->
	Scanf.sscanf l "%f %s %f"
	  (fun d mode time ->
	    let (r1,r2) = read_sine_schedule_lines lr (t,g) in
	    ((d,mode,time)::r1,r2))
    
let rec read_schedule_lines ll =
  match ll with
  | [] -> ([],[])
  | l::lr ->
      try
	Scanf.sscanf l "SINE %f %d"
	  (fun t g ->
	    let (r1,r2) = read_sine_schedule_lines lr (t,g) in
	    ([],(t,g,r1)::r2))
      with Scanf.Scan_failure(_) ->
	Scanf.sscanf l "%s %f"
	  (fun mode time ->
	    let (r1,r2) = read_schedule_lines lr in
	    ((mode,time)::r1,r2))

let read_schedule_file c =
  read_schedule_lines (get_content_lines c)

let load_schedule s =
  let schedfile = !Config.modedir ^ "/" ^ s in
  if not (Sys.file_exists schedfile) then
    raise (InputError ("Could not find schedule " ^ schedfile));
  let (defaultsched,_) = Utils.with_in schedfile read_schedule_file in
  defaultsched

let load_sine_schedule s =
  let schedfile = !Config.modedir ^ "/" ^ s in
  if not (Sys.file_exists schedfile) then
    raise (InputError ("Could not find schedule " ^ schedfile));
  let (_,sinesched) = Utils.with_in schedfile read_schedule_file in
  sinesched
    
let load_mode m =
  let modefile = (!Config.modedir ^ "/" ^ m) in
  if (not (Sys.file_exists modefile)) then
    raise (InputError ("Could not find mode " ^ modefile));
  Utils.with_in modefile read_mode_file

let read_coq_file (f:string) =
  if (!verbosity > 20) then Printf.printf "Starting to read Coq file %s\n" f;
  coqinchannel := if (f = "") then stdin else (open_in f);
  let ch = Lexing.from_channel !coqinchannel in
  try
    while true do
      Coqparser.documentitem Coqlexer.token ch
    done
  with
    Coqlexer.Eof ->
      begin
	if (!verbosity > 20) then Printf.printf "End of Coq file\n";
	if ((!coqglobalfile) && (not ((!coqinchannel) = stdin))) then
	  let p = pos_in !coqinchannel in
	  let j = ref 0 in
	  begin
	    seek_in !coqinchannel 0;
	    List.iter
	      (fun (x,i) ->
		if (!verbosity > 20) then Printf.printf "End of Coq file %d %d\n" i (!j);
		match x with
		| Some g -> if (!verbosity > 20) then g stdout; g !coqoutchannel; seek_in !coqinchannel i; j := i
		| None -> while (!j < i) do (incr j; let z = input_char !coqinchannel in output_char !coqoutchannel z) done
		      )
	      (List.rev (!State.coqinticks));
	    while (!j < p) do (incr j; let z = input_char !coqinchannel in output_char !coqoutchannel z) done;
	  end;
	  close_in !coqinchannel;
	  close_out !coqoutchannel
      end

let read_thf_file (f:string) (include_fun : string -> unit) =
  let ch = Lexing.from_channel (if (f = "") then stdin else (open_in f)) in
  let old_include_fun = !st_include_fun in
  st_include_fun := include_fun;
(***  List.iter Tptp_config.process_thf (Tptp_parser.tptp_file Tptp_lexer.token ch); ***)
  ignore (Tptp_parser.tptp_file Tptp_lexer.token ch);
  if (!verbosity > 4) then Printf.printf "Finished reading thf file %s\n" f;
  st_include_fun := old_include_fun

let rec find_read_thf_file_r odir dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file odir)
  else if (String.length dir > 1) then
    find_read_thf_file_r odir (Filename.dirname dir) f
  else
    raise (FileNotFound f)
and find_read_thf_file dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file dir)
  else
    begin
      try
	let tptpdir = Sys.getenv "TPTP" in
	let tff = (tptpdir ^ "/" ^ f) in
	if (Sys.file_exists tff) then
	  read_thf_file tff (find_read_thf_file dir)
	else
	  find_read_thf_file_r dir dir f
      with
      | Not_found -> find_read_thf_file_r dir dir f
    end;;

st_find_read_thf_fun := find_read_thf_file;;

let read_proofkind = function
  "tstp" -> TSTP
| "coqscript" -> CoqScript
| "coqspfterm" -> CoqSPfTerm
| "hocore" -> HOCore
| "model" -> Model
| "modeltrue" -> ModelTrue
| "isar" -> IsarScript
| "info" -> PfInfo
| "useful" -> PfUseful
| "formdeps" -> PfFormdeps
| p -> raise (InputHelpError ("Unknown kind of proof " ^ p ^ " for -p"))

let setup_proofkind = function
  IsarScript ->
    mkproofterm := Some IsarScript;
    Flag.result_coq := false;
    Flag.result_isabellehol := true
| p -> mkproofterm := Some p

let enslave args = slaveargs := List.rev_append args !slaveargs

let process_short_cmd_line_arg = function
  'v' -> enslave ["-v"]; verbosity := 5
| 'V' -> print_endline Version.version; exit 0
| 'h' -> enslave ["-h"]; recognize_hashroots := true
| 'i' -> enslave ["-i"]; completep := false
|  a  -> raise (InputHelpError ("Unknown command line argument " ^ String.make 1 a))

let set_problemfile p =
  if !problemfile = "" then problemfile := p
  else raise (InputHelpError ("Multiple problem files passed: " ^ !problemfile
  ^ " and " ^ p))

let process_cmd_line_arg = function
  "-m"::m::r -> mode := m :: !mode; r
| "-s"::s::r -> schedule_files := s :: !schedule_files; r
| "-M"::m::r -> Config.modedir := m; enslave ["-M"; m]; r
| "-P"::p::r -> Config.picomus := p; enslave ["-P"; p]; r
| "-E"::e::r -> Config.eprover := e; enslave ["-E"; e]; r
| "-t"::t::r -> timeout := Some (float_of_string t); r
| "-ht"::t::r -> hardtimeout := Some (float_of_string t); r
| "-C"::r -> coqlocalfile := true; enslave ["-C"]; r
| "-G"::r -> coqglobalfile := true; r
| "-c"::c::r ->
    coq := true;
    if (c = "-") then coqoutchannel := stdout
    else (coqfile := c; coqoutchannel := open_out c);
    enslave ["-c"; c]; r
| "-slave"::r -> slave := true; r
| "-preparsed"::r -> preparsed := true; r
| "-N"::r -> nontheorem := true; enslave ["-N"]; r
| "-flag"::f::v::r ->
    flag_overrides := (f, v)::!flag_overrides;
    enslave ["-flag"; f; v]; r
| "-p"::p::r ->
    setup_proofkind (read_proofkind (String.lowercase p));
    enslave ["-p"; p]; r
| "-pfusefulout"::f::r -> pfusefulout := Some(f); enslave ["-pfusefulout";f]; r
| "-pfformdepsout"::f::r -> pfformdepsout := Some(f); enslave ["-pfformdepsout";f]; r
| "-verb"::v::r -> verbosity := int_of_string v; enslave ["-verb"; v]; r
| "--sine_tolerance"::t::r -> let t1 = float_of_string t in if t1 >= 1.0 then (sine_tolerance := t1; sine_tolerance_vary := false; r) else raise (Failure "--sine_tolerance t must have t >= 1.0")
| "--sine_depth"::d::r -> let d1 = float_of_string d in if d1 >= 0.0 then (sine_depth := Some(d1); r) else raise (Failure "--sine_depth d must have d >= 0.0")
| "--sine_generality_threshold"::g::r -> let g1 = int_of_string g in if g1 >= 0 then (sine_generality_threshold := g1; sine_generality_threshold_vary := false; r) else raise (Failure "--sine_generality_threshold g must have g >= 0")
| "-inferences"::i::r -> Searchoption.max_searchoptions := Some (int_of_string i); enslave ["-inferences"; i]; r
| "-numaxioms"::r -> printnumaxiomsflag := true; enslave ["-numaxioms"]; r
| "-selectaxioms"::n::r -> (*** This is only to experiment with different selections and (order) of the axioms/conjecture. ***)
    selectaxiomsflag := true;
    let num = int_of_string n in
    let axs = List.map int_of_string (Utils.take num r) in
    select_axioms_list := List.rev_append axs !select_axioms_list;
    Utils.drop num r
| "-foffiles"::r -> (*** This is only for testing and debugging interaction with FO provers like E. ***)
    Eprover.foffiles := true; enslave ["-foffiles"]; r
| "-training"::f::r -> use_learning := true; training_file := Some f; enslave ["-training"; f]; r
| "-tclassify"::f::r -> use_learning := true; tclassifier_file := Some f; enslave ["-tclassify"; f]; r
| "-fclassify"::f::r -> use_learning := true; fclassifier_file := Some f; enslave ["-fclassify"; f]; r
| "-simple"::sf::r -> use_learning := true; simple_file := Some sf; enslave ["-simple"; sf]; r
| ""::r -> raise (InputHelpError "Problem processing command line arguments")
| "-"::r -> problemfile := ""; r
| opt::r ->
    if opt.[0] = '-'
    then String.iter process_short_cmd_line_arg (Utils.string_tail opt)
    else set_problemfile opt;
    r
| [] -> []

let process_command_line_args = function
  [] -> print_endline Version.version; List.iter print_endline help_lines; exit 1
| args -> Utils.iterate_list process_cmd_line_arg args


(*** -p implies -c if -c was not given. Output proof via Coq out channel. - Chad, July 2012 ***)
let set_coq () = if (not (!coq)) then (coq := true; coqoutchannel := stdout)

let load_fclassifier f =
  Utils.with_in f Fclassify.load_classifier;
  Format.printf "!!! Loaded feature classifier from %s\n%!" f;
  begin match !conjecture with
    Some (m, _) -> Fclassify.register_processed m
  | None -> () end

let load_tclassifier f =
  Utils.with_in f Tclassify.load_classifier;
  Format.printf "!!! Loaded term classifier from %s\n%!" f;
  Tclassify.set_axioms !initial_branch

let load_simple f =
  Utils.with_in f Learn.load_simple;
  Format.printf "!!! Loaded simple from %s\n%!" f

let load_learn_data () =
  Utils.mapm_option load_tclassifier !tclassifier_file;
  Utils.mapm_option load_fclassifier !fclassifier_file;
  Utils.mapm_option load_simple !simple_file

let save_training_data r f =
  Utils.with_out f (Training.save_training (Training.conjectureTerm !conjecture, !initial_branch, processed, Proofterm.refut_trms r));
  Format.printf "!!! Wrote training data to %s\n%!" f

let save_learn_data r =
  match (r, !training_file) with
    (Some r, Some f) -> save_training_data r f
  | _ -> ()

let prepare_proofterm = function
  | TSTP -> set_coq ()
  | CoqScript -> set_coq ()
  | CoqSPfTerm -> coq2 := true; set_coq ()
  | IsarScript -> set_coq () (*FIXME code in this section, and in related sections, need refactoring. names are a bit misleading.*)
  | _ -> ()


let code_status = function
  | (true , Some _) -> 10, "CounterSatisfiable"
  | (true , None  ) -> 15, "Satisfiable"
  | (false, Some _) -> 20, "Theorem"
  | (false, None  ) -> 25, "Unsatisfiable"

let n_inferences () = Queue.length Searchoption.searchoptions_retrieved
let inferences_str () = "% Inferences: " ^ string_of_int (n_inferences ())

let show_status s =
  [ "% SZS status " ^ s
  ; "% Mode: " ^ (String.concat " " !mode)
  ; inferences_str ()
  ]

let print_status s =
  if !verbosity > 0 then List.iter print_endline (show_status s)

let print_proofmsg c l =
  let enbracket s =
    if !mkproofterm = Some IsarScript then "(*" ^ s ^ "*)" else s in
  if c = stdout then List.iter print_endline (List.map enbracket l)

let print_start c l =
  let (_, status) = code_status (false, !conjecture) in
  print_proofmsg c (show_status status @ l)

let print_end c l = print_proofmsg c l; if c != stdout then close_out c else flush c

let try_proofout f =
  try f()
  with CoqProofTooBig coqproofsize ->
    if (!verbosity > 0) then Printf.printf "%% SZS status Success\nProof Too Big: %d steps\n" coqproofsize;
    exit 26

let print_proofterm_full c r = function
  | TSTP ->
      print_start c ["% SZS output start Proof"];
      try_proofout (fun () -> Proofterm.print_tstp c r);
      print_end c ["% SZS output end Proof"]
  | CoqScript ->
      print_start c ["% SZS output start Proof"; "% Coq Proof Script"];
      try_proofout (fun () -> Proofterm.print_coq_proofscript c r);
      print_end c ["% SZS output end Proof"]
  | CoqSPfTerm ->
      print_start c ["% SZS output start Proof"; "% Coq Proof Script"];
      try_proofout (fun () -> Proofterm.print_coq_sproofterm c r);
      print_end c ["% SZS output end Proof"]
  | HOCore ->
      print_start c ["% Higher-Order Unsat Core BEGIN"];
      Proofterm.print_hocore c r;
      print_end c ["% Higher-Order Unsat Core END"]
  | IsarScript ->
      print_start c ["% SZS output start Proof"; "% Isar Proof Script"];
      try_proofout (fun () -> Proofterm.print_coq_proofscript c r);
      print_end c ["% SZS output end Proof"]
  | PfInfo ->
      print_start c ["% Pf Info BEGIN"];
      Proofterm.print_pfinfo c r;
      print_end c ["% Pf Info END"]
  | PfUseful ->
      print_start c ["% Pf Useful BEGIN"];
      Proofterm.print_pfuseful c r !pfusefulout;
      print_end c ["% Pf Useful END"]
  | PfFormdeps ->
      print_start c ["% Pf Formdeps BEGIN"];
      Proofterm.print_pfformdeps c r !pfformdepsout;
      print_end c ["% Pf Formdeps END"]
  | _ -> ()


let prepare_coq () =
  if (!coq) then coq_init();
  if (!coqlocalfile) then read_coq_file (!problemfile) else read_thf_file (!problemfile) (find_read_thf_file (Filename.dirname (!problemfile)));
  if ((!coq) && (not (!coq2))) && (not (!slave)) then print_coqsig !coqoutchannel


let set_timeouts s =
  if (s > 0.0) then begin
    if (!nontheorem && get_bool_flag "SPLIT_GLOBAL_DISJUNCTIONS" && s >= 0.2)
    then (set_timer (s *. 0.5); mult_timeout 0.5)
    else (set_timer s; timeout := Some 0.0)
  end

let auto_schedule () =
  if (!nontheorem) then "schedule_nontheorem"
  else "schedule_3_1"

let get_schedule = function
  [] -> load_schedule (auto_schedule ())
| st -> List.concat (List.rev_map load_schedule st)

let get_sine_schedule = function
  [] -> load_sine_schedule (auto_schedule ())
| st -> List.concat (List.rev_map load_sine_schedule st)
      
let slave_cmd s m arg =
  let mode = ["-m"; m] in
  let u = Unix.gettimeofday() in
  let rmtm = !endtime -. u in
  let timo =
    match s with
    | Some(s) ->
	if rmtm > s then
	  ["-t"; string_of_float s;"-ht"; string_of_float rmtm]
	else
	  ["-t"; string_of_float rmtm]
    | None -> []
  in
  String.concat " " ((List.rev !slaveargs) @ arg @ mode @ timo @ ["-preparsed"])

(*** If the slave got a final result, then use it. ***)
let handle_slave_return pstatus =
  if (!verbosity > 4) then
    Printf.printf "slave returned with status %d\n" pstatus;
  if (pstatus >= 10) && (pstatus <= 26) then exit pstatus else if (pstatus = 6) then raise IncompleteSatisfiable else ()

let run_slave final s m =
  let cmd = slave_cmd s m (if final then [] else ["-slave"]) in
  if (!verbosity > 4) then print_endline ("Starting slave: " ^ cmd);
  flush stdout; (*** 2015, to prevent race conditions with output of main process vs. slave process ***)
  let (myout,myin,myerr) = Unix.open_process_full cmd [| |] in
  output_value myin !name_base_list;
  output_value myin (List.map (fun (x,_,tp) -> (x,tp)) !name_trm_list);
  output_value myin !is_of_names;
  output_value myin !all_of_names;
  output_value myin !probsig;
  close_out myin;
  try
    while true do
      let l = input_line myout in
      Printf.printf "%s\n" l
    done
  with End_of_file ->
    match Unix.close_process_full (myout,myin,myerr) with
    | (Unix.WEXITED pstatus) ->
	handle_slave_return pstatus; if final then exit pstatus
    | _ ->
	if (!verbosity > 0) then
	  begin
            print_endline "slave returned with unknown status";
            if final then exit 3
	  end

let run_mode m s1 =
  match s1 with
  | Some(s1) -> run_slave false (Some(s1)) m
  | None -> run_slave true None m

(*** total time of schedule ***)
let schedule_time = List.fold_left (fun s1 (x,s2) -> (s1 +. s2)) 0.0

let run_schedule schedule stoptime =
  let rec run_schedule_r sch stoptime schtime =
  match sch with
  | [] -> raise Timeout
  | ((m,s)::rsch) ->
      let remtime = stoptime -. Unix.gettimeofday() in
      if remtime < s then
	raise Timeout
      else
	let s2 = s *. (max 1. (remtime /. schtime)) in
	run_mode m (Some(s2));
	run_schedule_r rsch stoptime (schtime -. s)
  in
  run_schedule_r schedule stoptime (schedule_time schedule)

let rec run_schedule_notimeout schedule =
  match schedule with
  | [] -> raise Timeout (*** If nothing on schedule, then Timeout anyway. ***)
  | [(m,_)] -> (*** Last mode on schedule, call it without a timeout and without telling it it's a slave. ***)
      run_slave true None m
  | ((m,s)::rsch) ->
      run_mode m (Some(s));
      run_schedule_notimeout rsch
    
let print_num_axioms () =
  Printf.printf "(NUMAXIOMS \"%s\" %d)\n" (!problemfile) (num_axioms ());
  exit 123

let name_distance_from_conjecture ps =
  let dist : (string,float) Hashtbl.t = Hashtbl.create 10 in
  let namefreq : (string,int) Hashtbl.t = Hashtbl.create 10 in
  let visitednames : (string,float) Hashtbl.t = Hashtbl.create 10 in
  let defs = ref [] in
  let axs = ref [] in
  let g c =
    try
      Hashtbl.replace namefreq c (1 + (Hashtbl.find namefreq c));
      c
    with Not_found ->
      Hashtbl.add namefreq c 1;
      c
  in
  List.iter
    (fun z ->
      match z with
      | ProbDef(x,_,m,_,_) ->
	  let cl = List.map (fun (c,_) -> g c) (consts_of_trm [] m) in
	  defs := (x,m,cl)::!defs
      | ProbAx(x,_,m,_,_) ->
	  let cl = List.map (fun (c,_) -> g c) (consts_of_trm [] m) in
	  axs := (x,m,cl)::!axs
      | ProbConj(x,m,_,_) ->
	  Hashtbl.add dist x 0.0;
	  List.iter
	    (fun (c,a) -> ignore (g c); if not (Hashtbl.mem visitednames c) then Hashtbl.add visitednames c 1.0)
	    (consts_of_trm [] m))
    ps;
  let cont = ref true in
  let prog = ref false in
  while !cont do
    cont := false;
    prog := false;
    let defs2 = !defs in
    let axs2 = !axs in
    defs := [];
    axs := [];
    List.iter
      (fun (x,m,cl) ->
	try
	  let u = Hashtbl.find visitednames x in
	  Hashtbl.add dist x u;
	  let v = u +. 1.0 in
	  List.iter (fun c -> if not (Hashtbl.mem visitednames c) then Hashtbl.add visitednames c v) cl;
	  prog := true
	with Not_found ->
	  defs := (x,m,cl)::!defs;
	  cont := true)
      defs2;
    List.iter
      (fun (x,m,cl) ->
	let n = ref None in
	List.iter
	  (fun c ->
	    try
	      let u = Hashtbl.find visitednames c in
	      match !n with
	      | None -> n := Some(u)
	      | Some(v) -> n := Some(min u v)
	    with Not_found -> ())
	  cl;
	match !n with
	| None ->
	    axs := (x,m,cl)::!axs;
	    cont := true
	| Some(u) ->
	    let v = u +. 1.0 in
	    Hashtbl.add dist x v;
	    List.iter (fun c -> if not (Hashtbl.mem visitednames c) then Hashtbl.add visitednames c v) cl;
	    prog := true)
      axs2;
    if not !prog then
      begin
	List.iter (fun (x,_,_) -> Hashtbl.add dist x 2000.0) !defs;
	List.iter (fun (x,_,_) -> Hashtbl.add dist x 3000.0) !axs;
	cont := false
      end
  done;
  dist

let sinelike_premsel ps = (*** based on Hoder Voronkov CADE-23 2011 ***)
  let occ : (string,int) Hashtbl.t = Hashtbl.create 200 in (*** number of occurrences of symbols in defs/axioms/conjecture ***)
  let g c =
    try
      Hashtbl.replace occ c (1 + (Hashtbl.find occ c));
      c
    with Not_found ->
      Hashtbl.add occ c 1;
      c
  in
  let defs = Hashtbl.create 100 in
  let axs = Hashtbl.create 100 in
  let conjs = ref [] in
  List.iter
    (fun z ->
      match z with
      | ProbDef(x,_,m,_,_) ->
	  let cl = List.map (fun (c,_) -> g c) (consts_of_trm [] m) in (*** Note: consts_of_trm does not contain duplicates, which is important for counting here ***)
	  Hashtbl.add defs x cl
      | ProbAx(x,_,m,_,_) ->
	  let cl = List.map (fun (c,_) -> g c) (consts_of_trm [] m) in (*** Note: consts_of_trm does not contain duplicates, which is important for counting here ***)
	  Hashtbl.add axs x cl
      | ProbConj(x,m,_,_) ->
	  let cl = List.map (fun (c,_) -> g c) (consts_of_trm [] m) in (*** Note: consts_of_trm does not contain duplicates, which is important for counting here ***)
	  conjs := (x,m,cl)::!conjs)
    ps;
  let triggerreln : (string,string) Hashtbl.t = Hashtbl.create 200 in (*** const names map to axiom names they trigger ***)
  let update_triggerreln ax cl =
    let col = ref [] in
    let bd = ref None in
    List.iter
      (fun c ->
	try
	  let o = Hashtbl.find occ c in
	  let o2 = float_of_int o in
	  col := (c,o,o2)::!col;
	  match !bd with
	  | None -> bd := Some(!sine_tolerance *. o2)
	  | Some(o3) -> bd := Some(min (!sine_tolerance *. o2) (!sine_tolerance *. o3))
	with Not_found -> ())
      cl;
    match !bd with
    | None -> ()
    | Some(bd) ->
	List.iter
	  (fun (c,o,o2) ->
	    if o <= !sine_generality_threshold || o2 <= bd then
	      begin
		if !verbosity > 4 then Printf.printf "%s triggers axiom %s\n" c ax;
		Hashtbl.add triggerreln c ax
	      end) !col
  in
  Hashtbl.iter (fun x cl -> update_triggerreln x cl) axs;
  (*** The trigger relation on axioms is now defined. We will treat defns as follows: Defn of x is k+1 triggered if name x is k triggered. ***)
  let nametriggered : (string,int) Hashtbl.t = Hashtbl.create 200 in
  let itemtriggered : (string,int) Hashtbl.t = Hashtbl.create 200 in
  let defnxt = ref [] in
  let axnxt = ref [] in
  let triggername c k =
    if not (Hashtbl.mem nametriggered c) then
      begin
	Hashtbl.add nametriggered c k;
	if Hashtbl.mem defs c then defnxt := c::!defnxt;
	List.iter
	  (fun ax -> axnxt := ax::!axnxt)
	  (Hashtbl.find_all triggerreln c)
      end
  in
  List.iter
    (fun (x,m,cl) -> List.iter (fun c -> triggername c 0) cl)
    !conjs;
  let k = ref 0 in
  while not (!defnxt = [] && !axnxt = []) do
    incr k;
    begin
      match !sine_depth with
      | Some(d) -> if d < float_of_int !k then (defnxt := []; axnxt := []) (*** stop computation prematurely because of explicit depth restriction ***)
      | None -> ()
    end;
    let defnxt2 = !defnxt in
    let axnxt2 = !axnxt in
    defnxt := [];
    axnxt := [];
    List.iter
      (fun x ->
	if not (Hashtbl.mem itemtriggered x) then
	  begin
	    Hashtbl.add itemtriggered x !k;
	    List.iter (fun c -> triggername c !k) (Hashtbl.find defs x)
	  end)
      defnxt2;
    List.iter
      (fun x ->
	if not (Hashtbl.mem itemtriggered x) then
	  begin
	    Hashtbl.add itemtriggered x !k;
	    List.iter (fun c -> triggername c !k) (Hashtbl.find axs x)
	  end)
      axnxt2;
  done;
  let l = List.length ps in
  let wl = Array.make l 0.0 in
  let f x i w =
    if !verbosity > 4 then (Printf.printf "Sinelike rank of %s: %f\n" x w; flush stdout);
    wl.(i) <- w;
    w
  in
  if !verbosity > 4 then (Printf.printf "Sine t = %f, g = %d\n" !sine_tolerance !sine_generality_threshold);
  List.mapi
    (fun i z ->
      match z with
      | ProbDef(x,a,m,al,w) -> ProbDef(x,a,m,al,f x i (try float_of_int (Hashtbl.find itemtriggered x) with Not_found -> 1000000.0))
      | ProbAx(x,r,m,al,w) -> ProbAx(x,r,m,al,f x i (try float_of_int (Hashtbl.find itemtriggered x) with Not_found -> 1000000.0))
      | ProbConj(x,m,al,w) -> ProbConj(x,m,al,f x i 0.0))
    ps

let rec run_sine_schedule_3 ps dml endtime schtime =
  match dml with
  | [] -> ()
  | (r,m,s)::dmr ->
      let remtime = endtime -. Unix.gettimeofday() in
      if remtime < s then
	raise Timeout
      else
	let s2 = s *. (max 1. (remtime /. schtime)) in
	begin
	  try
	    probsig :=
	      List.filter
		(fun z ->
		  match z with
		  | ProbDef(_,_,_,_,w) -> w <= r
		  | ProbAx(_,_,_,_,w) -> w <= r
		  | ProbConj(_,_,_,w) -> w <= r)
		ps;
	    if !verbosity > 4 then
	      begin
		Printf.printf "Including those ranked <= %f\n" r;
		List.iter
		  (fun z ->
		    match z with
		    | ProbDef(x,_,_,_,_) -> Printf.printf "Including def %s\n" x
		    | ProbAx(x,_,_,_,_) -> Printf.printf "Including ax %s\n" x
		    | ProbConj(x,_,_,_) -> Printf.printf "Including conj %s\n" x)
		  !probsig
	      end;
	    let u = Unix.gettimeofday() in
	    run_slave false (Some(s2)) m (*** do not tell the last one it is the last one, since IncompleteSatisfiable might need to be raised ***)
	  with
	  | IncompleteSatisfiable -> () (*** in this case, the subproblem was determined to be satisfiable, so abort the schedule and include more of the problem (next rank limit r) ***)
	  | Timeout -> () (*** upon timeout, go to the next rank limit r ***)
	end;
	run_sine_schedule_3 ps dmr endtime (schtime -. s)

let with_sine_params t g f =
  let fullprobsig = !probsig in
  let tv = !sine_tolerance_vary in
  let gv = !sine_generality_threshold_vary in
  if tv then sine_tolerance := t;
  if gv then sine_generality_threshold := g;
  sine_tolerance_vary := false;
  sine_generality_threshold_vary := false;
  try
    f();
    sine_tolerance_vary := tv;
    sine_generality_threshold_vary := gv;
    probsig := fullprobsig
  with e ->
    sine_tolerance_vary := tv;
    sine_generality_threshold_vary := gv;
    probsig := fullprobsig;
    raise e

let run_sine_schedule_2 t g dml endtime schtime =
  match dml with
  | [(d,m,s)] ->
      with_sine_params t g
	(fun () ->
	  sine_depth := Some(d);
	  let ps2 = sinelike_premsel !probsig in
	  run_sine_schedule_3 ps2 dml endtime schtime)
  | _ ->
      with_sine_params t g
	(fun () ->
	  sine_depth := None;
	  let ps2 = sinelike_premsel !probsig in
	  run_sine_schedule_3 ps2 dml endtime schtime)
	
let rec run_sine_schedule_1 sineschedule endtime schtime =
  match sineschedule with
  | [] -> raise Timeout
  | ((t,g,dml)::sr) ->
      run_sine_schedule_2 t g dml endtime schtime;
      run_sine_schedule_1 sr endtime (schtime -. (List.fold_left (fun s1 (_,_,s2) -> (s1 +. s2)) 0.0 dml))

let run_sine_schedule sineschedule endtime schtime =
  let oldslaveargs = !slaveargs in
  let oldcompletep = !completep in
  enslave ["-i"];
  run_sine_schedule_1 sineschedule endtime schtime
	
let run_modes modes =
  try
    List.iter load_mode modes;
    List.iter (fun (k, v) -> set_flag k v) !flag_overrides;
    if (!verbosity > 8) then print_flags ();
    Utils.mapm_option prepare_proofterm !mkproofterm;
    set_timeouts (get_timeout_default 0.0);
    if !preparsed then
      begin
	let basetps : string list = input_value stdin in
	List.iter declare_base_type basetps;
	let nametrms : (string * stp) list = input_value stdin in
	List.iter
	  (fun (x,tp) ->
	    Hashtbl.add name_tp x tp;
	    Hashtbl.add name_trm x ((Name(x,tp),tp),ref false);
	    name_trm_list := (x,Name(x,tp),tp)::!name_trm_list)
	  nametrms;
	is_of_names := input_value stdin;
	List.iter (fun x -> Hashtbl.add is_of_name x ()) !is_of_names;
	all_of_names := input_value stdin;
	List.iter (fun x -> Hashtbl.add all_of_name x ()) !all_of_names;
	probsig := input_value stdin;
	List.iter init_probitem !probsig
      end
    else
      begin
	prepare_coq ();
	probsig := List.rev !probsig;
	let preweightsum =
	  List.fold_left
	    (fun v z ->
	      match z with
	      | ProbDef(_,_,_,_,w) -> v +. w
	      | ProbAx(_,_,_,_,w) -> v +. w
	      | ProbConj(_,_,_,w) -> v +. w)
	    0.0 !probsig
	in
	if preweightsum < 128.0 then (*** the problem is small enough; don't do sine ***)
	  List.iter init_probitem !probsig
	else
	  let r = match !sine_depth with Some(r) -> r | None -> 3.0 in
	  sine_depth := Some(r);
	  let ps2 = sinelike_premsel !probsig in
	  probsig :=
	    List.filter
	      (fun z ->
		match z with
		| ProbDef(_,_,_,_,w) -> w <= r
		| ProbAx(_,_,_,_,w) -> w <= r
		| ProbConj(_,_,_,w) -> w <= r)
	      ps2;
	  if !verbosity > 4 then
	    begin
	      Printf.printf "Including those ranked <= %f\n" r;
	      List.iter
		(fun z ->
		  match z with
		  | ProbDef(x,_,_,_,_) -> Printf.printf "Including def %s\n" x
		  | ProbAx(x,_,_,_,_) -> Printf.printf "Including ax %s\n" x
		  | ProbConj(x,_,_,_) -> Printf.printf "Including conj %s\n" x)
		!probsig
	    end;
	  List.iter init_probitem !probsig
      end;
    if (!printnumaxiomsflag) then print_num_axioms ();
    if (!selectaxiomsflag) then select_axioms ();
    if !use_learning then load_learn_data ();
    Eprover.setup_eprover ();
    search ()
  with
  | Refut.Unsatisfiable(r) ->
      (*** Some subgoals may have timed out and the last one reported Unsatisfiable ***)
      if (!nontheorem) then raise Timeout;
      (*** change the timeout to be the hard timeout, so that instead of continuing with the strategy schedule the rest of the time is spent trying to output this proof ***)
      begin
	match !hardtimeout with
	| Some hs ->
	    let rm = remaining_time() in
	    set_timer (hs +. rm)
	| _ -> ()
      end;
      let (code, status) = code_status (false, !conjecture) in
      if !use_learning then save_learn_data r;
      begin match (r, !mkproofterm) with
          (Some r, Some pt) -> print_proofterm_full !coqoutchannel r pt; exit code
        | (_, _) -> print_status status; exit code
      end
  | Refut.Satisfiable ->
      if not !completep then raise IncompleteSatisfiable;
      let (code, status) = code_status (true, !conjecture) in
      print_status status; exit code

let search_main () =
  match (!mode) with
  | [] ->
      starttime := Unix.gettimeofday();
      endtime := !starttime +. get_timeout_default 600.0;
      prepare_coq(); (*** parse the problem before calling slaves - Chad, Nov 2016 ***)
      probsig := List.rev !probsig;
      let preweightsum =
	List.fold_left
	  (fun v z ->
	    match z with
	    | ProbDef(_,_,_,_,w) -> v +. w
	    | ProbAx(_,_,_,_,w) -> v +. w
	    | ProbConj(_,_,_,w) -> v +. w)
	  0.0 !probsig
      in
      if preweightsum < 128.0 then (*** the problem is small enough to do the default strategy schedule ***)
	if !timeout = None then
	  run_schedule_notimeout (get_schedule !schedule_files)
	else
	  run_schedule (get_schedule !schedule_files) !endtime
      else (*** the problem is large; do premise selection and then call the schedule on slices ***)
	let sineschedule = get_sine_schedule !schedule_files in
	run_sine_schedule sineschedule !endtime (List.fold_left (fun s1 (_,_,dml) -> s1 +. (List.fold_left (fun s1 (_,_,s2) -> (s1 +. s2)) 0.0 dml)) 0.0 sineschedule)
  | modes -> run_modes modes
