open Syntax

module TermOrdered = struct type t = trm let compare = compare end

module TClassifier = Naivebayes.Classifier (TermOrdered) (Monoid.IntPairMonoid)

type classifier_t = trm TClassifier.t

let classifier : classifier_t ref = ref (TClassifier.empty ())
let axioms : (float TClassifier.Fm.t) ref = ref (TClassifier.Fm.empty)

let set_axioms ax = axioms := TClassifier.freq_map_of_list !axioms ax


(******************************************************************************)
(* convert from training examples to classification data *)

let ftrs_of_list l = TClassifier.Fm.map (fun _ -> 1.)
  (TClassifier.set_map_of_list TClassifier.Fm.empty l)

let training_to_classifier (cj, axs, proc, refut) =
  let ftrs = ftrs_of_list axs in
  let pol m = if List.mem m refut then (1,0) else (0,1) in
  Hashtbl.fold (fun m p acc -> (ftrs, (m, pol m)) :: acc) proc []


(******************************************************************************)
(* Bayes classification *)

let load_classifier c = classifier := TClassifier.load c

let get_lbl_data lbl = TClassifier.get_lbl_data !classifier lbl

let frelevance ftrs ((pos, neg), sfs) =
  let fl idf w = 0.0 (*idf *. !Flags2.bayes_lweight*)
  and fi idf (w, sf) = 0.0 (*idf *. !Flags2.bayes_iweight *. log (float_of_int sf /. tf)*)
  and fr idf sf = 0.0 (*idf *. !Flags2.bayes_rweight *. log (1. -. float_of_int sf /. (tf +. 1.)) *) in
  TClassifier.relevance (TClassifier.get_idf !classifier) fl fi fr ftrs sfs

let srelevance (pos, neg) =
  let pw = !Flags2.simp_pweight
  and nw = !Flags2.simp_nweight in
  let confidence = abs_float (pos -. neg) /. (pos +. neg) in
  let sc = confidence *. (pw *. pos +. nw *. neg) in
  sc

let relevance ftrs lbl =
  let ((pos, neg), sfs) = get_lbl_data lbl in
  if (pos, neg) = (0, 0) then 0.0
  else
  begin
    let tot = (float_of_int pos, float_of_int neg) in
    frelevance ftrs (tot, sfs) +. srelevance tot
  end

let bayes_rel ftrs m =
  if TClassifier.is_empty !classifier then 0.
  else
  begin
    let rel = relevance ftrs m in
    (*Format.printf "Bayes %f for %s\n" rel (trm_str m);*)
    rel
  end
