(* File: satallaxmain.ml *)
(* Author: Chad E Brown *)
(* Created: September 2010 - moved most of this from satallax.ml to satallaxmain.ml in September 2011 *)

open Flags
open Syntax
open State
open Log
open Search
open Error

exception InputHelpError of string
exception InputError of string

let coqfile = ref ""
let flag_overrides = ref []
let schedule_files : string list ref = ref []

let printnumaxiomsflag : bool ref = ref false
let selectaxiomsflag : bool ref = ref false

let fclassifier_file : string option ref = ref None
let tclassifier_file : string option ref = ref None
let simple_file : string option ref = ref None


let help_lines =
[ "Usage: satallax [-[VvN]] [-verb <int>] [-P <PicoMus>] [-M <modedir>] [-s <schedule>] [-m <mode>] [-flag <name> <value>] [-t <timeout in seconds>] [-inferences <int>] [-p <pfformat>] <problemfile>"
; "-M <dir> : Set the directory in which the mode/schedule files are stored."
; "       The default mode directory is the 'modes' subdirectory of the Satallax directory."
; "-s : Schedule of modes to try (previously called the 'strategy schedule')"
; "-m : Mode"
; "-P <file> : PicoMus executable file"
; "-E <file> : E prover executable file"
; "-V : Print version number and quit"
; "-v : Verbose"
; "-N : Try to determine if the problem is a non-theorem (Satisfiable or CounterSatisfiable)"
; "-verb <int> : Verbosity of given level, -verb 0 means silent"
; "-c [<file>|-] : Create a Coq version of problem, with a proof script if -p is included and proof search succeeds"
; "-C : The problem is given as a Coq file instead of as a THF file."
; "-G : A Coq file containing multiple conjectures is given. Try to prove each of them independently."
; "-p [tstp|coqscript|coqspfterm|hocore|modeltrue|model|isar]: Output a kind of proof object"
]

let comment_line_p l =
  if String.length l = 0 then true
  else l.[0] = ';'

let rec get_input_lines c =
  try 
    let line = input_line c in
    line :: get_input_lines c
  with End_of_file -> []

let get_content_lines c =
  List.filter (fun line -> not (comment_line_p line)) (get_input_lines c)

let rec set_flags = function
  flag :: value :: rest -> set_flag flag value; set_flags rest
| flag :: [] -> raise (InputError ("Value after flag " ^ flag ^ " expected"))
| [] -> ()

let read_mode_file c = set_flags (get_content_lines c)

let read_schedule_line l =
  Scanf.sscanf l "%s %f" (fun mode time -> (mode, time))

let read_schedule_file c = List.map read_schedule_line (get_content_lines c)

let load_schedule s =
  let schedfile = !Config.modedir ^ "/" ^ s in
  if not (Sys.file_exists schedfile) then
    raise (InputError ("Could not find schedule " ^ schedfile));
  Utils.with_in schedfile read_schedule_file
  
let load_mode m =
  let modefile = (!Config.modedir ^ "/" ^ m) in
  if (not (Sys.file_exists modefile)) then
    raise (InputError ("Could not find mode " ^ modefile));
  Utils.with_in modefile read_mode_file

let read_coq_file (f:string) =
  if (!verbosity > 20) then Printf.printf "Starting to read Coq file %s\n" f;
  coqinchannel := if (f = "") then stdin else (open_in f);
  let ch = Lexing.from_channel !coqinchannel in
  try
    while true do
      Coqparser.documentitem Coqlexer.token ch
    done
  with
    Coqlexer.Eof ->
      begin
	if (!verbosity > 20) then Printf.printf "End of Coq file\n";
	if ((!coqglobalfile) && (not ((!coqinchannel) = stdin))) then
	  let p = pos_in !coqinchannel in
	  let j = ref 0 in
	  begin
	    seek_in !coqinchannel 0;
	    List.iter
	      (fun (x,i) ->
		if (!verbosity > 20) then Printf.printf "End of Coq file %d %d\n" i (!j);
		match x with
		| Some g -> if (!verbosity > 20) then g stdout; g !coqoutchannel; seek_in !coqinchannel i; j := i
		| None -> while (!j < i) do (incr j; let z = input_char !coqinchannel in output_char !coqoutchannel z) done
		      )
	      (List.rev (!State.coqinticks));
	    while (!j < p) do (incr j; let z = input_char !coqinchannel in output_char !coqoutchannel z) done;
	  end;
	  close_in !coqinchannel;
	  close_out !coqoutchannel
      end

let read_thf_file (f:string) (include_fun : string -> unit) =
  let ch = Lexing.from_channel (if (f = "") then stdin else (open_in f)) in
  let old_include_fun = !st_include_fun in
  st_include_fun := include_fun;
(***  List.iter Tptp_config.process_thf (Tptp_parser.tptp_file Tptp_lexer.token ch); ***)
  ignore (Tptp_parser.tptp_file Tptp_lexer.token ch);
  if (!verbosity > 4) then Printf.printf "Finished reading thf file %s\n" f;
  st_include_fun := old_include_fun

let rec find_read_thf_file_r odir dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file odir)
  else if (String.length dir > 1) then
    find_read_thf_file_r odir (Filename.dirname dir) f
  else
    raise (FileNotFound f)
and find_read_thf_file dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file dir)
  else
    begin
      try
	let tptpdir = Sys.getenv "TPTP" in
	let tff = (tptpdir ^ "/" ^ f) in
	if (Sys.file_exists tff) then
	  read_thf_file tff (find_read_thf_file dir)
	else
	  find_read_thf_file_r dir dir f
      with
      | Not_found -> find_read_thf_file_r dir dir f
    end;;

st_find_read_thf_fun := find_read_thf_file;;

let read_proofkind = function
  "tstp" -> TSTP
| "coqscript" -> CoqScript
| "coqspfterm" -> CoqSPfTerm
| "hocore" -> HOCore
| "model" -> Model
| "modeltrue" -> ModelTrue
| "isar" -> IsarScript
| p -> raise (InputHelpError ("Unknown kind of proof " ^ p ^ " for -p"))

let setup_proofkind = function
  IsarScript ->
    mkproofterm := Some IsarScript;
    Flag.result_coq := false;
    Flag.result_isabellehol := true
| p -> mkproofterm := Some p

let enslave args = slaveargs := List.rev_append args !slaveargs

let process_short_cmd_line_arg = function
  'v' -> enslave ["-v"]; verbosity := 5
| 'V' -> print_endline Version.version; exit 0
|  a  -> raise (InputHelpError ("Unknown command line argument " ^ String.make 1 a))

let set_problemfile p =
  if !problemfile = "" then problemfile := p
  else raise (InputHelpError ("Multiple problem files passed: " ^ !problemfile
  ^ " and " ^ p))


let process_cmd_line_arg = function
  "-m"::m::r -> mode := m :: !mode; r
| "-s"::s::r -> schedule_files := s :: !schedule_files; r
| "-M"::m::r -> Config.modedir := m; enslave ["-M"; m]; r
| "-P"::p::r -> Config.picomus := p; enslave ["-P"; p]; r
| "-E"::e::r -> Config.eprover := e; enslave ["-E"; e]; r
| "-t"::t::r -> timeout := Some (float_of_string t); r
| "-C"::r -> coqlocalfile := true; enslave ["-C"]; r
| "-G"::r -> coqglobalfile := true; r
| "-c"::c::r ->
    coq := true;
    if (c = "-") then coqoutchannel := stdout
    else (coqfile := c; coqoutchannel := open_out c);
    enslave ["-c"; c]; r
| "-slave"::r -> slave := true; r
| "-N"::r -> nontheorem := true; enslave ["-N"]; r
| "-flag"::f::v::r ->
    flag_overrides := (f, v)::!flag_overrides;
    enslave ["-flag"; f; v]; r
| "-p"::p::r ->
    setup_proofkind (read_proofkind (String.lowercase p));
    enslave ["-p"; p]; r
| "-verb"::v::r -> verbosity := int_of_string v; enslave ["-verb"; v]; r
| "-inferences"::i::r -> Searchoption.max_searchoptions := Some (int_of_string i); enslave ["-inferences"; i]; r
| "-numaxioms"::r -> printnumaxiomsflag := true; enslave ["-numaxioms"]; r
| "-selectaxioms"::n::r -> (*** This is only to experiment with different selections and (order) of the axioms/conjecture. ***)
    selectaxiomsflag := true;
    let num = int_of_string n in
    let axs = List.map int_of_string (Utils.take num r) in
    select_axioms_list := List.rev_append axs !select_axioms_list;
    Utils.drop num r
| "-foffiles"::r -> (*** This is only for testing and debugging interaction with FO provers like E. ***)
    Eprover.foffiles := true; enslave ["-foffiles"]; r
| "-training"::f::r -> use_learning := true; training_file := Some f; enslave ["-training"; f]; r
| "-tclassify"::f::r -> use_learning := true; tclassifier_file := Some f; enslave ["-tclassify"; f]; r
| "-fclassify"::f::r -> use_learning := true; fclassifier_file := Some f; enslave ["-fclassify"; f]; r
| "-simple"::sf::r -> use_learning := true; simple_file := Some sf; enslave ["-simple"; sf]; r
| ""::r -> raise (InputHelpError "Problem processing command line arguments")
| "-"::r -> problemfile := ""; r
| opt::r ->
    if opt.[0] = '-'
    then String.iter process_short_cmd_line_arg (Utils.string_tail opt)
    else set_problemfile opt;
    r
| [] -> []

let process_command_line_args = function
  [] -> print_endline Version.version; List.iter print_endline help_lines; exit 1
| args -> Utils.iterate_list process_cmd_line_arg args


let check_search_pfterm_ok () =
  if (get_bool_flag "USE_E") then
    raise (GenericError("Proofs cannot currently be reported when E is used (either set flag USE_E to false or do not use -p)"))


(*** -p implies -c if -c was not given. Output proof via Coq out channel. - Chad, July 2012 ***)
let set_coq () = if (not (!coq)) then (coq := true; coqoutchannel := stdout)

let load_fclassifier f =
  Utils.with_in f Fclassify.load_classifier;
  Format.printf "!!! Loaded feature classifier from %s\n%!" f;
  begin match !conjecture with
    Some (_, m, _) -> Fclassify.register_processed m
  | None -> () end

let load_tclassifier f =
  Utils.with_in f Tclassify.load_classifier;
  Format.printf "!!! Loaded term classifier from %s\n%!" f;
  Tclassify.set_axioms !initial_branch

let load_simple f =
  Utils.with_in f Learn.load_simple;
  Format.printf "!!! Loaded simple from %s\n%!" f

let load_learn_data () =
  Utils.mapm_option load_tclassifier !tclassifier_file;
  Utils.mapm_option load_fclassifier !fclassifier_file;
  Utils.mapm_option load_simple !simple_file

let save_training_data r f =
  Utils.with_out f (Training.save_training (Training.conjectureTerm !conjecture, !initial_branch, processed, Proofterm.refut_trms r));
  Format.printf "!!! Wrote training data to %s\n%!" f

let save_learn_data r =
  match (r, !training_file) with
    (Some r, Some f) -> save_training_data r f
  | _ -> ()

let prepare_proofterm = function
  | TSTP -> set_coq ()
  | CoqScript -> set_coq ()
  | CoqSPfTerm -> coq2 := true; set_coq ()
  | IsarScript -> set_coq () (*FIXME code in this section, and in related sections, need refactoring. names are a bit misleading.*)
  | _ -> ()


let code_status = function
  | (true , Some _) -> 10, "CounterSatisfiable"
  | (true , None  ) -> 15, "Satisfiable"
  | (false, Some _) -> 20, "Theorem"
  | (false, None  ) -> 25, "Unsatisfiable"

let n_inferences () = Queue.length Searchoption.searchoptions_retrieved
let inferences_str () = "% Inferences: " ^ string_of_int (n_inferences ())

let show_status s =
  [ "% SZS status " ^ s
  ; "% Mode: " ^ (String.concat " " !mode)
  ; inferences_str ()
  ]

let print_status s =
  if !verbosity > 0 then List.iter print_endline (show_status s)

let print_proofmsg c l =
  let enbracket s =
    if !mkproofterm = Some IsarScript then "(*" ^ s ^ "*)" else s in
  if c = stdout then List.iter print_endline (List.map enbracket l)

let print_start c l =
  let (_, status) = code_status (false, !conjecture) in
  print_proofmsg c (show_status status @ l)

let print_end c l = print_proofmsg c l; if c != stdout then close_out c

let try_proofout f =
  try f()
  with CoqProofTooBig coqproofsize ->
    if (!verbosity > 0) then Printf.printf "%% SZS status Success\nProof Too Big: %d steps\n" coqproofsize;
    exit 26

let print_proofterm_full c r = function
  | TSTP ->
      print_start c ["% SZS output start Proof"];
      try_proofout (fun () -> Proofterm.print_tstp c r);
      print_end c ["% SZS output end Proof"]
  | CoqScript ->
      print_start c ["% SZS output start Proof"; "% Coq Proof Script"];
      try_proofout (fun () -> Proofterm.print_coq_proofscript c r);
      print_end c ["% SZS output end Proof"]
  | CoqSPfTerm ->
      print_start c ["% SZS output start Proof"; "% Coq Proof Script"];
      try_proofout (fun () -> Proofterm.print_coq_sproofterm c r);
      print_end c ["% SZS output end Proof"]
  | HOCore ->
      print_endline "% Higher-Order Unsat Core BEGIN";
      Proofterm.print_hocore stdout r;
      print_endline "% Higher-Order Unsat Core END"
  | IsarScript ->
      print_start c ["% SZS output start Proof"; "% Isar Proof Script"];
      try_proofout (fun () -> Proofterm.print_coq_proofscript c r);
      print_end c ["% SZS output end Proof"]
  | _ -> ()


let prepare_coq () =
  if (!coq) then coq_init();
  if (!coqlocalfile) then read_coq_file (!problemfile) else read_thf_file (!problemfile) (find_read_thf_file (Filename.dirname (!problemfile)));
  if ((!coq) && (not (!coq2))) && (not (!slave)) then print_coqsig !coqoutchannel


let set_timeouts s =
  if (s > 0.0) then begin
    if (!nontheorem && get_bool_flag "SPLIT_GLOBAL_DISJUNCTIONS" && s >= 0.2)
    then (set_timer (s *. 0.5); mult_timeout 0.5)
    else (set_timer s; timeout := Some 0.0)
  end

let auto_schedule () =
  if (!nontheorem) then "schedule_nontheorem"
  else "schedule_3_0"

let get_schedule = function
  [] -> load_schedule (auto_schedule ())
| st -> List.concat (List.rev_map load_schedule st)

let slave_cmd s m arg =
  let mode = ["-m"; m]
  and timo = match s with
      Some s -> ["-t"; string_of_float s]
    | None -> [] in
  String.concat " " ((List.rev !slaveargs) @ arg @ mode @ timo @ [!problemfile])

(*** If the slave got a final result, then use it. ***)
let handle_slave_return pstatus =
  if (!verbosity > 4) then
    Printf.printf "slave returned with status %d\n" pstatus;
  if (pstatus >= 10) then exit pstatus else ()

let run_slave final s m =
  let cmd = slave_cmd s m (if final then [] else ["-slave"]) in
  if (!verbosity > 4) then print_endline ("Starting slave: " ^ cmd);
  flush stdout; (*** 2015, to prevent race conditions with output of main process vs. slave process ***)
  match Unix.system cmd with
  | (Unix.WEXITED pstatus) ->
      handle_slave_return pstatus; if final then exit pstatus
  | _ ->
      if (!verbosity > 0) then
      begin
        print_endline "slave returned with unknown status";
        if final then exit 3
      end

let run_mode m s1 = match (s1, !timeout) with
    (Some s1, Some s2) ->
      if (s2 > s1) then
      begin
        run_slave false (Some s1) m;
        timeout := (Some (s2 -. s1))
      end
      else
      begin
        run_slave true (Some s2) m;
        raise Timeout
      end
  | (Some s1, None) -> run_slave false (Some s1) m
  | (None, s2) -> run_slave true s2 m

(*** total time of schedule ***)
let schedule_time = List.fold_left (fun s1 (x,s2) -> (s1 +. s2)) 0.0

let timeout_factor schedule =
  let t = schedule_time schedule in
  max 1. ((get_timeout_default t) /. t)

let schedule_timeouts schedule =
  let timeoutfac = timeout_factor schedule
  and maxi = List.length schedule - 1 in
  List.mapi (fun i (m, s) ->
    (m, if i < maxi then Some (s *. timeoutfac) else None)) schedule

let run_schedule schedule =
  List.iter (fun (m, s) -> run_mode m s) (schedule_timeouts schedule);
  raise Timeout

let print_num_axioms () =
  Printf.printf "(NUMAXIOMS \"%s\" %d)\n" (!problemfile) (num_axioms ());
  exit 123

let run_modes modes =
  try
    List.iter load_mode modes;
    List.iter (fun (k, v) -> set_flag k v) !flag_overrides;
    if (!verbosity > 8) then print_flags ();
    Utils.mapm_option prepare_proofterm !mkproofterm;
(*    if (mkprooftermp ()) then check_search_pfterm_ok (); *)
    set_timeouts (get_timeout_default 0.0);
    prepare_coq ();
    if (!printnumaxiomsflag) then print_num_axioms ();
    if (!selectaxiomsflag) then select_axioms ();
    if !use_learning then load_learn_data ();
    Eprover.setup_eprover ();
    search ()
  with
  | Refut.Unsatisfiable(r) ->
      (*** Some subgoals may have timed out and the last one reported Unsatisfiable ***)
      if (!nontheorem) then raise Timeout;
      let (code, status) = code_status (false, !conjecture) in
      if !use_learning then save_learn_data r;
      begin match (r, !mkproofterm) with
          (Some r, Some pt) -> print_proofterm_full !coqoutchannel r pt; exit code
        | (_, _) -> print_status status; exit code
      end
  | Refut.Satisfiable ->
      let (code, status) = code_status (true, !conjecture) in
      print_status status; exit code


let search_main () =
  match (!mode) with
  | [] -> run_schedule (get_schedule !schedule_files)
  | modes -> run_modes modes
