open State
open String
open Syntax
open Refutation
open Flag
open Norm

(** The module Variables is used to translate DeBrujin Indices into variables**)
module Variables = struct
	(** next variable counter and list of used variable names**)
	type t = int * (string list)
	let make () = (1,[])
	(** Input: Variables (n,v) 
		Output: Variables (m,v') with m>n and a new variable name x in v'  **)
	let rec push (n,v) = 
		let x = "x" ^ (string_of_int n) in
		let n = n+1 in
		if (Hashtbl.mem coq_used_names x) 
			then push (n,v)	    
  			else (n,x::v)  
	let top (_,v) = List.hd v
	let get i (_,v) = List.nth v i	
end

(** Prints type m as a Coq-formatted string on the out_channel c  **)
let rec print_stp_coq c m h p =
  match m with
  | Base x ->
	let x = try (Hashtbl.find h x) with Not_found -> failwith("print_stp_coq can't find coqname of "^x) in
      Printf.fprintf c "%s" x
  | Prop ->
      Printf.fprintf c "o"
  | Ar(a,b) ->
      begin
	if p then Printf.fprintf c "(";
	print_stp_coq c a h true;
	Printf.fprintf c " --> ";
	print_stp_coq c b h false;
	if p then Printf.fprintf c ")"
      end

(** Input: out_channel c, term m, list of bound variables 
	Invariant: m is closed, if  it is enclosed in quantifiers for the bound variables 
	Prints the term m on the channel c**)
let rec trm_to_coq c m bound lp rp =
  match m with
    Name(x,_) -> (* Definitions *)
	let x = try (Hashtbl.find coq_names x) with Not_found -> x in
      Printf.fprintf c "%s" x
  | False -> (* Bottom *)
	Printf.fprintf c "False"
  | Ap(Ap(Imp,m1),False) ->  (* Negation *)
	if ((lp < 0) && (rp < 30)) then
	begin
	  Printf.fprintf c "~ ";
	  trm_to_coq c m1 bound 30 rp;
	end
      else
	begin
	  Printf.fprintf c "(~ ";
	  trm_to_coq c m1 bound 30 (-1);
	  Printf.fprintf c ")";
	end
   | Ap(Ap(Imp,m1),m2) -> (* Implication *)
      if ((lp < 17) && (rp < 16)) then
	begin
	  trm_to_coq c m1 bound lp 17;
	  Printf.fprintf c " -> ";
	  trm_to_coq c m2 bound 16 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  trm_to_coq c m1 bound (-1) 17;
	  Printf.fprintf c " -> ";
	  trm_to_coq c m2 bound 16 (-1);
	  Printf.fprintf c ")";
	end
  | Ap(Imp,m1) -> trm_to_coq c (Lam(Prop,Ap(Ap(Imp,shift m1 0 1),DB(0,Prop)))) bound lp rp;
  | Imp -> trm_to_coq c (Lam(Prop,Lam(Prop,Ap(Ap(Imp,DB(1,Prop)),DB(0,Prop))))) bound lp rp; 
  | Ap(Forall(a),Lam(_,m1)) -> (* forall with Lam *)
      begin
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c "(";
	begin
	  Printf.fprintf c "forall";
	  print_all_coq c a m1 bound
	end;
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c ")";
      end
  | Forall(a) ->
      begin
	if ((lp >= 5000) || (rp >= 5001)) then Printf.fprintf c "(";
	Printf.fprintf c "SPi "; print_stp_coq c a coq_names true;
	if ((lp >= 5000) || (rp >= 5001)) then Printf.fprintf c ")";
      end
  | Ap(Ap(Eq(a),m1),m2) -> (* Equality *)
      if ((lp < 40) && (rp < 40)) then
	begin
	  trm_to_coq c m1 bound lp 40;
	  Printf.fprintf c " = ";
	  trm_to_coq c m2 bound 40 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  trm_to_coq c m1 bound (-1) 40;
	  Printf.fprintf c " = ";
	  trm_to_coq c m2 bound 40 (-1);
	  Printf.fprintf c ")";
	end
  | Eq(a) ->     
	if ((lp < 5000) && (rp < 5001)) then
	begin
	  Printf.fprintf c "Seq ";
	  print_stp_coq c a coq_names true;
	end
      else
	begin
	  Printf.fprintf c "(Seq ";
	  print_stp_coq c a coq_names true;
	  Printf.fprintf c ")";
	end      
(*** I'm now always explicitly giving the Stype
  | Ap(Choice(a),m) ->   (* Choice *)  
	if ((lp < 5000) && (rp < 5001)) then
	begin
	  Printf.fprintf c "@Sepsilon ";
	  trm_to_coq c m bound 5001 rp;
	end
      else
	begin
	  Printf.fprintf c "(@Sepsilon ";
	  
	  trm_to_coq c m bound 5001 (-1);
	  Printf.fprintf c ")";
	end      
***)
  | Choice(a) ->
      if ((lp < 5000) && (rp < 5001)) then
	begin
	  Printf.fprintf c "@Sepsilon ";
	  print_stp_coq c a coq_names true
	end
      else
	begin
	  Printf.fprintf c "(@Sepsilon ";
	  print_stp_coq c a coq_names true;
	  Printf.fprintf c ")"
	end
  | True -> (* Top *)
	Printf.fprintf c "True"
  | Ap(Ap(And,m1),m2) -> (* conjunction *)
      if ((lp < 21) && (rp < 20)) then
	begin
	  trm_to_coq c m1 bound lp 21;
	  Printf.fprintf c " /\\ ";
	  trm_to_coq c m2 bound 20 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  trm_to_coq c m1 bound (-1) 21;
	  Printf.fprintf c " /\\ ";
	  trm_to_coq c m2 bound 20 (-1);
	  Printf.fprintf c ")";
	end
  | And ->Printf.fprintf c "and"
  | Ap(Ap(Or,m1),m2) -> (* disjunction *)
      if ((lp < 19) && (rp < 18)) then
	begin
	  trm_to_coq c m1 bound lp 19;
	  Printf.fprintf c " \\/ ";
	  trm_to_coq c m2 bound 18 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  trm_to_coq c m1 bound (-1) 19;
	  Printf.fprintf c " \\/ ";
	  trm_to_coq c m2 bound 18 (-1);
	  Printf.fprintf c ")";
	end
  | Or -> Printf.fprintf c "or"
  | Ap(Ap(Iff,m1),m2) -> (* equivalenz *)
      if ((lp < 14) && (rp < 14)) then
	begin
	  trm_to_coq c m1 bound lp 14;
	  Printf.fprintf c " <-> ";
	  trm_to_coq c m2 bound 14 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  trm_to_coq c m1 bound (-1) 14;
	  Printf.fprintf c " <-> ";
	  trm_to_coq c m2 bound 14 (-1);
	  Printf.fprintf c ")";
	end
  | Iff -> Printf.fprintf c "iff"
  | Neg -> Printf.fprintf c "not"
  | Ap(Exists(a),Lam(_,m1)) -> (* exist *)
      begin
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c "(";
	begin
	  print_ex_coq c a m1 bound
	end;
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c ")";
      end
  | Exists(a) ->
      begin
	if ((lp >= 5000) || (rp >= 5001)) then Printf.fprintf c "(";
	Printf.fprintf c "SSigma "; print_stp_coq c a coq_names true;
	if ((lp >= 5000) || (rp >= 5001)) then Printf.fprintf c ")";
      end
  | DB(i,a) -> (* Bound variable *)
	Printf.fprintf c "%s" (Variables.get i bound)
  | Lam(a,m) ->
      begin
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c "(";
	begin
	  Printf.fprintf c "fun";
	  print_lam_coq c a m bound
	end;
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c ")";
      end
  | Ap(m1,m2) ->     
	if ((lp < 5000) && (rp < 5001)) then
	begin
	  trm_to_coq c m1 bound lp 5000;
	  Printf.fprintf c " ";
	  trm_to_coq c m2 bound 5001 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  trm_to_coq c m1 bound (-1) 5000;
	  Printf.fprintf c " ";
	  trm_to_coq c m2 bound 5001 (-1);
	  Printf.fprintf c ")";
	end      
  | _ -> raise (GenericSyntaxError ("Unknown case in trm_to_coq version : " ^ (trm_str m)))

 (* Prints consecutive lambda-terms as a single fun in Coq. *) 
and print_lam_coq c a m bound =
	let bound = Variables.push bound in
	Printf.fprintf c " ("; Printf.fprintf c "%s" (Variables.top bound); Printf.fprintf c ":"; print_stp_coq c a coq_names false; Printf.fprintf c ")";
	match m with
		| Lam(b,m') -> print_lam_coq c b m' bound
		| _ -> Printf.fprintf c " => "; trm_to_coq c m bound (-1) (-1)

(* Prints consecutive forall-terms together with the corresponding lambda-terms as a single forall in Coq. *) 		
and print_all_coq c a m bound =
  let bound = Variables.push bound in
  Printf.fprintf c " ("; Printf.fprintf c "%s" (Variables.top bound); Printf.fprintf c ":"; print_stp_coq c a coq_names false; Printf.fprintf c ")";
  match m with
  | Ap(Forall(a'),Lam(_,m'))-> print_all_coq c a' m' bound
  | _ -> Printf.fprintf c ", "; trm_to_coq c m bound (-1) (-1)

(* Prints an exist-term together with the corresponding lambda-term as an exists in Coq. *) 		
and print_ex_coq c a m bound =
 	let bound = Variables.push bound in
	Printf.fprintf c "exists"; Printf.fprintf c " %s" (Variables.top bound); 
	Printf.fprintf c ":"; print_stp_coq c a coq_names false; 
        Printf.fprintf c ", ";
	trm_to_coq c m bound (-1) (-1)

(** Input: Name x, Type a, association list (term -> hypothesis name) hyp 
	Output: name of the hypothesis that defines x as a choice operator **)
let get_Choicop_axiom x a hyp = 
let ao = Ar(a,Prop) in
let m1 = Ap (Forall (ao),Lam (ao,Ap (Forall a,Lam (a,Ap (Ap (Imp, Ap (DB (1, ao), DB (0, a))),
	 Ap (DB (1, ao),Ap (Name (x, Ar (ao, a)), DB (1, ao)))))))) in
let m2 = Ap (Forall (ao),Lam (ao,Ap(Ap (Imp,Ap(Ap (Imp,Ap (Forall a,Lam (a,
	Ap (Ap (Imp, Ap (DB (1, ao), DB (0, a))),False)))),False)),Ap (DB (0, ao),
	Ap (Name (x, Ar (ao, a)),DB (0, ao))))))in
let m3 = Ap (Forall (ao),Lam (ao,Ap(Ap (Imp,Ap (Exists a,Lam (a,
	Ap (DB (1, ao), DB (0, a))))),Ap (DB (0, ao),
	Ap (Name (x, Ar (ao, a)),DB (0, ao))))))in
try
let (m,h)= List.find (fun (m,h) -> m = m1 || m = m2 || m = m3 ) hyp in h
with Not_found -> "missing_choice_axiom_for"^x


let next_fresh_hyp : int ref = ref 0

(** Input: unit
	Output: returns next fresh hypothesis name **)
let rec get_hyp_name hyp =
	let x = "H" ^ (string_of_int (!next_fresh_hyp)) in
	incr next_fresh_hyp;
	if (Hashtbl.mem coq_used_names x) 
	then get_hyp_name hyp
  	else x 

let rec find_fresh_consts n const =
  begin
    match n with 
    | Name(x,a) ->
	let x =try Hashtbl.find coq_names x with
	  Not_found
	  ->
	    Hashtbl.iter (fun x y -> Printf.printf "> %s %s\n" x y) coq_names; (* delete me *)
	    failwith ("add_fresh_const can't find "^x^" in coq_names") in
	if List.mem_assoc x const then [] else [(x,a)] 
    | Ap(m1,m2) -> find_fresh_consts m1 const @ find_fresh_consts m2 const
    | Lam(_,m) -> find_fresh_consts m const
    | _ -> []
  end

(** Input: out_channel c, association list (constant name -> type) const, term n, Space string sp 
	Output: prints inhabitation tactic for fresh constants on c and returns an updated list const **)	
let add_fresh_const c const n sp =
  List.fold_left
    (fun cons (x,a) -> 
      if List.mem (x,a) cons then cons 
      else (Printf.fprintf c "%stab_inh (" sp; print_stp_coq c a coq_names false; Printf.fprintf c ") %s.\n" x ;(x,a)::cons))
    const (find_fresh_consts (coqnorm n) const)

let rec lookup w s hyp =
  try
    List.assoc s hyp
  with
  | Not_found ->
      Printf.printf "%s: Could not find Coq hyp name\ns = %s\nhyp:\n" w (trm_str s);
      List.iter (fun (m,h) -> Printf.printf "%s: %s\n" h (trm_str m)) hyp;
      failwith ("Could not find Coq hyp name")
 
(** Input: out_channel c, refutation r, association list (term -> hypothesis name) hyp, association list (constant name -> type) const, Space string sp 
	Output: unit, prints refutation r to c **)
let rec ref_coq1 c r hyp const sp=
	match r with
 | Conflict(s,ns) -> 			
	Printf.fprintf c "%stab_conflict %s %s.\n" sp (lookup "0" (coqnorm s) hyp) (lookup "1" (coqnorm ns) hyp)
 | Fal(_) -> 				
	Printf.fprintf c "%stab_false %s.\n" sp (lookup "2" False hyp) 
 | NegRefl(s) -> 			
	Printf.fprintf c "%stab_refl %s.\n" sp (lookup "3" (coqnorm s) hyp)
 | Implication(h,s,t,r1,r2) -> 	
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_imp %s %s.\n" sp (lookup "4" (coqnorm h) hyp) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm t,h1)::hyp) const (sp^" ");
 | Disjunction(h,s,t,r1,r2) ->
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_or %s %s.\n" sp (lookup "5" (coqnorm h) hyp) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm t,h1)::hyp) const (sp^" "); 	
 | NegConjunction(h,s,t,r1,r2) ->
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_nand %s %s.\n" sp (lookup "6" (coqnorm h) hyp) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm t,h1)::hyp) const (sp^" ");  
 | NegImplication(h,s,t,r1) ->
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_negimp %s %s %s.\n" sp (lookup "7" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const sp
 | Conjunction(h,s,t,r1) ->
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_and %s %s %s.\n" sp (lookup "8" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const sp
 | NegDisjunction(h,s,t,r1) ->
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_nor %s %s %s.\n" sp (lookup "9" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const sp
 | All(h,s,r1,a,m,n) ->
	let const = add_fresh_const c const n sp in
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_all %s (" sp (lookup "10" (coqnorm h) hyp); 
	(trm_to_coq c n (Variables.make ()) (-1) (-1));
	Printf.fprintf c ") %s.\n" h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const sp
 | NegAll(h,s,r1,a,m,x) ->
	let h1 = get_hyp_name() in
	let x = ( Hashtbl.find coq_names x ) in
	Printf.fprintf c "%stab_negall %s %s %s.\n" sp (lookup "11" (coqnorm h) hyp) x h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) ((x,a)::const) sp
 | Exist(h,s,r1,a,m,x) ->
	let h1 = get_hyp_name() in
	let x = ( Hashtbl.find coq_names x ) in
	Printf.fprintf c "%stab_ex %s %s %s.\n" sp (lookup "12" (coqnorm h) hyp) x h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) ((x,a)::const) sp
 | NegExist(h,s,r1,a,m,n) ->
	let const = add_fresh_const c const n sp in
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_negex %s (" sp (lookup "13" (coqnorm h) hyp); 
	(trm_to_coq c n (Variables.make ()) (-1) (-1));
	Printf.fprintf c ") %s.\n" h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const sp	
 | Mating(h1,h2, ss, rs) ->
	let h3 = get_hyp_name() in	
	Printf.fprintf c "%stab_mat %s %s %s.\n" sp (lookup "14" (coqnorm h1) hyp) (lookup "15" (coqnorm h2) hyp) h3;
	List.iter (fun (s,r) -> ref_coq1 c r ((coqnorm s,h3)::hyp) const (sp^" ")) (List.combine ss rs)
 | Decomposition(h1, ss, rs) ->
	let h3 = get_hyp_name() in	
	Printf.fprintf c "%stab_dec %s %s.\n" sp (lookup "16" (coqnorm h1) hyp) h3;
	List.iter (fun (s,r) -> ref_coq1 c r ((coqnorm s,h3)::hyp) const (sp^" ")) (List.combine ss rs) 	
 | Confront(h1,h2,su,tu,sv,tv,r1,r2) ->
	let h3 = get_hyp_name() in
	let h4 = get_hyp_name() in	
	Printf.fprintf c "%stab_con %s %s %s %s.\n" sp (lookup "17" (coqnorm h1) hyp) (lookup "18" (coqnorm h2) hyp) h3 h4;
	ref_coq1 c r1 ((coqnorm su,h3)::(coqnorm tu,h4)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm sv,h3)::(coqnorm tv,h4)::hyp) const (sp^" ");	
 | Trans(h1,h2,su,r1) ->
	let h3 = get_hyp_name() in	
	Printf.fprintf c "%stab_trans %s %s %s.\n" sp (lookup "19" (coqnorm h1) hyp) (lookup "20" (coqnorm h2) hyp) h3;
	ref_coq1 c r1 ((coqnorm su,h3)::hyp) const (sp^" ");
 | NegEqualProp(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_be %s %s %s.\n" sp (lookup "21" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm (neg t),h2)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm (neg s),h1)::(coqnorm t,h2)::hyp) const (sp^" ");
 | EqualProp(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_bq %s %s %s.\n" sp (lookup "22" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm (neg s),h1)::(coqnorm (neg t),h2)::hyp) const (sp^" ");
 | NegAequivalenz(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_negiff %s %s %s.\n" sp (lookup "23" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm (neg t),h2)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm (neg s),h1)::(coqnorm t,h2)::hyp) const (sp^" ");
 | Aequivalenz(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "%stab_iff %s %s %s.\n" sp (lookup "24" (coqnorm h) hyp) h1 h2;
	ref_coq1 c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const (sp^" ");
	ref_coq1 c r2 ((coqnorm (neg s),h1)::(coqnorm (neg t),h2)::hyp) const (sp^" ");
 | NegEqualFunc(h,s,r1) ->
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_fe %s %s.\n" sp (lookup "25" (coqnorm h) hyp) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const sp
 | EqualFunc(h,s,r1) ->
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_fq %s %s.\n" sp (lookup "26" (coqnorm h) hyp) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const  sp
 | ChoiceR(eps,pred,s,t,r1,r2) -> 
     let const = add_fresh_const c const pred sp in
     let h1 = get_hyp_name() in
     begin
       match eps with
       | Choice(a) -> 
	   Printf.fprintf c "%stab_choice " sp;
	   print_stp_coq c a coq_names true;
	   Printf.fprintf c " (";
	   (trm_to_coq c pred (Variables.make ()) (-1) (-1));
	   Printf.fprintf c ") %s.\n" h1;
	   ref_coq1 c r1 ((coqnorm s,h1)::hyp) const (sp^" ");
	   ref_coq1 c r2 ((coqnorm t,h1)::hyp) const (sp^" ");
       | Name(x,Ar(Ar(a,Prop),_)) ->
	   Printf.fprintf c "%stab_choice' " sp;
	   print_stp_coq c a coq_names true;
	   Printf.fprintf c " (";
	   (trm_to_coq c eps (Variables.make ()) (-1) (-1));
	   Printf.fprintf c ") (";
	   (trm_to_coq c pred (Variables.make ()) (-1) (-1));
	   Printf.fprintf c ") %s %s.\n" (get_Choicop_axiom x a hyp) h1;
	   ref_coq1 c r1 ((coqnorm s,h1)::hyp) const (sp^" ");
	   ref_coq1 c r2 ((coqnorm t,h1)::hyp) const (sp^" ");
       | _ -> failwith "eps is not a valid epsilon"
     end
 | Cut(s,r1,r2) -> 
	let const = add_fresh_const c const s sp in
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_cut (" sp;
	(trm_to_coq c s (Variables.make ()) (-1) (-1));
	Printf.fprintf c ") %s.\n" h1;
	ref_coq1 c r2 ((coqnorm (neg s),h1)::hyp) const (sp^" ");
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const (sp^" ");
 | DoubleNegation(h,s,r1) ->
	let h1 = get_hyp_name() in	
	Printf.fprintf c "%stab_dn %s %s.\n" sp (lookup "27" (coqnorm h) hyp) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const sp;
 | Rewrite(prefix,pt,pt',r1) ->
	let h =  coqnorm (Ap(prefix,pt)) in
	let h1 = lookup "28" h hyp in	
	let s =  coqnorm (Ap(prefix,pt')) in 
	let h2 = get_hyp_name() in
	begin
	match pt with
		| True -> 	Printf.fprintf c "%stab_rew_true %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n"; 
		| And -> 	Printf.fprintf c "%stab_rew_and %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n"; 
		| Or -> 	Printf.fprintf c "%stab_rew_or %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Iff -> 	Printf.fprintf c "%stab_rew_iff %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Exists(_) -> 	Printf.fprintf c "%stab_rew_ex %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Eq(_) -> 	Printf.fprintf c "%stab_rew_sym %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Lam(_,Lam(_,Ap(DB(1,_),DB(0,_)))) -> 
				Printf.fprintf c "%stab_rew_eta %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Lam(Ar(Prop,Prop),Ap(Ap(Imp,Ap(Ap(Imp,DB(0,Prop)),False)),False)) -> 
				Printf.fprintf c "%stab_rew_dn %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Lam(_,Lam(_,Ap(Forall(_),Lam(_,(Ap(Ap(Imp,(Ap(DB(0,_),DB(2,_)))),(Ap(DB(0,_),DB(1,_)))) ))) )) -> 
				Printf.fprintf c "%stab_rew_leib1 %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Lam(_,Lam(_,Ap(Forall(_),Lam(_,(Ap(Ap(Imp,Ap(Ap(Imp,(Ap(DB(0,_),DB(2,_)))),False)),Ap(Ap(Imp,(Ap(DB(0,_),DB(1,_)))),False)) ))) )) -> 
				Printf.fprintf c "%stab_rew_leib2 %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Lam(_,Lam(_,Ap(Forall(_),Lam(_,(Ap(Ap(Imp,(Ap(Forall(_),Lam(_,(Ap(Ap(DB(1,_),DB(0,_)),DB(0,_)))))) ),(Ap(Ap(DB(0,_),DB(2,_)),DB(1,_)))) ) )) )) -> 
				Printf.fprintf c "%stab_rew_leib3 %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| Lam(_,Lam(_, Ap(Forall(_),Lam(_,(Ap(Ap(Imp,(Ap(Ap(Imp,(Ap(Ap(DB(0,_),DB(2,_)),DB(1,_)))),False) )),(Ap(Ap(Imp,(Ap(Forall(_),Lam(_,(Ap(Ap(DB(1,_),DB(0,_)),DB(0,_))))) )),False) )) ) )) )) -> 
				Printf.fprintf c "%stab_rew_leib4 %s %s (" sp h1 h2;
				(trm_to_coq c prefix (Variables.make ()) (-1) (-1));  Printf.fprintf c ") .\n";
		| _ -> failwith("unknown rewrite step found in ref_coq" ^ (trm_str pt))
	end;
	ref_coq1 c r1 ((s,h2)::hyp) const sp
 | Delta(h,s,x,r1) ->
	let h1 = (lookup "29" (coqnorm h) hyp) in	
	Printf.fprintf c "%sunfold %s in %s.\n" sp ( Hashtbl.find coq_names x ) h1;
	ref_coq1 c r1 ((coqnorm s,h1)::hyp) const sp;
 | KnownResult(s,name,al,r1) ->
     begin
       match al with
       | (_::_) ->
	   let h1 = get_hyp_name() in
	   Printf.fprintf c "%sset (%s := (%s" sp h1 name;
	   List.iter
	     (fun a ->
	       Printf.fprintf c " ";
	       print_stp_coq c a coq_names true)
	     al;
	   Printf.fprintf c ")).\n";
	   ref_coq1 c r1 ((coqnorm s,h1)::hyp) const sp
       | [] ->
	   ref_coq1 c r1 ((coqnorm s,name)::hyp) const sp
     end;
 | NYI(h,s,r1) -> failwith("NYI step found in ref_coq" )
 | Timeout -> failwith("Timeout step found in ref_coq" )
 | _ -> failwith("unknown refutation case in ref_coq" )    

 (** Prints refutation r to out_channel c **)
let ref_coq c r = 
	(* get conjecture *)
	let con =match !conjecture with Some(_,con,_)->coqnorm con | None-> False in
	(* initialise hypotheses *)
	let hyp = List.fold_left (fun l (s,pt) -> (coqnorm pt,s)::l ) [] !coqsig_hyp_trm in
	let h1 = get_hyp_name() in
  Printf.fprintf c "\ntab_start %s.\n" h1;
  ref_coq1 c r ((neg con,h1)::hyp) (!coqsig_const) ""; 
  Printf.fprintf c "Qed.\n";
  Printf.fprintf c "End SatallaxProblem.\n" 

(*** Oct 2011 (Chad): A version for simply typed version in Coq. ***)

(** Input: stp a ***)
let rec coq_stp c a p =
  begin
    match a with
    | Prop ->
	Printf.fprintf c "prop"
    | Base(_) ->
	Printf.fprintf c "set" (*** Only allow set as a base type here ***)
    | Ar(a1,a2) ->
	if p then Printf.fprintf c "(";
	coq_stp c a1 true;
	Printf.fprintf c ">";
	coq_stp c a2 false;
	if p then Printf.fprintf c ")";
  end

let rec coq_sterm c m bound lp rp =
  match m with
    Name(x,_) -> (* Definitions *)
      let x = try (Hashtbl.find coq_names x) with Not_found -> x in
      Printf.fprintf c "%s" x
  | False -> (* Bottom *)
      Printf.fprintf c "False"
  | Ap(Ap(Imp,m1),False) ->  (* Negation *)
      if ((lp < 0) && (rp < 30)) then
	begin
	  Printf.fprintf c "~ ";
	  coq_sterm c m1 bound 30 rp;
	end
      else
	begin
	  Printf.fprintf c "(~ ";
	  coq_sterm c m1 bound 30 (-1);
	  Printf.fprintf c ")";
	end
   | Ap(Ap(Imp,m1),m2) -> (* Implication *)
      if ((lp < 17) && (rp < 16)) then
	begin
	  coq_sterm c m1 bound lp 17;
	  Printf.fprintf c " -> ";
	  coq_sterm c m2 bound 16 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  coq_sterm c m1 bound (-1) 17;
	  Printf.fprintf c " -> ";
	  coq_sterm c m2 bound 16 (-1);
	  Printf.fprintf c ")";
	end
  | Ap(Imp,m1) -> coq_sterm c (Lam(Prop,Ap(Ap(Imp,shift m1 0 1),DB(0,Prop)))) bound lp rp;
  | Imp -> coq_sterm c (Lam(Prop,Lam(Prop,Ap(Ap(Imp,DB(1,Prop)),DB(0,Prop))))) bound lp rp; 
  | Ap(Forall(a),Lam(_,m1)) -> (* forall with Lam *)
      begin
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c "(";
	begin
	  Printf.fprintf c "forall";
	  coq_sall c a m1 bound
	end;
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c ")";
      end
  | Forall(a) -> coq_sterm c (Lam(Ar(a,Prop),Ap(Forall(a),Lam(a,Ap(DB(1,Ar(a,Prop)),DB(0,a)))))) bound lp rp
  | Ap(Ap(Eq(Base(_)),m1),m2) -> (* Equality *)
      if ((lp < 40) && (rp < 40)) then
	begin
	  coq_sterm c m1 bound lp 40;
	  Printf.fprintf c " = ";
	  coq_sterm c m2 bound 40 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  coq_sterm c m1 bound (-1) 40;
	  Printf.fprintf c " = ";
	  coq_sterm c m2 bound 40 (-1);
	  Printf.fprintf c ")";
	end
  | Eq(a) ->
      if ((lp < 5000) && (rp < 5001)) then
	begin
	  Printf.fprintf c "eq ";
	  coq_stp c a true;
	end
      else
	begin
	  Printf.fprintf c "(eq ";
	  coq_stp c a true;
	  Printf.fprintf c ")";
	end      
  | Choice(a) ->
      if ((lp < 5000) && (rp < 5001)) then
	begin
	  Printf.fprintf c "Eps ";
	  coq_stp c a true;
	end
      else
	begin
	  Printf.fprintf c "(Eps ";
	  coq_stp c a true;
	  Printf.fprintf c ")"
	end
  | True -> (* Top *)
      Printf.fprintf c "True"
  | Ap(Ap(And,m1),m2) -> (* conjunction *)
      if ((lp < 21) && (rp < 20)) then
	begin
	  coq_sterm c m1 bound lp 21;
	  Printf.fprintf c " /\\ ";
	  coq_sterm c m2 bound 20 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  coq_sterm c m1 bound (-1) 21;
	  Printf.fprintf c " /\\ ";
	  coq_sterm c m2 bound 20 (-1);
	  Printf.fprintf c ")";
	end
  | And ->Printf.fprintf c "and"
  | Ap(Ap(Or,m1),m2) -> (* disjunction *)
      if ((lp < 19) && (rp < 18)) then
	begin
	  coq_sterm c m1 bound lp 19;
	  Printf.fprintf c " \\/ ";
	  coq_sterm c m2 bound 18 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  coq_sterm c m1 bound (-1) 19;
	  Printf.fprintf c " \\/ ";
	  coq_sterm c m2 bound 18 (-1);
	  Printf.fprintf c ")";
	end
  | Or -> Printf.fprintf c "or"
  | Ap(Ap(Iff,m1),m2) -> (* equivalenz *)
      if ((lp < 14) && (rp < 14)) then
	begin
	  coq_sterm c m1 bound lp 14;
	  Printf.fprintf c " <-> ";
	  coq_sterm c m2 bound 14 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  coq_sterm c m1 bound (-1) 14;
	  Printf.fprintf c " <-> ";
	  coq_sterm c m2 bound 14 (-1);
	  Printf.fprintf c ")";
	end
  | Iff -> Printf.fprintf c "iff"
  | Neg -> Printf.fprintf c "not"
  | Ap(Exists(a),Lam(_,m1)) -> (* exist *)
      begin
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c "(";
	coq_sex c a m1 bound;
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c ")";
      end
  | Exists(a) ->
      begin
	if ((lp >= 5000) || (rp >= 5001)) then Printf.fprintf c "(";
	Printf.fprintf c "ex "; coq_stp c a true;
	if ((lp >= 5000) || (rp >= 5001)) then Printf.fprintf c ")";
      end
  | DB(i,a) -> (* Bound variable *)
	Printf.fprintf c "%s" (Variables.get i bound)
  | Lam(a,m) ->
      begin
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c "(";
	begin
	  Printf.fprintf c "fun";
	  coq_slam c a m bound
	end;
	if ((lp >= 0) || (rp >= 0)) then Printf.fprintf c ")";
      end
  | Ap(m1,m2) ->     
	if ((lp < 5000) && (rp < 5001)) then
	begin
	  coq_sterm c m1 bound lp 5000;
	  Printf.fprintf c " ";
	  coq_sterm c m2 bound 5001 rp;
	end
      else
	begin
	  Printf.fprintf c "(";
	  coq_sterm c m1 bound (-1) 5000;
	  Printf.fprintf c " ";
	  coq_sterm c m2 bound 5001 (-1);
	  Printf.fprintf c ")";
	end      
  | _ -> raise (GenericSyntaxError ("Unknown case in trm_to_coq version : " ^ (trm_str m)))

 (* Prints consecutive lambda-terms as a single fun in Coq. *) 
and coq_slam c a m bound =
	let bound = Variables.push bound in
	Printf.fprintf c " ("; Printf.fprintf c "%s" (Variables.top bound); Printf.fprintf c ":"; coq_stp c a false; Printf.fprintf c ")";
	match m with
		| Lam(b,m') -> coq_slam c b m' bound
		| _ -> Printf.fprintf c " => "; coq_sterm c m bound (-1) (-1)

(* Prints consecutive forall-terms together with the corresponding lambda-terms as a single forall in Coq. *) 		
and coq_sall c a m bound =
  let bound = Variables.push bound in
  Printf.fprintf c " ("; Printf.fprintf c "%s" (Variables.top bound); Printf.fprintf c ":"; coq_stp c a false; Printf.fprintf c ")";
  match m with
  | Ap(Forall(a'),Lam(_,m'))-> coq_sall c a' m' bound
  | _ -> Printf.fprintf c ", "; coq_sterm c m bound (-1) (-1)

(* Prints an exist-term together with the corresponding lambda-term as an exists in Coq. *) 		
and coq_sex c a m bound =
 	let bound = Variables.push bound in
	Printf.fprintf c "exists"; Printf.fprintf c " %s" (Variables.top bound); 
	Printf.fprintf c ":"; coq_stp c a false; 
        Printf.fprintf c ", ";
	coq_sterm c m bound (-1) (-1)

(** Input: refutation r, association list (term -> hypothesis name) hyp, association list (constant name -> type) const
	Output: unit, prints refutation r to c **)
let rec coq_spfterm c r hyp const bound =
  match r with
 | Conflict(s,ns) ->
     let s2 = coqnorm s in
     let ns2 = coqnorm ns in
     begin
       match (s2,ns2) with
       | (Ap(Ap(Eq(a),s21),s22),Ap(Ap(Imp,Ap(Ap(Eq(_),ns22),ns21)),False)) when s21 = ns21 && s22 = ns22 ->
	   begin
	     try
	       let h1 = List.assoc (Ap(Ap(Eq(a),s22),s21)) hyp in
	       Printf.fprintf c "%s %s" (lookup "1" ns2 hyp) h1
	     with
	     | Not_found ->
		 begin
		   try
		     let h2 = List.assoc (Ap(Ap(Imp,(Ap(Ap(Eq(a),s21),s22))),False)) hyp in
		     Printf.fprintf c "%s %s" h2 (lookup "1" s2 hyp)
		   with
		   | Not_found ->
		       Printf.fprintf c "%s (eq_sym " (lookup "1" ns2 hyp);
		       coq_stp c a true;
		       Printf.fprintf c " ";
		       coq_sterm c s21 bound 5001 5000;
		       Printf.fprintf c " ";
		       coq_sterm c s22 bound 5001 5000;
		       Printf.fprintf c " %s)" (lookup "0" s2 hyp)
		 end
	   end
       | _ -> Printf.fprintf c "%s %s" (lookup "1" ns2 hyp) (lookup "0" s2 hyp)
     end
 | Fal(_) ->
     Printf.fprintf c "%s" (lookup "2" False hyp) 
 | NegRefl(Ap(Ap(Imp,Ap(Ap(Eq(a),s),_)),False) as h) ->
     Printf.fprintf c "TRef ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " %s" (lookup "3" (coqnorm h) hyp);
 | DoubleNegation(h,s,r1) ->
     let h1 = get_hyp_name() in	
     Printf.fprintf c "%s (fun %s => " (lookup "27" (coqnorm h) hyp) h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c ")"
 | Implication(h,((Ap(Ap(Imp,s),False)) as s'),t,r1,r2) -> 	
     let h1 = get_hyp_name() in
     Printf.fprintf c "TImp ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " %s (fun %s => " (lookup "4" (coqnorm h) hyp) h1;
     coq_spfterm c r1 ((coqnorm s',h1)::hyp) const bound;
     Printf.fprintf c ") (fun %s => " h1;
     coq_spfterm c r2 ((coqnorm t,h1)::hyp) const bound;
     Printf.fprintf c ")"
 | NegImplication(h,s,((Ap(Ap(Imp,t),False)) as t'),r1) ->
     let h1 = get_hyp_name() in
     let h2 = get_hyp_name() in	
     Printf.fprintf c "TNImp ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " %s (fun %s %s => " (lookup "7" (coqnorm h) hyp) h1 h2;
     coq_spfterm c r1 ((coqnorm s,h1)::(coqnorm t',h2)::hyp) const bound;
     Printf.fprintf c ")"
 | Disjunction(h,s,t,r1,r2) ->
     let h1 = get_hyp_name() in
     Printf.fprintf c "TOr ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " %s (fun %s => " (lookup "4" (coqnorm h) hyp) h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c ") (fun %s => " h1;
     coq_spfterm c r2 ((coqnorm t,h1)::hyp) const bound;
     Printf.fprintf c ")"
 | NegConjunction(h,((Ap(Ap(Imp,s),False)) as s'),((Ap(Ap(Imp,t),False)) as t'),r1,r2) ->
     let h1 = get_hyp_name() in
     Printf.fprintf c "TNAnd ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " %s (fun %s => " (lookup "4" (coqnorm h) hyp) h1;
     coq_spfterm c r1 ((coqnorm s',h1)::hyp) const bound;
     Printf.fprintf c ") (fun %s => " h1;
     coq_spfterm c r2 ((coqnorm t',h1)::hyp) const bound;
     Printf.fprintf c ")"
 | Conjunction(h,s,t,r1) ->
     let h1 = get_hyp_name() in
     let h2 = get_hyp_name() in	
     Printf.fprintf c "TAnd ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " %s (fun %s %s => " (lookup "7" (coqnorm h) hyp) h1 h2;
     coq_spfterm c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const bound;
     Printf.fprintf c ")"
 | NegDisjunction(h,((Ap(Ap(Imp,s),False)) as s'),((Ap(Ap(Imp,t),False)) as t'),r1) ->
     let h1 = get_hyp_name() in
     let h2 = get_hyp_name() in	
     Printf.fprintf c "TNOr ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " %s (fun %s %s => " (lookup "7" (coqnorm h) hyp) h1 h2;
     coq_spfterm c r1 ((coqnorm s',h1)::(coqnorm t',h2)::hyp) const bound;
     Printf.fprintf c ")"
 | All((Ap(Forall(_),m1) as h),s,r1,a,m,n) ->
     let xl = find_fresh_consts n const in
     let const = List.fold_left
       (fun cons (x,b) ->
	 Printf.fprintf c "Inh ";
	 coq_stp c b true;
	 Printf.fprintf c " False (fun %s => " x;
	 (x,b)::cons
	 )
       const xl in
     let h1 = get_hyp_name() in	
     Printf.fprintf c "TAll ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_sterm c m1 bound 5001 5000;
     Printf.fprintf c " %s " (lookup "10" (coqnorm h) hyp);
     coq_sterm c n bound 5001 5000;
     Printf.fprintf c " (fun %s => " h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c ")";
     List.iter
       (fun (x,b) ->
	 Printf.fprintf c ")";
	 )
       xl;
 | NegExist(((Ap(Ap(Imp,Ap(Exists(_),m1)),False)) as h),s,r1,a,m,n) ->
     let xl = find_fresh_consts n const in
     let const = List.fold_left
       (fun cons (x,b) ->
	 Printf.fprintf c "Inh ";
	 coq_stp c b true;
	 Printf.fprintf c " False (fun %s => " x;
	 (x,b)::cons
	 )
       const xl in
     let h1 = get_hyp_name() in	
     Printf.fprintf c "TNEx ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_sterm c m1 bound 5001 5000;
     Printf.fprintf c " %s " (lookup "10" (coqnorm h) hyp);
     coq_sterm c n bound 5001 5000;
     Printf.fprintf c " (fun %s => " h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c ")";
     List.iter
       (fun (x,b) ->
	 Printf.fprintf c ")";
	 )
       xl;
 | Exist(((Ap(Exists(_),m1)) as h),s,r1,a,m,x) ->
     let h1 = get_hyp_name() in
     let x = ( Hashtbl.find coq_names x ) in
     Printf.fprintf c "TEx ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_sterm c m1 bound 5001 5000;
     Printf.fprintf c " %s (fun %s %s => " (lookup "10" (coqnorm h) hyp) x h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) ((x,a)::const) bound;
     Printf.fprintf c ")";
 | NegAll(((Ap(Ap(Imp,Ap(Forall(_),m1)),False)) as h),s,r1,a,m,x) ->
     let h1 = get_hyp_name() in
     let x = ( Hashtbl.find coq_names x ) in
     Printf.fprintf c "TNAll ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_sterm c m1 bound 5001 5000;
     Printf.fprintf c " %s (fun %s %s => " (lookup "10" (coqnorm h) hyp) x h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) ((x,a)::const) bound;
     Printf.fprintf c ")";
 | Cut(s,r1,r2) -> 
     let xl = find_fresh_consts s const in
     let const = List.fold_left
	 (fun cons (x,b) ->
	 Printf.fprintf c "Inh ";
	 coq_stp c b true;
	 Printf.fprintf c " False (fun %s => " x;
	   (x,b)::cons
	 )
       const xl in
     let h1 = get_hyp_name() in	
     Printf.fprintf c "((fun %s => " h1;
     coq_spfterm c r2 ((coqnorm (neg s),h1)::hyp) const bound;
     Printf.fprintf c ") (fun %s => " h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c "))";
     List.iter
       (fun (x,b) ->
	 Printf.fprintf c ")";
	 )
       xl;
 | Trans(((Ap(Ap(Eq(a),w),z)) as h1),((Ap(Ap(Eq(_),v),u)) as h2),(Ap(Ap(Eq(_),s),t) as st),r1) ->
     begin
     let h3 = get_hyp_name() in
     Printf.fprintf c "Ttrans ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_sterm c s bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c t bound 5001 5000;
     Printf.fprintf c " ";
     if (coqnorm w = coqnorm s) then
       begin
	 if (coqnorm v = coqnorm t) then
	   begin
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " %s (eq_sym " (lookup "10" (coqnorm h1) hyp);
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c v bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c u bound 5001 5000;
   Printf.fprintf c " %s) " (lookup "10" (coqnorm h2) hyp)
           end
         else if (coqnorm u = coqnorm t) then
	   begin
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " %s %s " (lookup "10" (coqnorm h1) hyp) (lookup "10" (coqnorm h2) hyp)
	   end
	 else
	   Printf.fprintf c "<TRANS-ERROR>"
       end
     else if (coqnorm z = coqnorm s) then
       begin
	 if (coqnorm v = coqnorm t) then
	   begin
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " (eq_sym ";
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " %s) (eq_sym " (lookup "10" (coqnorm h1) hyp);
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c v bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c u bound 5001 5000;
   Printf.fprintf c " %s) " (lookup "10" (coqnorm h2) hyp)
	   end
	 else if (coqnorm u = coqnorm t) then
	   begin
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " (eq_sym ";
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " %s) %s " (lookup "10" (coqnorm h1) hyp) (lookup "10" (coqnorm h2) hyp)
	   end
	 else
	   Printf.fprintf c "<TRANS-ERROR>"
       end
     else if (coqnorm w = coqnorm t) then
       begin
	 if (coqnorm v = coqnorm s) then
	   begin
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " (eq_sym ";
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c v bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c u bound 5001 5000;
   Printf.fprintf c " %s) %s " (lookup "10" (coqnorm h2) hyp) (lookup "10" (coqnorm h1) hyp)
	   end
	 else if (coqnorm u = coqnorm s) then
	   begin
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " (eq_sym ";
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c v bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c u bound 5001 5000;
   Printf.fprintf c " %s) (eq_sym " (lookup "10" (coqnorm h2) hyp);
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " %s) " (lookup "10" (coqnorm h1) hyp)
	   end
	 else
	   Printf.fprintf c "<TRANS-ERROR>"
       end
     else if (coqnorm z = coqnorm t) then
       begin
	 if (coqnorm v = coqnorm s) then
	   begin
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " %s %s " (lookup "10" (coqnorm h2) hyp) (lookup "10" (coqnorm h1) hyp);
	   end
	 else if (coqnorm u = coqnorm s) then
	   begin
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " %s (eq_sym " (lookup "10" (coqnorm h2) hyp);
   coq_stp c a true;
   Printf.fprintf c " ";
   coq_sterm c w bound 5001 5000;
   Printf.fprintf c " ";
   coq_sterm c z bound 5001 5000;
   Printf.fprintf c " %s) " (lookup "10" (coqnorm h1) hyp)
	   end
	 else
	   Printf.fprintf c "<TRANS-ERROR>"
       end
     else
       Printf.fprintf c "<TRANS-ERROR>";
     Printf.fprintf c "(fun %s => " h3;
     coq_spfterm c r1 ((coqnorm st,h3)::hyp) const bound;
     Printf.fprintf c ")";
     end
 | Delta(h,s,x,r1) ->
   let h1 = (lookup "29" (coqnorm h) hyp) in	
   coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
 | ChoiceR(eps,pred,s,t,r1,r2) -> 
     let xl = find_fresh_consts s const in
     let const = List.fold_left
       (fun cons (x,b) ->
	 Printf.fprintf c "Inh ";
	 coq_stp c b true;
	 Printf.fprintf c " False (fun %s => " x;
   (x,b)::cons
	 )
       const xl in
     let h1 = get_hyp_name() in
     begin
       match eps with
       | Choice(a) -> 
	   Printf.fprintf c "TEps ";
           coq_stp c a true;
	   Printf.fprintf c " ";
           coq_sterm c pred bound 5001 5000;
	   Printf.fprintf c " (fun %s => " h1;
	   coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
	   Printf.fprintf c ") (fun %s => " h1;
	   coq_spfterm c r2 ((coqnorm t,h1)::hyp) const bound;
	   Printf.fprintf c ")";
       | Name(x,Ar(Ar(a,Prop),_)) ->
	   Printf.fprintf c "CHOICE-TODO";
       | _ -> failwith "eps is not a valid epsilon"
     end;
     List.iter
       (fun (x,b) ->
	 Printf.fprintf c ")";
	 )
       xl;
 | NegEqualProp(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in
	Printf.fprintf c "TBE ";
	coq_sterm c s bound 5001 5000;
	Printf.fprintf c " ";
	coq_sterm c t bound 5001 5000;
	Printf.fprintf c " %s (fun %s %s => " (lookup "21" (coqnorm h) hyp) h1 h2;
	coq_spfterm c r1 ((coqnorm s,h1)::(coqnorm (neg t),h2)::hyp) const bound;
	Printf.fprintf c ") (fun %s %s => " h1 h2;
	coq_spfterm c r2 ((coqnorm (neg s),h1)::(coqnorm t,h2)::hyp) const bound;
	Printf.fprintf c ")";
 | EqualProp(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "TBQ ";
	coq_sterm c s bound 5001 5000;
	Printf.fprintf c " ";
	coq_sterm c t bound 5001 5000;
	Printf.fprintf c " %s (fun %s %s => " (lookup "21" (coqnorm h) hyp) h1 h2;
	coq_spfterm c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const bound;
	Printf.fprintf c ") (fun %s %s => " h1 h2;
	coq_spfterm c r2 ((coqnorm (neg s),h1)::(coqnorm (neg t),h2)::hyp) const bound;
	Printf.fprintf c ")";
 | NegAequivalenz(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in
	Printf.fprintf c "TNIff ";
	coq_sterm c s bound 5001 5000;
	Printf.fprintf c " ";
	coq_sterm c t bound 5001 5000;
	Printf.fprintf c " %s (fun %s %s => " (lookup "21" (coqnorm h) hyp) h1 h2;
	coq_spfterm c r1 ((coqnorm s,h1)::(coqnorm (neg t),h2)::hyp) const bound;
	Printf.fprintf c ") (fun %s %s => " h1 h2;
	coq_spfterm c r2 ((coqnorm (neg s),h1)::(coqnorm t,h2)::hyp) const bound;
	Printf.fprintf c ")";
 | Aequivalenz(h,s,t,r1,r2) -> 
	let h1 = get_hyp_name() in
	let h2 = get_hyp_name() in	
	Printf.fprintf c "TIff ";
	coq_sterm c s bound 5001 5000;
	Printf.fprintf c " ";
	coq_sterm c t bound 5001 5000;
	Printf.fprintf c " %s (fun %s %s => " (lookup "21" (coqnorm h) hyp) h1 h2;
	coq_spfterm c r1 ((coqnorm s,h1)::(coqnorm t,h2)::hyp) const bound;
	Printf.fprintf c ") (fun %s %s => " h1 h2;
	coq_spfterm c r2 ((coqnorm (neg s),h1)::(coqnorm (neg t),h2)::hyp) const bound;
	Printf.fprintf c ")";
 | Rewrite(prefix,pt,pt',r1) ->
   let h =  coqnorm (Ap(prefix,pt)) in
   let h1 = lookup "28" h hyp in	
   let s =  coqnorm (Ap(prefix,pt')) in 
   begin
     match pt with
     | True ->
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew prop True (~ False) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " eq_true %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | And ->
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (prop>prop>prop) and (fun x y:prop => ~(x -> ~y)) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " eq_and_imp %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Or ->
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (prop>prop>prop) or (fun x y:prop => ~x -> y) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " eq_or_imp %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Iff ->
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (prop>prop>prop) iff (eq prop) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " eq_iff %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Lam(Ar(Prop,Prop),Ap(Ap(Imp,Ap(Ap(Imp,DB(0,Prop)),False)),False)) -> 
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (prop>prop) (fun x:prop => ~~x) (fun x:prop => x) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " eq_neg_neg_id %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Exists(a) ->
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew ((";
         coq_stp c a true;
	 Printf.fprintf c ">prop)>prop) (fun f => exists x, f x) (fun f => ~forall x, ~f x) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " (eq_exists_nforall ";
         coq_stp c a true;
	 Printf.fprintf c ") %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Eq(_) -> failwith("unexpected rewrite step with Eq") (*** symmetry handled by known now ***)
     | Lam((Ar(a,b) as ab),Lam(_,Ap(DB(1,_),DB(0,_)))) ->
       (*** Could Skip etas, but don't for now so Coq can type check the result. ***)
       (*** But mark it with a comment ***)
	 let h2 = h1 ^ "_e" in
	 Printf.fprintf c "\n(** eta 1 **) TRew ((";
         coq_stp c ab true;
	 Printf.fprintf c ")>";
         coq_stp c ab true;
	 Printf.fprintf c ") (fun f x => f x) (fun f => f) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " (eq_eta2 ";
         coq_stp c a true;
	 Printf.fprintf c " ";
         coq_stp c b true;
	 Printf.fprintf c ") %s (fun %s =>\n" h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c "\n) (** eta 2 **)\n"
     | Lam(a,Lam(_,Ap(Forall(_),Lam(_,(Ap(Ap(Imp,(Ap(DB(0,_),DB(2,_)))),(Ap(DB(0,_),DB(1,_)))) ))) )) -> 
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (";
         coq_stp c a true;
	 Printf.fprintf c ">";
         coq_stp c a true;
	 Printf.fprintf c ">prop) (fun s t => forall p:";
         coq_stp c a true;
	 Printf.fprintf c ">prop, p s -> p t) (fun s t => s = t) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " (eq_leib1 ";
         coq_stp c a true;
	 Printf.fprintf c ") %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Lam(a,Lam(_,Ap(Forall(_),Lam(_,(Ap(Ap(Imp,Ap(Ap(Imp,(Ap(DB(0,_),DB(2,_)))),False)),Ap(Ap(Imp,(Ap(DB(0,_),DB(1,_)))),False)) ))) )) -> 
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (";
         coq_stp c a true;
	 Printf.fprintf c ">";
         coq_stp c a true;
	 Printf.fprintf c ">prop) (fun s t => forall p:";
         coq_stp c a true;
	 Printf.fprintf c ">prop, ~ p s -> ~ p t) (fun s t => s = t) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " (eq_leib2 ";
         coq_stp c a true;
	 Printf.fprintf c ") %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Lam(a,Lam(_,Ap(Forall(_),Lam(_,(Ap(Ap(Imp,(Ap(Forall(_),Lam(_,(Ap(Ap(DB(1,_),DB(0,_)),DB(0,_)))))) ),(Ap(Ap(DB(0,_),DB(2,_)),DB(1,_)))) ) )) )) -> 
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (";
         coq_stp c a true;
	 Printf.fprintf c ">";
         coq_stp c a true;
	 Printf.fprintf c ">prop) (fun s t => forall r:";
         coq_stp c a true;
	 Printf.fprintf c ">";
         coq_stp c a true;
	 Printf.fprintf c ">prop, (forall x, r x x) -> r s t) (fun s t => s = t) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " (eq_leib3 ";
         coq_stp c a true;
	 Printf.fprintf c ") %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | Lam(a,Lam(_, Ap(Forall(_),Lam(_,(Ap(Ap(Imp,(Ap(Ap(Imp,(Ap(Ap(DB(0,_),DB(2,_)),DB(1,_)))),False) )),(Ap(Ap(Imp,(Ap(Forall(_),Lam(_,(Ap(Ap(DB(1,_),DB(0,_)),DB(0,_))))) )),False) )) ) )) )) -> 
	 let h2 = get_hyp_name() in
	 Printf.fprintf c "TRew (";
         coq_stp c a true;
	 Printf.fprintf c ">";
         coq_stp c a true;
	 Printf.fprintf c ">prop) (fun s t => forall r:";
         coq_stp c a true;
	 Printf.fprintf c ">";
         coq_stp c a true;
	 Printf.fprintf c ">prop, ~ r s t -> ~ (forall x, r x x)) (fun s t => s = t) ";
	 coq_sterm c prefix bound 5001 5000;
	 Printf.fprintf c " (eq_leib4 ";
         coq_stp c a true;
	 Printf.fprintf c ") %s (fun %s => " h1 h2;
	 coq_spfterm c r1 ((s,h2)::hyp) const bound;
	 Printf.fprintf c ")"
     | _ -> failwith("unknown rewrite step found in ref_coq" ^ (trm_str pt))
   end;
 | NegEqualFunc(((Ap(Ap(Imp,Ap(Ap(Eq(Ar(a,b)),s1),s2)),False)) as h),s,r1) ->
     let h1 = get_hyp_name() in
     Printf.fprintf c "TFE ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_stp c b true;
     Printf.fprintf c " ";
     coq_sterm c s1 bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c s2 bound 5001 5000;
     Printf.fprintf c " %s (fun %s => " (lookup "90" (coqnorm h) hyp) h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c ")";
 | EqualFunc(((Ap(Ap(Eq(Ar(a,b)),s1),s2)) as h),s,r1) ->
     let h1 = get_hyp_name() in
     Printf.fprintf c "TFQ ";
     coq_stp c a true;
     Printf.fprintf c " ";
     coq_stp c b true;
     Printf.fprintf c " ";
     coq_sterm c s1 bound 5001 5000;
     Printf.fprintf c " ";
     coq_sterm c s2 bound 5001 5000;
     Printf.fprintf c " %s (fun %s => " (lookup "90" (coqnorm h) hyp) h1;
     coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound;
     Printf.fprintf c ")";
 | KnownResult(s,name,al,r1) ->
     begin
       match al with
       | (_::_) ->
	   let h1 = get_hyp_name() in
	   Printf.fprintf c "let %s := %s" h1 name;
	   List.iter
	     (fun a ->
	       Printf.fprintf c " ";
	       coq_stp c a true)
	     al;
	   Printf.fprintf c " in ";
	   coq_spfterm c r1 ((coqnorm s,h1)::hyp) const bound
       | [] ->
	   coq_spfterm c r1 ((coqnorm s,name)::hyp) const bound
     end
 | Decomposition(((Ap(Ap(Imp,Ap(Ap(Eq(b),tl),tr)),False)) as h1),ss, rs) ->
     coq_spfterm_dec c (lookup "91" (coqnorm h1) hyp) b tl tr (List.rev ss) (List.rev rs) hyp const bound
 | Mating(h1,h2, ss, rs) ->
     let h1c = coqnorm h1 in
     let h2c = coqnorm h2 in
     let h3 = get_hyp_name() in
     begin
       match (h1,h2,List.rev ss,List.rev rs) with (*** ss and rs **)
       | (Ap(Ap(Imp,Ap(h1p,h1s)),False),Ap(h2q,h2t),(((Ap(Ap(Imp,Ap(Ap(Eq(a),s11),s12)),False)) as s1)::sr),(r1::rr)) when h1s = s11 && h2t = s12 ->
	   Printf.fprintf c "TMat ";
	   coq_stp c (tpof h1s) true;
	   Printf.fprintf c " ";
	   coq_sterm c h2t bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h1s bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h2q bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h1p bound 5001 5000;
	   Printf.fprintf c " %s %s (fun %s => " (lookup "91" h2c hyp) (lookup "92" h1c hyp) h3;
	   coq_spfterm_dec c h3 (Ar(a,Prop)) h2q h1p sr rr ((coqnorm (Ap(Ap(Imp,Ap(Ap(Eq(Ar(a,Prop)),h1p),h2q)),False)),h3)::hyp) const bound; (** fix **)
	   Printf.fprintf c ") (fun %s => " h3;
	   coq_spfterm c r1 ((coqnorm s1,h3)::hyp) const bound;
	   Printf.fprintf c ")";
       | (Ap(Ap(Imp,Ap(h1p,h1s)),False),Ap(h2q,h2t),(((Ap(Ap(Imp,Ap(Ap(Eq(a),s11),s12)),False)) as s1)::sr),(r1::rr)) ->
	   let h4 = get_hyp_name() in
	   Printf.fprintf c "TMat ";
	   coq_stp c (tpof h1s) true;
	   Printf.fprintf c " ";
	   coq_sterm c h2t bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h1s bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h2q bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h1p bound 5001 5000;
	   Printf.fprintf c " %s %s (fun %s => let %s = eq_sym " (lookup "91" h2c hyp) (lookup "92" h1c hyp) h4 h3;
	   coq_stp c a true;
	   Printf.fprintf c " ";
	   coq_sterm c h1p bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h2q bound 5001 5000;
	   Printf.fprintf c " in ";
	   coq_spfterm_dec c h3 (Ar(a,Prop)) h2q h1p sr rr ((coqnorm (Ap(Ap(Imp,Ap(Ap(Eq(Ar(a,Prop)),h1p),h2q)),False)),h3)::(coqnorm (Ap(Ap(Imp,Ap(Ap(Eq(Ar(a,Prop)),h2q),h1p)),False)),h4)::hyp) const bound;
	   Printf.fprintf c ") (fun %s => " h3;
	   coq_spfterm c r1 ((coqnorm s1,h3)::hyp) const bound;
	   Printf.fprintf c ")";
       | (Ap(h1p,h1s),Ap(Ap(Imp,Ap(h2q,h2t)),False),(((Ap(Ap(Imp,Ap(Ap(Eq(a),s11),s12)),False)) as s1)::sr),(r1::rr)) ->
	   Printf.fprintf c "TMat ";
	   coq_stp c a true;
	   Printf.fprintf c " ";
	   coq_sterm c h1s bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h2t bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h1p bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c h2q bound 5001 5000;
	   Printf.fprintf c " %s %s (fun %s => " (lookup "91" h1c hyp) (lookup "92" h2c hyp) h3;
	   coq_spfterm_dec c h3 (Ar(a,Prop)) h1p h2q sr rr ((coqnorm (Ap(Ap(Imp,Ap(Ap(Eq(Ar(a,Prop)),h1p),h2q)),False)),h3)::hyp) const bound;
	   Printf.fprintf c ") (fun %s => " h3;
	   coq_spfterm c r1 ((coqnorm s1,h3)::hyp) const bound;
	   Printf.fprintf c ")";
       | _ -> failwith("mating step did not match expected in ref_coq::" ^ (trm_str h1) ^ "::" ^ (trm_str h2))
     end
 | Confront(h1,h2,su,tu,sv,tv,r1,r2) ->
     let h1c = coqnorm h1 in
     let h2c = coqnorm h2 in
     let sun = coqnorm su in
     let tvn = coqnorm tv in
     begin
       match (sun,tvn) with
       | (Ap(Ap(Imp,Ap(Ap(Eq(a),s),u)),False),Ap(Ap(Imp,Ap(Ap(Eq(_),t),v)),False)) ->
	   let h3 = get_hyp_name() in
	   let h4 = get_hyp_name() in	
	   Printf.fprintf c "TCon ";
	   coq_stp c a true;
	   Printf.fprintf c " ";
	   coq_sterm c s bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c t bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c u bound 5001 5000;
	   Printf.fprintf c " ";
	   coq_sterm c v bound 5001 5000;
	   (*** h1 and h2 may be reversed ***)
	   begin
	     match h1c with
	     | (Ap(Ap(Eq(_),_),_)) ->
		 Printf.fprintf c " %s %s (fun %s %s => " (lookup "81" (coqnorm h1) hyp) (lookup "82" (coqnorm h2) hyp) h3 h4
	     | _ ->
		 Printf.fprintf c " %s %s (fun %s %s => " (lookup "81" (coqnorm h2) hyp) (lookup "82" (coqnorm h1) hyp) h3 h4
	   end;
	   coq_spfterm c r1 ((sun,h3)::(coqnorm tu,h4)::hyp) const bound;
	   Printf.fprintf c ") (fun %s %s => " h3 h4;
	   coq_spfterm c r2 ((coqnorm sv,h3)::(tvn,h4)::hyp) const bound;
	   Printf.fprintf c ")";
       | _ -> failwith("confront does not match")
     end
 | NYI(h,s,r1) -> failwith("NYI step found in ref_coq" )
 | Timeout -> failwith("Timeout step found in ref_coq" )
 | _ -> failwith("unknown refutation case in ref_coq" )    
and coq_spfterm_dec c h1n b tl tr ss rs hyp const bound =
  if ((coqnorm tl) = (coqnorm tr)) then
    begin
      Printf.fprintf c "TRef ";
      coq_stp c b true;
      Printf.fprintf c " ";
      coq_sterm c tl bound 5001 5000;
      Printf.fprintf c " %s" h1n
    end
  else
    begin
      match (ss,rs,tl,tr) with
	((((Ap(Ap(Imp,Ap(Ap(Eq(a),s11),s12)),False)) as s1)::sr),(r1::rr),Ap(tl1,_),Ap(tr1,_)) ->
	  let h2 = get_hyp_name() in
	  begin
	    Printf.fprintf c "TDec ";
	    coq_stp c a true;
	    Printf.fprintf c " ";
	    coq_stp c b true;
	    Printf.fprintf c " ";
	    coq_sterm c s11 bound 5001 5000;
	    Printf.fprintf c " ";
	    coq_sterm c s12 bound 5001 5000;
	    Printf.fprintf c " ";
	    coq_sterm c tl1 bound 5001 5000;
	    Printf.fprintf c " ";
	    coq_sterm c tr1 bound 5001 5000;
	    Printf.fprintf c " %s (fun %s => " h1n h2;
	    coq_spfterm_dec c h2 (Ar(a,b)) tl1 tr1 sr rr ((coqnorm (Ap(Ap(Imp,Ap(Ap(Eq(Ar(a,b)),tl1),tr1)),False)),h2)::hyp) const bound;
	    Printf.fprintf c ") (fun %s => " h2;
	    coq_spfterm c r1 ((coqnorm s1,h2)::hyp) const bound;
	    Printf.fprintf c ")";
	  end
      | _ -> failwith "decomposition failed to render as a coq spfterm"
    end

 (** Prints refutation r to out_channel c **)
let ref_coq_spfterm c r = 
  try
    match !conjecture with
      Some(_,con,_)->
	begin
	  let con = coqnorm con in
	  match con with
	  | False -> raise Not_found
	  | _ ->
	      let hyp = List.fold_left (fun l (s,pt) -> (coqnorm pt,s)::l) [] !coqsig_hyp_trm in
	      let h1 = get_hyp_name() in
	      Printf.fprintf c "exact (NNPP ";
	      coq_sterm c con (Variables.make ()) 5001 5000;
	      Printf.fprintf c " (fun %s => " h1;
	      coq_spfterm c r ((neg con,h1)::hyp) (!coqsig_const) (Variables.make ());
	      Printf.fprintf c ")).\nQed.\n"
	end
    | None -> raise Not_found
  with
  | Not_found ->
      let hyp = List.fold_left (fun l (s,pt) -> (coqnorm pt,s)::l) [] !coqsig_hyp_trm in
      Printf.fprintf c "exact (";
      coq_spfterm c r hyp (!coqsig_const) (Variables.make ());
      Printf.fprintf c ").\nQed.\n"
