/* File: Ointerface.C
 * Author: Chad E. Brown, Nov 30, 2010
 * OCaml interface to MiniSat used by Satallax
 * Code based on Main.C in MiniSat (Niklas Eeen, Niklas Sorensson)
 * and MiniSat-ocaml (Flavio Lerda, Pietro Abate).
 * Also used Chapter 25 of _Practical OCaml_ Joshua B. Smith.
 * The most useful thing I found was the tutorial "How to wrap C functions to OCaml" by Florent Monnier:
 * http://www.linux-nantes.org/~fmonnier/ocaml/ocaml-wrapping-c.php
 */

#include <ctime>
#include <cstring>
#include <stdint.h>
#include <errno.h>

#include <signal.h>
#include <zlib.h>

#include "SimpSolver.h"

extern "C" {
#include <caml/mlvalues.h>
#include <caml/memory.h>
#include <caml/alloc.h>
#include <caml/fail.h>
#include <caml/callback.h>
}

SimpSolver*    solver;
vec<Lit>       lits;

bool redirect = true;

void printState () {
  printf("minisat state:\nnClauses: %d\nnVars:%d\n",(*solver).nClauses(),(*solver).nVars());
}

extern "C" value minisat_init(value _l) {
  CAMLparam1(_l);
  int v = Int_val(_l);
  if (solver != NULL) { delete solver; }
  solver = new SimpSolver;
  (*solver).verbosity = v;
  if (redirect && (v < 10)) { freopen("/dev/null", "w", stderr); redirect = false; } // redirect stderr to avoid reportf info being printed
  CAMLreturn(Val_unit);
}

extern "C" value minisat_addLit(value _l) {
  CAMLparam1(_l);
  int parsed_lit = Int_val(_l);
  int var = abs(parsed_lit)-1;
  while (var >= (*solver).nVars()) (*solver).newVar();
  lits.push( (parsed_lit > 0) ? Lit(var) : ~Lit(var));
  CAMLreturn(Val_unit);
}

extern "C" value minisat_addClause() {
  //  printf("calling minisat_addClause\n");
  //  printState();
  bool ret = (*solver).addClause(lits);
  lits.clear();
  //  printf("minisat_addClause: %d\n",ret);
  //  printState();
  return Val_bool(ret);
}

extern "C" value minisat_search () {
  //  printf("calling minisat_search\n");
  //  printState();
  if (!(*solver).simplify()){
    return Val_bool(0);
  }
  bool ret = (*solver).solve(true,true);
  //  printf("minisat_search: %d\nnClauses: %d\nnVars:%d\n",ret,(*solver).nClauses(),(*solver).nVars());
  //  printState();
  return Val_bool(ret);
}

// search, with 1 assumed lit
extern "C" value minisat_search1 (value _l) {
  CAMLparam1(_l);
  int parsed_lit = Int_val(_l);
  int var = abs(parsed_lit)-1;
  bool ret = true;
  if (var < (*solver).nVars()) {
    vec<Lit> a;
    a.push( (parsed_lit > 0) ? Lit(var) : ~Lit(var));
    ret = (*solver).solve(a,true,true);
  }
  CAMLreturn(Val_bool(ret));
}
