(* File: satallax.ml *)
(* Author: Chad E Brown *)
(* Created: September 2010 *)

open Flags
open Syntax
open State
open Search
open Proofterm

exception Timeout
exception InputHelpError of string
exception InputError of string
exception UnclassifiedError

let coqfile = ref "";; 

(*** Satallax 2.1 schedules.  One is chosen based on the timeout given. ***)
(*** 11 modes, 2.0 total time, run when given <= 2 s ***)
let strategy_schedule_1 = [
("mode205",0.1);
("mode175",0.2);
("mode213",0.2);
("mode233",0.2);
("mode224",0.3);
("mode223",0.2);
("mode194",0.2);
("mode257",0.2);
("mode200",0.1);
("mode204",0.2);
("mode245",0.1);
];;

(*** 22 modes, 8.0 total time, run when given <= 8 s ***)
let strategy_schedule_2 = [
("mode238",0.2);
("mode223",0.2);
("mode205",0.5);
("mode222",0.2);
("mode198",0.2);
("mode224",0.6);
("mode233",0.4);
("mode175",0.3);
("mode203",0.4);
("mode213",0.6);
("mode257",0.4);
("mode174",0.6);
("mode86",0.2);
("mode104",0.2);
("mode220",0.2);
("mode245",0.5);
("mode163",0.5);
("mode225",0.3);
("mode19c",0.3);
("mode229",0.5);
("mode253",0.5);
("mode4a",0.2);
];;

(*** 35 modes, 17.0 total time, run when given <= 17 s ***)
let strategy_schedule_3 = [
("mode194",0.2);
("mode238",0.2);
("mode205",0.5);
("mode222",0.2);
("mode198",0.2);
("mode220",0.2);
("mode175",0.5);
("mode204",1.0);
("mode233",0.6);
("mode4a",0.2);
("mode252",1.0);
("mode86",0.2);
("mode104",0.2);
("mode245",0.5);
("mode223",0.9);
("mode253",0.5);
("mode163",0.5);
("mode213",0.9);
("mode229",0.9);
("mode19c",0.3);
("mode226",0.3);
("mode225",0.8);
("mode14a",0.1);
("mode256",0.8);
("mode108",0.3);
("mode244",0.5);
("mode165",0.4);
("mode188",0.7);
("mode248",0.3);
("mode185",0.8);
("mode236",0.4);
("mode228",0.4);
("mode168",0.5);
("mode249",0.5);
("mode173",0.5);
];;

(*** 35 modes, 32.0 total time, run when given <= 32 s ***)
let strategy_schedule_4 = [
("mode238",0.2);
("mode19c",0.3);
("mode222",0.2);
("mode233",0.6);
("mode175",0.5);
("mode252",0.9);
("mode198",0.2);
("mode204",1.4);
("mode4a",0.2);
("mode245",0.5);
("mode104",0.2);
("mode213",1.6);
("mode253",0.5);
("mode224",1.8);
("mode256",0.8);
("mode244",0.5);
("mode173",0.9);
("mode214",0.2);
("mode225",1.5);
("mode163",1.1);
("mode203",1.3);
("mode229",0.9);
("mode236",0.4);
("mode188",2.0);
("mode248",0.3);
("mode223",1.9);
("mode185",0.8);
("mode249",0.5);
("mode86",1.7);
("mode174",1.4);
("mode14a",1.2);
("mode165",1.2);
("mode8a",1.4);
("mode228",1.8);
("mode226",1.1);
];;

(*** 40 modes, 64.3 total time, run when given <= 67 s ***)
let strategy_schedule_5 = [
("mode205",0.1);
("mode194",0.2);
("mode222",0.2);
("mode238",0.2);
("mode19c",0.3);
("mode233",0.6);
("mode175",0.5);
("mode213",1.6);
("mode229",0.9);
("mode4a",0.2);
("mode104",0.2);
("mode245",0.5);
("mode173",0.9);
("mode253",0.5);
("mode224",3.8);
("mode256",0.8);
("mode244",0.5);
("mode188",2.0);
("mode236",0.4);
("mode204",4.0);
("mode165",1.2);
("mode248",0.3);
("mode14a",1.2);
("mode163",3.0);
("mode225",2.7);
("mode185",0.8);
("mode86",1.7);
("mode252",3.2);
("mode223",3.2);
("mode228",1.8);
("mode226",1.9);
("mode257",2.0);
("mode249",2.1);
("mode8a",3.2);
("mode197",3.5);
("mode179",1.4);
("mode219",3.3);
("mode80",2.2);
("mode3a",3.3);
("mode187",3.9);
];;

(*** 40 modes, 103.9 total time, run when given <= 113 s ***)
let strategy_schedule_6 = [
("mode238",0.2);
("mode19c",0.3);
("mode222",0.2);
("mode233",0.6);
("mode175",0.5);
("mode198",0.2);
("mode213",1.6);
("mode4a",0.2);
("mode245",0.5);
("mode104",0.2);
("mode253",0.5);
("mode204",4.0);
("mode236",0.4);
("mode244",0.5);
("mode165",1.2);
("mode256",0.8);
("mode188",2.0);
("mode163",1.1);
("mode185",0.8);
("mode248",0.3);
("mode224",6.6);
("mode173",5.0);
("mode252",4.7);
("mode14a",1.2);
("mode226",1.9);
("mode225",6.7);
("mode228",1.8);
("mode202",6.7);
("mode257",2.0);
("mode86",5.1);
("mode249",2.1);
("mode207",7.3);
("mode8a",5.7);
("mode223",7.4);
("mode179",4.2);
("mode80",2.2);
("mode206",5.3);
("mode3a",3.3);
("mode187",3.9);
("mode171",4.7);
];;

(*** 38 modes, 167.6 total time, run when given <= 190 s ***)
let strategy_schedule_7 = [
("mode238",0.2);
("mode19c",0.3);
("mode222",0.2);
("mode175",0.3);
("mode220",0.2);
("mode236",0.4);
("mode213",1.6);
("mode244",0.5);
("mode245",0.5);
("mode253",0.5);
("mode4a",0.2);
("mode188",2.0);
("mode104",0.2);
("mode185",0.2);
("mode252",4.7);
("mode163",1.1);
("mode256",0.8);
("mode248",0.3);
("mode14a",1.2);
("mode223",7.4);
("mode226",0.3);
("mode250",11.7);
("mode233",14.6);
("mode204",16.0);
("mode228",1.8);
("mode257",2.0);
("mode249",2.1);
("mode86",5.1);
("mode206",5.3);
("mode207",7.3);
("mode8a",5.7);
("mode251",10.1);
("mode173",14.5);
("mode3a",3.3);
("mode208",11.2);
("mode179",15.6);
("mode14",11.5);
("mode202",6.7);
];;

(*** 37 modes, 391.8 total time, run when given > 190s, or no timeout is given. ***)
let strategy_schedule_8 = [
("mode222",0.6);
("mode19c",0.5);
("mode175",0.3);
("mode236",0.4);
("mode213",1.6);
("mode244",0.5);
("mode245",0.5);
("mode188",2.0);
("mode253",0.5);
("mode4a",0.5);
("mode104",0.2);
("mode185",0.2);
("mode252",4.7);
("mode163",1.1);
("mode256",0.8);
("mode248",0.3);
("mode223",7.4);
("mode233",15.0);
("mode204",28.5);
("mode228",1.8);
("mode257",2.0);
("mode249",2.1);
("mode206",5.3);
("mode8a",5.7);
("mode207",7.3);
("mode14a",17.0);
("mode250",40.0);
("mode251",22.1);
("mode173",14.5);
("mode208",11.6);
("mode179",15.8);
("mode14",20.2);
("mode86",30.4);
("mode202",6.9);
("mode238",31.7);
("mode218",23.9);
("mode165",75.6);
];;

let print_help () =
  (print_string "Usage: satallax [-[Vvp]] [-verbose <int>] [-P <PicoMus>] [-M <modedir>] [-m <modefile>] [-t <timeout in seconds>] <problemfile>";
   print_newline();
   print_string "-M <dir> : Set the directory in which the mode file is stored.";
   print_newline();
   print_string "       The default mode directory is the 'modes' subdirectory of the Satallax directory.";
   print_newline();
   print_string "-m : Mode";
   print_newline();
   print_string "-P <file> : PicoMus binary file";
   print_newline();
   print_string "-V : Print version number and quit";
   print_newline();
   print_string "-v : Verbose";
   print_newline();
   print_string "-verbose <int> : Verbosity of given level";
   print_newline();
   print_string "-c [<file>|-] : Create a Coq version of problem, with the proof term if -p is included and proof search succeeds";
   print_newline();
   print_string "-p : Print proof term in Coq notation";
   print_newline());;

(*** Load the mode (error if cannot find it) ***)
let comment_line_p l =
  if (String.length l = 0) then
    true
  else
    (l.[0] = ';');;

let rec read_mode_file mf =
  let l = input_line mf in
  if (comment_line_p l) then
    read_mode_file mf
  else
    read_mode_file_value mf l
and read_mode_file_value mf flagname =
  let l = input_line mf in
  if (comment_line_p l) then
    read_mode_file_value mf flagname
  else if (l = "true") then
    begin
      set_bool_flag flagname true;
      read_mode_file mf
    end
  else if (l = "false") then
    begin
      set_bool_flag flagname false;
      read_mode_file mf
    end
  else
    begin
      set_int_flag flagname (int_of_string l);
      read_mode_file mf
    end

let modedir = ref (Config.satallaxdir ^ "/modes")
  
let load_mode m =
  let modefile = (!modedir ^ "/" ^ m) in
  if (Sys.file_exists modefile) then
    begin
      init_flags();
      try
	read_mode_file (open_in modefile)
      with
      | End_of_file -> ()
    end
  else
    raise (InputError ("Could not find mode " ^ modefile));;

let read_thf_file (f:string) (include_fun : string -> unit) =
  let ch = Lexing.from_channel (open_in f) in
  let old_include_fun = !st_include_fun in
  st_include_fun := include_fun;
(***  List.iter Tptp_config.process_thf (Tptp_parser.tptp_file Tptp_lexer.token ch); ***)
  ignore (Tptp_parser.tptp_file Tptp_lexer.token ch);
  if (!verbosity > 4) then Printf.printf "Finished reading thf file %s\n" f;
  st_include_fun := old_include_fun

let rec find_read_thf_file_r odir dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file odir)
  else if (String.length dir > 1) then
    find_read_thf_file_r odir (Filename.dirname dir) f
  else
    raise (FileNotFound f)
and find_read_thf_file dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file dir)
  else
    begin
      try
	let tptpdir = Sys.getenv "TPTP" in
	let tff = (tptpdir ^ "/" ^ f) in
	if (Sys.file_exists tff) then
	  read_thf_file tff (find_read_thf_file dir)
	else
	  find_read_thf_file_r dir dir f
      with
      | Not_found -> find_read_thf_file_r dir dir f
    end

(*** If the slave got a final result, then use it. ***)
let handle_slave_return pstatus =
  begin
    if (!verbosity > 4) then
      Printf.printf "slave returned with status %d\n" pstatus
    else ();
    if (pstatus >= 10) then exit pstatus else ()
  end;;

(*** set_timer for timeout signal, see http://alan.petitepomme.net/cwn/2006.01.03.html#2 ***)
let set_timer s =
  ignore (Unix.setitimer Unix.ITIMER_REAL { Unix.it_interval = 0.0; Unix.it_value = s });;

try
  let argc = Array.length Sys.argv in
  let slaveargs = ref [Sys.argv.(0)] in
  begin
    if (argc = 1) then (Version.print_version(); print_newline(); print_help(); exit 1) else
    let problemfile = ref "" in
    let i = ref 1 in
    while (!i < argc) do
      (match Sys.argv.(!i) with
      | "-P" -> (incr i; slaveargs := ("-P"::!slaveargs); Config.picomus := Sys.argv.(!i))
      | "-m" -> (incr i; mode := ((Sys.argv.(!i)) :: (!mode)))
      | "-M" -> (incr i; modedir := (Sys.argv.(!i)))
      | "-t" -> (incr i; timeout := (Some (float_of_string Sys.argv.(!i))))
      | "-slave" -> (incr i; slave := true)
      | "-c" ->
	  begin
	    incr i;
	    let cf = Sys.argv.(!i) in
	    slaveargs := (cf::"-c"::!slaveargs);
	    coq := true;
	    if (cf = "-") then
	      coqchannel := Some stdout
	    else
	      begin
		coqfile := cf;
		coqchannel := Some (open_out cf)
	      end
	  end
      | "-verb" ->
	  begin
	    incr i;
	    let cf = Sys.argv.(!i) in
	    slaveargs := (cf::"-verb"::!slaveargs);
	    verbosity := (int_of_string cf)
	  end
      | "" -> raise (InputHelpError "Problem processing command line arguments")
      | option ->
	  (if (option.[0] = '-') then
            for j = 1 to String.length option - 1 do
              match option.[j] with
              | 'v' -> slaveargs := ("-v"::!slaveargs); verbosity := 5
              | 'V' -> Version.print_version() ; print_newline(); exit 0
              | 'p' -> slaveargs := ("-p"::!slaveargs); mkproofterm := true
              | _ -> raise (InputHelpError (String.concat " " ["Unknown command line argument";String.make 1 (option.[j])]))
            done
	  else
	    begin
	      problemfile := option
	    end);
	  incr i)
    done;
    (match (!mode) with
    | (_::_) ->
	begin
	  try
	    ignore (List.map load_mode (!mode));
	    if (!mkproofterm) then
	      begin
		if ((get_bool_flag "ENABLE_PATTERN_CLAUSES") || (get_bool_flag "PATTERN_CLAUSES_TRANSITIVITY_EQ")) then
		  begin (*** Proof terms are not yet supported for pattern clauses - June 2011 ***)
		    raise (GenericError ("Proof terms are not supported for given mode yet"))
		  end;
		if (not (!coq)) then
		  begin (*** -p implies -c - if -c was not given. - Chad, June 2011 ***)
		    coq := true;
		    coqchannel := Some stdout;
		  end;
	      end;
	    if (!verbosity > 8) then print_flags () else ();
	    let s = (get_timeout_default 0.0) in
	    if (s > 0.0) then
	      begin
		ignore (Sys.signal Sys.sigalrm (Sys.Signal_handle (fun signo -> raise Timeout)));
		set_timer s;
		if (!coq) then coq_init();
		read_thf_file (!problemfile) (find_read_thf_file (Filename.dirname (!problemfile)));
		if (not (!slave)) then
		  begin
		    match (!coqchannel) with
		    | Some c -> print_coqsig c
		    | None -> ()
		  end;
		search ()
	      end
	    else
	      begin
		if (!coq) then coq_init();
		read_thf_file (!problemfile) (find_read_thf_file (Filename.dirname (!problemfile)));
		if (not (!slave)) then
		  begin
		    match (!coqchannel) with
		    | Some c -> print_coqsig c
		    | None -> ()
		  end;
		search ()
	      end
	  with
	  | Unsatisfiable(r) ->
	      begin
		let c = match (!coqchannel) with Some c -> c | None -> stdout in
		begin
		  if (!mkproofterm) then begin match r with Some(r) -> print_proofterm c r | None -> () end;
		  if (c != stdout) then
		    close_out c;
		end;
		match !conjecture with
		| Some _ -> (if (!verbosity > 0) then Printf.printf "%% SZS status Theorem\n"; exit 20)
		| None -> (if (!verbosity > 0) then Printf.printf "%% SZS status Unsatisfiable\n"; exit 25)
	      end
	  | Satisfiable ->
	      begin
		match !conjecture with
		| Some _ -> (if (!verbosity > 0) then Printf.printf "%% SZS status CounterSatisfiable\n"; exit 10)
		| None -> (if (!verbosity > 0) then Printf.printf "%% SZS status Satisfiable\n"; exit 15)
	      end
	end
    | [] -> (*** Use a strategy schedule ***)
	let ltimeout = get_timeout_default 600.0 in (*** If no timeout is given and no mode is given, take a timeout of 10 minutes for these purposes.  The last mode will be run forever. ***)
	let strategy_schedule =
	  begin
	    if (ltimeout <= 2.0) then
	      strategy_schedule_1
	    else if (ltimeout <= 8.0) then
	      strategy_schedule_2
	    else if (ltimeout <= 17.0) then
	      strategy_schedule_3
	    else if (ltimeout <= 32.0) then
	      strategy_schedule_4
	    else if (ltimeout <= 67.0) then
	      strategy_schedule_5
	    else if (ltimeout <= 113.0) then
	      strategy_schedule_6
	    else if (ltimeout <= 190.0) then
	      strategy_schedule_7
	    else
	      strategy_schedule_8
	  end in
	let scheduletime = List.fold_left (fun s1 (x,s2) -> (s1 +. s2)) 0.0 strategy_schedule in (*** total time of schedule ***)
	let timeoutfac = max 1. ((get_timeout_default scheduletime) /. scheduletime) in (*** timeout factor ***)
	let schedulenum = ref (List.length strategy_schedule) in
	List.iter
	  (fun (m,s) ->
	    decr schedulenum;
	    if ((!schedulenum) > 0) then
	      let s1 = s *. timeoutfac in
	      begin
		match (!timeout) with
		| Some s2 ->
		    begin
		      if (s2 > s1) then
			begin
			  begin
			    if (!verbosity > 4) then
			      (Printf.printf "Starting slave::%s\n" (String.concat " " (List.rev ((!problemfile)::(string_of_float s1)::"-t"::m::"-m"::"-slave"::(!slaveargs)))); flush stdout);
			    match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_float s1)::"-t"::m::"-m"::"-slave"::(!slaveargs))))) with
			    | (Unix.WEXITED pstatus) ->
				handle_slave_return pstatus
			    | _ ->
				if (!verbosity > 0) then
				  Printf.printf "slave returned with unknown status\n"
				else ()
			  end;
			  timeout := (Some (s2 -. s1))
			end
		      else
			begin
			  begin
			    (*** Final call - don't tell it it's a slave. ***)
			    if (!verbosity > 4) then
			      (Printf.printf "Starting final slave::%s\n" (String.concat " " (List.rev ((!problemfile)::(string_of_float s2)::"-t"::m::"-m"::(!slaveargs)))); flush stdout);
			    match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_float s2)::"-t"::m::"-m"::(!slaveargs))))) with
			    | (Unix.WEXITED pstatus) ->
				handle_slave_return pstatus ; exit pstatus
			    | _ ->
				if (!verbosity > 0) then
				  (Printf.printf "slave returned with unknown status\n" ; exit 3)
				else ()
			  end;
			  raise Timeout
			end
		    end
		| None ->
		    begin
		      if (!verbosity > 4) then
			(Printf.printf "Starting slave::%s\n" (String.concat " " (List.rev ((!problemfile)::(string_of_float s1)::"-t"::m::"-m"::"-slave"::(!slaveargs)))); flush stdout);
		      match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_float s1)::"-t"::m::"-m"::"-slave"::(!slaveargs))))) with
		      | (Unix.WEXITED pstatus) ->
			  handle_slave_return pstatus
		      | _ ->
			  if (!verbosity > 0) then
			    Printf.printf "slave returned with unknown status\n"
			  else ()
		    end
	      end
	    else
	      begin
		match (!timeout) with
		| Some s2 ->
		    begin
		(*** Final call - don't tell it it's a slave. ***)
		      if (!verbosity > 4) then
			(Printf.printf "Starting final slave::%s\n" (String.concat " " (List.rev ((!problemfile)::(string_of_float s2)::"-t"::m::"-m"::(!slaveargs)))); flush stdout);
		      match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_float s2)::"-t"::m::"-m"::(!slaveargs))))) with
		      | (Unix.WEXITED pstatus) ->
			  handle_slave_return pstatus ; exit pstatus
		      | _ ->
			  if (!verbosity > 0) then
			    (Printf.printf "slave returned with unknown status\n" ; exit 3)
			  else ()
		    end	      
		| None ->
		    begin
		(*** Final call with no timeout - don't tell it it's a slave. ***)
		      if (!verbosity > 4) then
			(Printf.printf "Starting final slave::%s\n" (String.concat " " (List.rev ((!problemfile)::m::"-m"::(!slaveargs)))); flush stdout);
		      match (Unix.system (String.concat " " (List.rev ((!problemfile)::m::"-m"::(!slaveargs))))) with
		      | (Unix.WEXITED pstatus) ->
			  handle_slave_return pstatus ; exit pstatus
		      | _ ->
			  if (!verbosity > 0) then
			    (Printf.printf "slave returned with unknown status\n" ; exit 3)
			  else ()
		    end
	      end
		)
	  strategy_schedule;
	raise Timeout
    )
  end
with
| InputHelpError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "Input Error: %s" x; print_newline (); print_help ()); exit 1
| InputError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "Input Error: %s" x; print_newline ()); exit 1
| FileNotFound(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "File Not Found: %s" x; print_newline ()); exit 1
| NotFlag(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s is not a flag" x; print_newline ()); exit 2
| NotBoolFlag(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s is an integer flag, not a boolean flag" x; print_newline ()); exit 2
| NotIntFlag(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s is a boolean flag, not an integer flag" x; print_newline ()); exit 2
| Parsing.Parse_error -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); print_string "Syntax Error"; print_newline ()); exit 2
| ParsingError(l1,i1,l2,i2,x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "Parsing Error: %s from line %d char %d to line %d char %d\n" x l1 i1 l2 i2; print_newline ()); exit 2
| ExpectedTypeError(m,a,b) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s\nhas type\n%s\nexpected type\n%s\n" (pretrm_str m) (stp_str b) (stp_str a); print_newline ()); exit 2
| GenericError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s\n" x; print_newline ()); exit 2
| GenericSyntaxError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s\n" x; print_newline ()); exit 2
| Redeclaration(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s cannot be redeclared " x; print_newline ()); exit 2
| UnclassifiedError -> if (not (!slave)) then (print_string "% SZS status Error"; print_newline ()); exit 3
| Timeout -> if (not (!slave)) then (if (!verbosity > 0) then (print_string "% SZS status Timeout"; print_newline ())); exit 5
| e -> if (not (!slave)) then (if (!verbosity > 0) then (print_string "% SZS status Error"; print_newline (); Printf.printf "Exception: %s\n" (Printexc.to_string e))); exit 3
