open State
open String
open Syntax
open Refutation
open Flag
open Suche
open Translation
open Latex
open Coq
open Branch
open Norm
  

(** to String functions for debugging**)

(** Debug function: Alternative way to print terms **)
 let rec trm_struct m =
  match m with
    Name(x,_) -> x
  | False -> "False"
  | Imp -> "Imp"
  | Forall(_) -> "Forall"
  | Eq(_) -> "Eq"
  | Choice(_) -> "Sepsilon"
  | True -> "True"
  | And -> "And"
  | Or -> "Or"
  | Iff -> "Iff"
  | Neg -> "Neg"
  | Exists(_) -> "Exists" 
  | DB(i,a) -> "DB(" ^ (string_of_int i) ^","^ (stp_str a)  ^")"
  | Lam(a,m) -> "Lam(" ^ (stp_str a) ^ "," ^ (trm_struct m)^")"
  | Ap(m1,m2) -> "Ap("^ (trm_struct m1) ^ "," ^ (trm_struct m2) ^")"                   
       
(** Debug function: turns a refutation into a string **)
let rec ref_str r =
  match r with
 | Conflict(s,ns) -> (trm_str s) ^ " is conflicting\n"
 | Fal(_) ->"False is on the branch\n"
 | NegRefl(s) -> (trm_str s) ^ " is on the branch\n"
 | Implication(h,s,t,r1,r2) -> "use Implication rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^" or "^ (trm_str t)^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | Disjunction(h,s,t,r1,r2) -> "use Or rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^" or "^ (trm_str t)^"\n"
                               ^ ref_str r1 ^ ref_str r2 
 | NegConjunction(h,s,t,r1,r2) -> "use NegAnd rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^" or "^ (trm_str t)^"\n"
                               ^ ref_str r1 ^ ref_str r2  
 | NegImplication(h,s,t,r1) ->"use NegImplication rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^" and "^ (trm_str t)^"\n"
                               ^ ref_str r1
 | Conjunction(h,s,t,r1) ->"use And rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^" and "^ (trm_str t)^"\n"
                               ^ ref_str r1 
 | NegDisjunction(h,s,t,r1) ->"use NegOr rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^" and "^ (trm_str t)^"\n"
                               ^ ref_str r1   
 | All(h,s,r1,a,m,n) ->"use ForAll rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1
 | NegAll(h,s,r1,a,m,x) ->"use NegForAll rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1
 | Exist(h,s,r1,a,m,x) ->"use Exist rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1  
 | NegExist(h,s,r1,a,m,n) ->"use NegExist rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1    
 | Mating(h1,h2, ss, rs) -> "use Mating rule on " ^ (trm_str h1) ^" and "^(trm_str h2)^"\nto get "^ (String.concat "," (List.map (fun a -> trm_str a) ss)) ^"\n"
                               ^ (String.concat "" (List.map ref_str rs))
 | Decomposition(h1, ss, rs) -> "use Decompostion rule on " ^ (trm_str h1) ^"\nto get "^ (String.concat "," (List.map (fun a -> trm_str a) ss)) ^"\n"
                               ^ (String.concat "" (List.map ref_str rs))
 
 | Confront(h1,h2,su,tu,sv,tv,r1,r2) ->"use Confrontation rule on " ^ (trm_str h1) ^" and "^(trm_str h2)^"\nto get "^ (trm_str su)^" and " ^ (trm_str tu) ^" or "^ (trm_str sv)^" and "^ (trm_str tv)^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | Trans(h1,h2,s,r1) ->"use Transitivity rule on " ^ (trm_str h1) ^" and "^(trm_str h2)^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1
 | NegEqualProp(h,s,t,r1,r2) -> "use Boolean Extensionality rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s)^" and " ^ (trm_str (neg t)) ^" or "^ (trm_str (neg s))^" and "^ (trm_str t)^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | EqualProp(h,s,t,r1,r2) -> "use Boolean Equality rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s)^" and " ^ (trm_str t) ^" or "^ (trm_str (neg s))^" and "^ (trm_str (neg t))^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | NegAequivalenz(h,s,t,r1,r2) -> "use NegAequivalenz rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s)^" and " ^ (trm_str (neg t)) ^" or "^ (trm_str (neg s))^" and "^ (trm_str t)^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | Aequivalenz(h,s,t,r1,r2) -> "use Aequivalenz rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s)^" and " ^ (trm_str t) ^" or "^ (trm_str (neg s))^" and "^ (trm_str (neg t))^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | NegEqualFunc(h,s,r1) ->"use functional Extensionality rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1
 | EqualFunc(h,s,r1) ->"use functional Equality rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1
 | ChoiceR(eps,pred,s,t,r1,r2) -> "use Choice rule \n to get "^ (trm_str s) ^" or "^ (trm_str t)^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | Cut(s,r1,r2) -> "use analytic Cut \n to get "^ (trm_str s) ^" or "^ (trm_str (neg s)) ^"\n"
                               ^ ref_str r1 ^ ref_str r2
 | DoubleNegation(h,s,r1) ->"use DoubleNegation rule on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1 
 | Rewrite(prefix,h,s,r1) ->"use Rewrite rule on " ^ (trm_str (onlybetanorm (Ap(prefix,h)))) ^"\nto get "^ (trm_str (onlybetanorm (Ap(prefix,s)))) ^"\n"
                               ^ ref_str r1   
 | Delta(h,s,x,r1) -> "unfold "^x ^" in " ^ (trm_str h)
 | NYI(h,s,r1) ->"use NYI-normalization on " ^ (trm_str h) ^"\nto get "^ (trm_str s) ^"\n"
                               ^ ref_str r1  
 | Timeout -> "timeout"
 | _ -> failwith "unknown refutation case in ref_str"
         
(** Statistic **)

(* statcount is an attempt to guess the size of the final refutation after timeout stopped the search *)
let statcount = ref (Hashtbl.create 100) 
let update_statcount h s w b =
if b then 
let (zs,zw,n) = try Hashtbl.find !statcount h with Not_found -> (0,0,0) in
Hashtbl.replace !statcount h (zs+s,zw+w,n+1)

(* statistic extract simple information from a refutation: size, depth, width, No. cuts, No. rewrites, No. NYIs, No. timeouts *)
let statistic r  =
let _ = Hashtbl.clear !statcount in
let rec statistic1 r h b =
 match r with
 | Conflict(s,ns) -> (1,1,1,0,0,0,0)
 | Fal(_) -> (1,1,1,0,0,0,0)
 | NegRefl(s) -> (1,1,1,0,0,0,0)
 | Implication(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
        let _ = update_statcount h (s1+s2+1) (w1+w2) b in 
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | Disjunction(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | NegConjunction(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | NegImplication(_,_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)
 | Conjunction(_,_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)
 | NegDisjunction(_,_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1) 
 | All(_,_,r1,_,_,_) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)
 | NegAll(_,_,r1,_,_,_) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)
 | Exist(_,_,r1,_,_,_) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1) 
 | NegExist(_,_,r1,_,_,_) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)   
 | Mating(_,_,_, rs) -> begin ( try ignore (List.hd rs) with Failure(_) -> failwith "Mating refutation list is empty" );
	let (s1,d1,w1,c1,re1,nyi1,t1) = List.fold_left (fun (s,d,w,c,re,nyi,t) r -> let (s',d',w',c',re',nyi',t') = statistic1 r (h+1) b in (s+s',max d' (d-1) +1,w+w',c+c',re+re',nyi+nyi',t+t')) (1,1,0,0,0,0,0) rs in
	let _ = update_statcount h s1 w1 b in	
	(s1,d1,w1,c1,re1,nyi1,t1)
	end
 | Decomposition(_,_,rs)-> begin ( try ignore (List.hd rs) with Failure(_) -> failwith"Decomposition refutation list is empty"  );
	let (s1,d1,w1,c1,re1,nyi1,t1) = List.fold_left (fun (s,d,w,c,re,nyi,t) r -> let (s',d',w',c',re',nyi',t') = statistic1 r (h+1) b in (s+s',max d' (d-1) +1,w+w',c+c',re+re',nyi+nyi',t+t')) (1,1,0,0,0,0,0) rs in
	let _ = update_statcount h s1 w1 b in	
	(s1,d1,w1,c1,re1,nyi1,t1)
	end
 | Confront(_,_,_,_,_,_,r1,r2) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | Trans(_,_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) (w1) b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)
 | NegEqualProp(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | EqualProp(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2) 
 | NegAequivalenz(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | Aequivalenz(_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | NegEqualFunc(_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) w1 b in	
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)   
 | EqualFunc(_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) w1 b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)   
 | ChoiceR(_,_,_,_,r1,r2) -> 
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2,re1+re2,nyi1+nyi2,t1+t2)
 | Cut(l,r1,r2) -> if debug_litcount then Printf.printf "cut on %d\n" (get_literal l);
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let (s2,d2,w2,c2,re2,nyi2,t2) = statistic1 r2 (h+1) b in
	let _ = update_statcount h (s1+s2+1) (w1+w2) b in
	(1+s1+s2,max d1 d2 +1,w1+w2,c1+c2+1,re1+re2,nyi1+nyi2,t1+t2)
 | DoubleNegation(_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) w1 b in
	(1+s1,d1 +1,w1,c1,re1,nyi1,t1)   
 | Rewrite(_,_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) w1 b in
	(1+s1,d1 +1,w1,c1,re1+1,nyi1,t1)    
 | Delta(_,_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) w1 b in
	(1+s1,d1 +1,w1,c1,re1+1,nyi1,t1)
 | NYI(_,_,r1) ->
	let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r1 (h+1) b in
	let _ = update_statcount h (s1+1) w1 b in
	(1+s1,d1 +1,w1,c1,re1,nyi1+1,t1) 
 | Timeout -> 
	let (zs,zw,n) =if b then (1,1,1) else try Hashtbl.find !statcount h with Not_found -> (1,1,1) in
	(zs/n,1,zw/n,0,0,0,1)   
 | _ -> failwith "unknown refutation case in statistic"
in 
let (s1,d1,w1,c1,re1,nyi1,t1) = statistic1 r 0 true in
let (s2,_,w2,_,_,_,_) = statistic1 r 0 false in
(s1,s2,d1,w1,w2,c1,re1,nyi1,t1)

(** MAIN **)    


let print_proofterm c r =
  
  if (!verbosity > vold) then Printf.printf  "starting print_proofterm.\n";
  if (!verbosity > vold) then print_refut r;

	(*** Search ***)

  (* get assumptions *)
  let initbranch = (List.rev !initial_branch) in
  let b =List.map get_literal initbranch in
  (* call search *)
  let (refutation,con,searchTime,isTimeout) = search_refutation b r in
  (* check wether the dependency of the returned refutation is a subset of the assumptions - fail indicates some bug*)
  if assert_condition && not (Dependency.is_empty (Dependency.diffunion [] [con] [b]) ) then 
    (if (!verbosity > 0) then Printf.printf  "Error with refutation: Still open conditions  \n") ;	 
  let (size,_,depth,width,_,cut,rewrite,notyetImpl,timeouts) = statistic refutation in
  (* if timeout was reached the refutation is unfinished - at least outputs some information about the completed part*)
	if isTimeout  || timeouts > 0 then  
  	  (if (!verbosity > 0) then Printf.printf  "Timeout!  time %d ms size %d depth %d width %d cuts %d rewrite %d NYI %d timeouts %d \n" searchTime size depth width cut rewrite notyetImpl timeouts)
  	else 	
  begin
    if result_print_search then Printf.printf  "%s\n" (ref_str refutation);
    if (!verbosity > 3) then Printf.printf "Search time %d ms size %d depth %d width %d \n" searchTime size depth width ;

	(*** Translation  ***)

(* get prenormalized assumptions *)
  let initbranch_prenorm =List.map 
	(fun pn ->let p= onlynegnorm pn in  if debug_translation then Printf.printf  "%s = %s \n" (trm_str pn) (trm_str p) ;p ) 
	(List.rev !initial_branch_prenorm) in

  let beforeTrans= Sys.time() in
(* Branch is an association list from normalized terms to their prenormalized counterparts *)
  let branch = (List.combine initbranch  initbranch_prenorm) in
(* The flag pftrm_eager_rewrite decides wether rewrites are eagerly or lazyly handled *) 
  let prenorm_refutation =if pftrm_eager_rewrite then eager_translate branch refutation else lazy_translate branch refutation in
  let transTime= int_of_float ((Sys.time() -. beforeTrans) *. 1000.0) in
  begin
  let (size,_,depth,width,_,cut,rewrite,notyetImpl,_) = statistic prenorm_refutation  in
  if result_print_translation then Printf.printf  "%s\n" (ref_str prenorm_refutation);
  if (!verbosity > 3) then Printf.printf  "Translation NYI %d time %d ms size %d depth %d width %d cuts %d rewrite %d  \n" notyetImpl transTime size depth width cut rewrite  ;

	(*** Output Coq ***)

  let beforeCoq= Sys.time() in
  if (result_coq && size < 800 ) then ref_coq c prenorm_refutation else raise (GenericError ("Coq Proof Too Big: " ^ (string_of_int size) ^ " steps"));
  let coqTime= int_of_float ((Sys.time() -. beforeCoq) *. 1000.0) in
  if (result_coq ) then if (!verbosity > 3) then Printf.printf  "Coq output done: time %d  \n" coqTime ;
	(*** Output Latex Search ***)

  if (result_latex && width < 50 && depth < 30) then
  Printf.fprintf c "(*** \n %%beginLatex \n size %d depth %d width %d \n \n \\begin{tabular}{c} \n %s \\end{tabular} \n\n %%endLatex \n ***)\n" 
  size depth width (ref_to_lat initbranch refutation)
  else if (result_latex) then Printf.fprintf c "(*** \n %%beginLatex \n size %d depth %d width %d \n \n  %%endLatex \n ***)\n" size depth width;
 
	(*** Output Latex Translation ***)

  if (result_latex &&  width < 50 && depth < 30) then
  Printf.fprintf c "(*** \n %%beginLatex \n size %d depth %d width %d cuts %d rewrite  %d NYI %d \n \n \\begin{tabular}{c} \n %s \\end{tabular} \n\n %%endLatex \n***)" 
  size depth width cut rewrite notyetImpl  (ref_to_lat initbranch_prenorm prenorm_refutation)
  else if (result_latex) then Printf.fprintf c "(*** \n %%beginLatex \n \n \n Translation successful, probably \n \n size %d depth %d width %d cuts %d rewrite %d  NYI %d \n %%endLatex \n***)" size depth width cut rewrite notyetImpl 
  ; flush stdout
  end
  end		
  
