(* File: match.ml *)
(* Author: Chad E Brown *)
(* Created: December 2010 *)

open Syntax

(*** For Terms with Meta Vars and Meta Type Vars ***)
exception NotGround
exception PatternMatchFailed

type metatrm =
   | MGround of trm (*** Assume this is DB, Name, or a logical constant ***)
   | MVar of metatrm option ref * metatrm list
   | MLam of stp * metatrm
   | MAp of metatrm * metatrm

type ctx = stp list
type dpair = ctx * metatrm * trm * stp

let rec id_subst i gamma =
  match gamma with
  | (a::gammar) -> ((MGround(DB(i,a)))::(id_subst (i + 1) gammar))
  | [] -> []

let evar_names : (metatrm option ref * string) list ref = ref []
let evarcount = ref 0

let evar_copy_count : (metatrm option ref * int ref) list ref = ref []

let evar_copy_of : (metatrm option ref * metatrm option ref) list ref = ref []

let string_of_evar x =
  try
    let (_,xs) = List.find (fun (x1,_) -> (x == x1)) (!evar_names) in
    xs
  with
  | Not_found -> "??"

let rec new_evar v gamma a =
  match a with
  | Ar(a1,a2) ->
      let (x,xs) = new_evar v (a1::gamma) a2 in
      (x,MLam(a1,xs))
  | _ ->
      let x = ref None in
      let s = id_subst 0 gamma in
      if (v > 20) then
	begin
	  evar_names := (x,("?" ^ (string_of_int (!evarcount))))::(!evar_names);
	  evar_copy_count := (x,ref 0)::(!evar_copy_count);
	  evar_copy_of := (x,x)::(!evar_copy_of);
	  incr evarcount
	end;
      (x,MVar(x,s))

let rec copy_evar v x =
  let y = ref None in
  if (v > 20) then
    begin
      let (_,z) = List.find (fun (x1,_) -> (x == x1)) (!evar_copy_of) in
      let (_,zi) = List.find (fun (z1,_) -> (z == z1)) (!evar_copy_count) in
      evar_copy_of := (y,z)::(!evar_copy_of);
      evar_names := (y,((string_of_evar z) ^ "_" ^ (string_of_int (!zi))))::(!evar_names);
      incr zi
    end;
  y

let lpar p = if p then "(" else ""

let rpar p = if p then ")" else ""

let rec metatrm_str_rec m p =
  match m with
  | MGround(m0) -> "<" ^ (trm_str m0) ^ ">"
  | MVar(x,s) -> (lpar p) ^ (string_of_evar x) ^ "[" ^ (String.concat "," (List.map (fun m -> metatrm_str_rec m false) s)) ^ "]" ^ (rpar p)
  | MLam(a,m1) -> (lpar p) ^ "\\_:" ^ (stp_str a) ^ "." ^ (metatrm_str_rec m1 false) ^ (rpar p)
  | MAp(m1,m2) ->
      begin
	match m1 with
	| MLam _ -> (lpar p) ^ (metatrm_str_rec m1 true) ^ " " ^ (metatrm_str_rec m2 true) ^ (rpar p)
	| _ -> (lpar p) ^ (metatrm_str_rec m1 false) ^ " " ^ (metatrm_str_rec m2 true) ^ (rpar p)
      end

let metatrm_str m = metatrm_str_rec m false
  
let rec to_meta m =
  match m with
  | Ap(m1,m2) -> MAp(to_meta m1,to_meta m2)
  | Lam(a,m1) -> MLam(a,to_meta m1)
  | _ -> MGround(m)

let rec meta_to_ground_rec m tau =
   match m with
   | MGround(m1) -> simulsubst m1 tau
   | MVar(x,sigma) ->
       begin
	 match (!x) with
	 | None -> raise NotGround
	 | Some n -> meta_to_ground_rec n (List.map (fun p -> meta_to_ground_rec p tau) sigma)
       end
   | MLam(a,m1) -> Lam(a,meta_to_ground_rec m1 ((DB(0,a))::(List.map (fun n -> shift n 0 1) tau)))
   | MAp(m1,m2) -> Ap(meta_to_ground_rec m1 tau,meta_to_ground_rec m2 tau)

let meta_to_ground d m =
   norm d (meta_to_ground_rec m [])

let rec metashift m i j =
  match m with
    MGround(DB(k,_)) when k < i -> m
  | MGround(DB(k,a)) -> MGround(DB(k + j,a))
  | MLam(a1,m2) -> MLam(a1,metashift m2 (i + 1) j)
  | MAp(m1,m2) -> MAp(metashift m1 i j,metashift m2 i j)
  | _ -> m

let rec metasubst m i n =
  match m with
    MGround(DB(k,_)) when k < i -> m
  | MGround(DB(k,_)) when k = i -> n
  | MGround(DB(k,a)) -> MGround(DB(k - 1,a))
  | MGround(_) -> m
  | MLam(a1,m2) -> MLam(a1,metasubst m2 (i + 1) (metashift n 0 1))
  | MAp(m1,m2) -> MAp(metasubst m1 i n,metasubst m2 i n)
  | MVar(x,sigma) -> MVar(x,List.map (fun m0 -> metasubst m0 i n) sigma)

let gen_mlam_body a m =
  match m with
  | MLam(_,m1) -> m1
  | MVar(x,sigma) -> raise (Failure ("Metavar of function type? " ^ (string_of_evar x)))
  | _ -> MAp(metashift m 0 1,MGround(DB(0,a)))

let rec metatermNotFreeIn m i =
  match m with
  | MGround(DB(j,_)) when i = j -> false
  | MVar(x,ml) -> metatermNotFreeInL ml i
  | MAp(m1,m2) -> (metatermNotFreeIn m1 i) && (metatermNotFreeIn m2 i)
  | MLam(a,m1) -> metatermNotFreeIn m1 (i + 1)
  | _ -> true
and metatermNotFreeInL ml i =
  match ml with
  | (m::mr) -> (metatermNotFreeIn m i) && (metatermNotFreeInL mr i)
  | [] -> true

let rec meta_simulsubst_meta m tau =
   match m with
   | MGround(DB(k,_)) -> List.nth tau k
   | MGround(m1) -> m
   | MVar(x,sigma) ->
       begin
	 match (!x) with
	 | None ->
	     MVar(x,List.map (fun p -> meta_simulsubst_meta p tau) sigma)
	 | Some n ->
	     meta_simulsubst_meta n (List.map (fun p -> meta_simulsubst_meta p tau) sigma)
       end
   | MLam(a,m1) -> MLam(a,meta_simulsubst_meta m1 ((MGround(DB(0,a)))::(List.map (fun n -> metashift n 0 1) tau)))
   | MAp(m1,m2) -> MAp(meta_simulsubst_meta m1 tau,meta_simulsubst_meta m2 tau)

let rec metanorm1 m =
  match m with
  | MGround(m1) ->
      let (n1,b) = onlybetanorm1 m1 in
      (MGround(n1),b)
  | MAp(MLam(a,m1),m2) -> (* beta *)
      let (n1,_) = metanorm1 m1 in
      let (n2,_) = metanorm1 m2 in
      (metasubst n1 0 n2,true)
(***  | MLam(_,MAp(m1,MGround(DB(0,_)))) when (metatermNotFreeIn m1 0) -> (* eta *) (*** Removed eta here.  The way dpairs at functional type are handled, they will be more or less eta expanded on the fly anyway.  Hence there is no point in wasting time doing this. - Chad, Jan 10, 2011 ***)
      (metashift m1 0 (- 1),true) ***)
	(*** dneg ***)
  | MAp(MAp(MGround(Imp),MAp(MAp(MGround(Imp),m1),MGround(False))),MGround(False)) -> (* double negation reduce *)
      let (n1,_) = metanorm1 m1 in
      (n1,true)
  | MAp(m1,m2) ->
      let (n1,b1) = metanorm1 m1 in
      let (n2,b2) = metanorm1 m2 in
      if (b1 || b2) then
	(MAp(n1,n2),true)
      else
	(m,false)
  | MLam(a1,m1) ->
      let (n1,b1) = metanorm1 m1 in
      if b1 then
	(MLam(a1,n1),true)
      else
	(m,false)
  | MVar(x,sigma) ->
      begin
	match (!x) with
	| None ->
	    let (sigmar,sigmab) = metasubstnorm1 sigma in
	    if sigmab then
	      (MVar(x,sigmar),true)
	    else
	      (m,false)
	| Some m1 ->
	    (meta_simulsubst_meta m1 sigma,true)
      end
and metasubstnorm1 sigma =
  match sigma with
  | (m1::sigmar) ->
      let (n1,b1) = metanorm1 m1 in
      let (sigma2,b2) = metasubstnorm1 sigmar in
      if (b1 || b2) then
	(n1::sigma2,true)
      else
	(sigma,false)
  | [] -> ([],false)

(* beta-dneg *)
let rec metanorm m =
  let (m1,reduced) = metanorm1 m in
  if reduced then (metanorm m1) else m

let rec meta_copy m evarassoc =
  begin
    match m with
    | MGround(m1) -> m
    | MVar(x,sigma) ->
	let sigma1 = List.map (fun m -> meta_copy m evarassoc) sigma in
	begin
	  match (!x) with
	  | None ->
	      begin
		try
		  let (_,y) = List.find (fun (x1,_) -> (x == x1)) evarassoc in
		  MVar(y,sigma1)
		with
		| Not_found ->
		    MVar(x,sigma1)
	      end
	  | Some m1 ->
	      let m1c = meta_copy m1 evarassoc in
	      metanorm (meta_simulsubst_meta m1c sigma1)
	end
    | MLam(a1,m1) ->
	let m1c = meta_copy m1 evarassoc in
	MLam(a1,m1c)
    | MAp(m1,m2) ->
	let m1c = meta_copy m1 evarassoc in
	let m2c = meta_copy m2 evarassoc in
	MAp(m1c,m2c)
  end

let rec distinct_bvar_list_p_rec ml dl =
  match ml with
  | [] -> true
  | ((MGround(DB(i,_)))::mr) when (not (List.mem i dl)) -> distinct_bvar_list_p_rec mr (i::dl)
  | _ -> false

let distinct_bvar_list_p ml =
  distinct_bvar_list_p_rec ml []

let rec pattern_p m =
  match m with
  | MGround(m1) -> true
  | MVar(x,ml) ->
      begin
	match (!x) with
	| None -> distinct_bvar_list_p ml
	| Some n -> raise NotGround (*** Actually, this is not yet written - just fail for now ***)
      end
  | MLam(a,m1) -> pattern_p m1
  | MAp(m1,m2) -> pattern_p m1 && pattern_p m2

let rec meta_head_spine_rec m args =
  match m with
  | MAp(m1,m2) -> meta_head_spine_rec m1 (m2::args)
  | MGround(m1) -> (m1,tpof m1,args)
  | _ -> raise (Failure "Unexpected case in meta_head_spine_rec")

let meta_head_spine m =
  meta_head_spine_rec m []

let rec occurs_check_p x m =
  match m with
  | MGround(_) -> false
  | MVar(y,_) when (x = y) -> true
  | MVar(y,ml) -> occurs_check_list_p x ml
  | MLam(a,m1) -> occurs_check_p x m1
  | MAp(m1,m2) -> occurs_check_p x m1 || occurs_check_p x m2
and occurs_check_list_p x ml =
  match ml with
  | (m::mr) -> occurs_check_p x m || occurs_check_list_p x mr
  | [] -> false

let rec pattern_invert_db ml n j i k =
  match ml with
  | ((MGround(DB(i2,a)))::mr) when i = i2 -> MGround(DB(j+k,a))
  | (_::mr) -> pattern_invert_db mr n j i (k + 1)
  | [] -> raise PatternMatchFailed

let rec pattern_invert x ml n j =
  match n with
  | DB(i,_) ->
      if (i < j) then
	MGround(n)
      else
	pattern_invert_db ml n j (i - j) 0
  | Lam(a,n1) -> MLam(a,pattern_invert x ml n1 (j + 1))
  | Ap(n1,n2) -> MAp(pattern_invert x ml n1 j,pattern_invert x ml n2 j)
  | _ -> MGround(n)
and pattern_list_invert x ml nl j =
  match nl with
  | (n::nr) -> (pattern_invert x ml n j::pattern_list_invert x ml nr j)
  | [] -> []

let rec pattern_match_rec dl cl =
  match dl with
  | [] -> cl
  | (d::dr) -> pattern_match_rec_1 d dr cl
and pattern_match_rec_1 d dl cl =
  match d with
  | (gamma,m1,n1,Ar(a,b)) ->
      pattern_match_rec_1 (a::gamma,gen_mlam_body a m1,gen_lam_body a n1,b) dl cl
  | (gamma,MVar(x,ml),n,b) ->
      begin
	match (!x) with
	| None ->
	    begin
	      if (distinct_bvar_list_p ml) then
		let ni = pattern_invert x ml n 0 in
		begin
		  x := Some ni;
		  let clr = ref [] in
		  let dlr = ref dl in
		  List.iter
		    (fun ((gammac,mc,nc,b) as c) ->
		      let (mc1,mb) = metanorm1 mc in
		      if mb then
			let mcn = metanorm mc1 in
			dlr := ((gammac,mcn,nc,b)::!dlr)
		      else
			clr := (c::(!clr))
		    )
		    cl;
		  pattern_match_rec (!dlr) (!clr)
		end
	      else
		begin
		  pattern_match_rec dl (d::cl)
		end
	    end
	| Some(mx) -> pattern_match_rec_1 (gamma,metanorm (meta_simulsubst_meta mx ml),n,b) dl cl
      end
  | (gamma,m,n,b) ->
      let (mh,mhtp,ml) = meta_head_spine m in
      let (nh,nl) = head_spine n in
      if (mh = nh) then
	pattern_match_rec_spine gamma mhtp ml nl dl cl
      else
	raise PatternMatchFailed
and pattern_match_rec_spine gamma tp ml nl dl cl =
  match (tp,ml,nl) with
  | (_,[],[]) -> pattern_match_rec dl cl
  | (Ar(a,b),(m::ml),(n::nl)) -> pattern_match_rec_spine gamma b ml nl ((gamma,m,n,a)::dl) cl
  | _ -> raise PatternMatchFailed
      
let pattern_match dl = pattern_match_rec dl []

let mem_eq a l =
  try
    ignore(List.find (fun x -> x == a) l);
    true
  with Not_found -> false

let rec update_strict xl m =
  match m with
  | MGround(_) -> xl
  | MVar(x,sigma) ->
      if (mem_eq x xl) then
	xl
      else if (distinct_bvar_list_p sigma) then
	(x::xl)
      else
	xl
  | MAp(m1,m2) -> update_strict (update_strict xl m1) m2
  | MLam(a1,m1) -> update_strict xl m1

