(* File: satallax.ml *)
(* Author: Chad E Brown *)
(* Created: September 2010 *)

open Flags
open Syntax
open State
open Search
open Proofterm

exception Timeout
exception InputHelpError of string
exception InputError of string
exception UnclassifiedError

(*** Satallax 2.0 default schedule. 36 modes. 610s total (10m10s). ***)
let strategy_schedule = [
("mode14",27);
("mode111",24);
("mode14a",25);
("mode174",9);
("mode188",9);
("mode176",14);
("mode219",15);
("mode80",16);
("mode173",6);
("mode187",6);
("mode168",7);
("mode163",6);
("mode8a",9);
("mode19c",6);
("mode190",6);
("mode171",6);
("mode175",6);
("mode164",6);
("mode108",8);
("mode186",6);
("mode4a",6);
("mode104",6);
("mode179",6);
("mode218",7);
("mode128",6);
("mode103",6);
("mode185",6);
("mode41c",6);
("mode3a",6);
("mode189",6);
("mode86",36);
("mode92",40);
("mode177",39);
("mode165",97);
("mode94",60);
("mode191",60)
];;

let coqfile = ref "";; 

let print_help () =
  (print_string "Usage: satallax [-[Vv]] [-verbose <int>] [-M <modedir>] [-m <modefile>] [-t <timeout in seconds>] <problemfile>";
   print_newline();
   print_string "-M <dir> : Set the directory in which the mode file is stored.";
   print_newline();
   print_string "       The default mode directory is the 'modes' subdirectory of the Satallax directory.";
   print_newline();
   print_string "-m : Mode";
   print_newline();
   print_string "-V : Print version number and quit";
   print_newline();
   print_string "-v : Verbose";
   print_newline();
   print_string "-verbose <int> : Verbosity of given level";
   print_newline();
   print_string "-c [<file>|-] : Create a Coq version of problem";
   print_newline());;

(*** Load the mode (error if cannot find it) ***)
let comment_line_p l =
  if (String.length l = 0) then
    true
  else
    (l.[0] = ';');;

let rec read_mode_file mf =
  let l = input_line mf in
  if (comment_line_p l) then
    read_mode_file mf
  else
    read_mode_file_value mf l
and read_mode_file_value mf flagname =
  let l = input_line mf in
  if (comment_line_p l) then
    read_mode_file_value mf flagname
  else if (l = "true") then
    begin
      set_bool_flag flagname true;
      read_mode_file mf
    end
  else if (l = "false") then
    begin
      set_bool_flag flagname false;
      read_mode_file mf
    end
  else
    begin
      set_int_flag flagname (int_of_string l);
      read_mode_file mf
    end
  
let modedir = ref (Config.satallaxdir ^ "/modes")
  
let load_mode m =
  let modefile = (!modedir ^ "/" ^ m) in
  if (Sys.file_exists modefile) then
    begin
      init_flags();
      try
	read_mode_file (open_in modefile)
      with
      | End_of_file -> ()
    end
  else
    raise (InputError ("Could not find mode " ^ modefile));;

let read_thf_file (f:string) (include_fun : string -> unit) =
  let ch = Lexing.from_channel (open_in f) in
  let old_include_fun = !st_include_fun in
  st_include_fun := include_fun;
(***  List.iter Tptp_config.process_thf (Tptp_parser.tptp_file Tptp_lexer.token ch); ***)
  ignore (Tptp_parser.tptp_file Tptp_lexer.token ch);
  if (!verbosity > 4) then Printf.printf "Finished reading thf file %s\n" f;
  st_include_fun := old_include_fun

let rec find_read_thf_file_r odir dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file odir)
  else if (String.length dir > 1) then
    find_read_thf_file_r odir (Filename.dirname dir) f
  else
    raise (FileNotFound f)
and find_read_thf_file dir f =
  let ff = (dir ^ "/" ^ f) in
  if (Sys.file_exists ff) then
    read_thf_file ff (find_read_thf_file dir)
  else
    begin
      try
	let tptpdir = Sys.getenv "TPTP" in
	let tff = (tptpdir ^ "/" ^ f) in
	if (Sys.file_exists tff) then
	  read_thf_file tff (find_read_thf_file dir)
	else
	  find_read_thf_file_r dir dir f
      with
      | Not_found -> find_read_thf_file_r dir dir f
    end

(*** If the slave got a final result, then use it. ***)
let handle_slave_return pstatus =
  begin
    if (!verbosity > 4) then
      Printf.printf "slave returned with status %d\n" pstatus
    else ();
    if (pstatus >= 10) then exit pstatus else ()
  end;;

(*** set_timer for timeout signal, see http://alan.petitepomme.net/cwn/2006.01.03.html#2 ***)
let set_timer s =
  ignore (Unix.setitimer Unix.ITIMER_REAL { Unix.it_interval = 0.0; Unix.it_value = s });;

try
  let argc = Array.length Sys.argv in
  let slaveargs = ref [Sys.argv.(0)] in
  begin
    if (argc = 1) then (Version.print_version(); print_newline(); print_help(); exit 1) else
    let problemfile = ref "" in
    let i = ref 1 in
    while (!i < argc) do
      (match Sys.argv.(!i) with
      | "-P" -> (incr i; slaveargs := ("-P"::!slaveargs); Config.picomus := Sys.argv.(!i))
      | "-m" -> (incr i; mode := (Some (Sys.argv.(!i))))
      | "-M" -> (incr i; modedir := (Sys.argv.(!i)))
      | "-t" -> (incr i; timeout := (Some (int_of_string Sys.argv.(!i))))
      | "-slave" -> (incr i; slave := true)
      | "-c" ->
	  begin
	    incr i;
	    let cf = Sys.argv.(!i) in
	    slaveargs := (cf::"-c"::!slaveargs);
	    coq := true;
	    if (cf = "-") then
	      coqchannel := Some stdout
	    else
	      begin
		coqfile := cf;
		coqchannel := Some (open_out cf)
	      end
	  end
      | "-verb" ->
	  begin
	    incr i;
	    let cf = Sys.argv.(!i) in
	    slaveargs := (cf::"-verb"::!slaveargs);
	    verbosity := (int_of_string cf)
	  end
      | "" -> raise (InputHelpError "Problem processing command line arguments")
      | option ->
	  (if (option.[0] = '-') then
            for j = 1 to String.length option - 1 do
              match option.[j] with
              | 'v' -> slaveargs := ("-v"::!slaveargs); verbosity := 5
              | 'V' -> Version.print_version() ; print_newline(); exit 0
(***              | 'p' -> slaveargs := ("-p"::!slaveargs); mkproofterm := true - skip for now ***)
              | _ -> raise (InputHelpError (String.concat " " ["Unknown command line argument";String.make 1 (option.[j])]))
            done
	  else
	    begin
	      problemfile := option
	    end);
	  incr i)
    done;
    (match (!mode) with
    | Some m ->
	begin
	  try
	    load_mode m;
	    if (!verbosity > 8) then print_flags () else ();
	    let s = (get_timeout_default 0) in
	    if (s > 0) then
	      begin
		ignore (Sys.signal Sys.sigalrm (Sys.Signal_handle (fun signo -> raise Timeout)));
		set_timer (float_of_int s);
		if (!coq) then coq_init();
		read_thf_file (!problemfile) (find_read_thf_file (Filename.dirname (!problemfile)));
		if (not (!slave)) then
		  begin
		    match (!coqchannel) with
		    | Some c -> print_coqsig c
		    | None -> ()
		  end;
		search ()
	      end
	    else
	      begin
		if (!coq) then coq_init();
		read_thf_file (!problemfile) (find_read_thf_file (Filename.dirname (!problemfile)));
		if (not (!slave)) then
		  begin
		    match (!coqchannel) with
		    | Some c -> print_coqsig c
		    | None -> ()
		  end;
		search ()
	      end
	  with
	  | Unsatisfiable(r) ->
	      begin
		let c = match (!coqchannel) with Some c -> c | None -> stdout in
		begin
		  if (!mkproofterm) then begin match r with Some(r) -> print_proofterm c r | None -> () end;
		  if (c != stdout) then
		    close_out c;
		end;
		match !conjecture with
		| Some _ -> (if (!verbosity > 0) then Printf.printf "%% SZS status Theorem\n"; exit 20)
		| None -> (if (!verbosity > 0) then Printf.printf "%% SZS status Unsatisfiable\n"; exit 25)
	      end
	  | Satisfiable ->
	      begin
		match !conjecture with
		| Some _ -> (if (!verbosity > 0) then Printf.printf "%% SZS status CounterSatisfiable\n"; exit 10)
		| None -> (if (!verbosity > 0) then Printf.printf "%% SZS status Satisfiable\n"; exit 15)
	      end
	end
    | None -> (*** Use the strategy schedule ***)
	let scheduletime = List.fold_left (fun s1 (x,s2) -> (s1 + s2)) 0 strategy_schedule in (*** total time of schedule ***)
	let timeoutfac = max 1. ((float_of_int (get_timeout_default scheduletime)) /. (float_of_int scheduletime)) in (*** timeout factor ***)
	let schedulenum = ref (List.length strategy_schedule) in
	List.iter
	  (fun (m,s) ->
	    decr schedulenum;
	    if ((!schedulenum) > 1) then
	      begin
		match (!timeout) with
		| Some s2 ->
		    let s1 = int_of_float ((float_of_int s) *. timeoutfac) in
		    begin
		      if (s2 > s1) then
			begin
			  begin
			    match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_int s1)::"-t"::m::"-m"::"-slave"::(!slaveargs))))) with
			    | (Unix.WEXITED pstatus) ->
				handle_slave_return pstatus
			    | _ ->
				if (!verbosity > 4) then
				  Printf.printf "slave returned with unknown status\n"
				else ()
			  end;
			  timeout := (Some (s2 - s1))
			end
		      else
			begin
			  begin
			    (*** Final call - don't tell it it's a slave. ***)
			    match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_int s2)::"-t"::m::"-m"::(!slaveargs))))) with
			    | (Unix.WEXITED pstatus) ->
				handle_slave_return pstatus ; exit pstatus
			    | _ ->
				if (!verbosity > 4) then
				  (Printf.printf "slave returned with unknown status\n" ; exit 3)
				else ()
			  end;
			  raise Timeout
			end
		    end
		| None ->
		    begin
		      match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_int s)::"-t"::m::"-m"::"-slave"::(!slaveargs))))) with
		      | (Unix.WEXITED pstatus) ->
			  handle_slave_return pstatus
		      | _ ->
			  if (!verbosity > 4) then
			    Printf.printf "slave returned with unknown status\n"
			  else ()
		    end
	      end
	    else
	      begin
		(*** Final call - don't tell it it's a slave. ***)
		match (Unix.system (String.concat " " (List.rev ((!problemfile)::(string_of_int s)::"-t"::m::"-m"::(!slaveargs))))) with
		| (Unix.WEXITED pstatus) ->
		    handle_slave_return pstatus ; exit pstatus
		| _ ->
		    if (!verbosity > 4) then
		      (Printf.printf "slave returned with unknown status\n" ; exit 3)
		    else ()
	      end
	  )
	  strategy_schedule;
	raise Timeout
    )
  end
with
| InputHelpError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "Input Error: %s" x; print_newline (); print_help ()); exit 1
| InputError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "Input Error: %s" x; print_newline ()); exit 1
| FileNotFound(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "File Not Found: %s" x; print_newline ()); exit 1
| NotFlag(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s is not a flag" x; print_newline ()); exit 2
| NotBoolFlag(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s is an integer flag, not a boolean flag" x; print_newline ()); exit 2
| NotIntFlag(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s is a boolean flag, not an integer flag" x; print_newline ()); exit 2
| Parsing.Parse_error -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); print_string "Syntax Error"; print_newline ()); exit 2
| ParsingError(l1,i1,l2,i2,x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "Parsing Error: %s from line %d char %d to line %d char %d\n" x l1 i1 l2 i2; print_newline ()); exit 2
| ExpectedTypeError(m,a,b) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s\nhas type\n%s\nexpected type\n%s\n" (pretrm_str m) (stp_str b) (stp_str a); print_newline ()); exit 2
| GenericError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s\n" x; print_newline ()); exit 2
| GenericSyntaxError(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s\n" x; print_newline ()); exit 2
| Redeclaration(x) -> if (not (!slave)) then (print_string "% SZS status Error :"; print_newline (); Printf.printf "%s cannot be redeclared " x; print_newline ()); exit 2
| UnclassifiedError -> if (not (!slave)) then (print_string "% SZS status Error"; print_newline ()); exit 3
| Timeout -> if (not (!slave)) then (if (!verbosity > 0) then (print_string "% SZS status Timeout"; print_newline ())); exit 5
| e -> if (not (!slave)) then (if (!verbosity > 0) then (print_string "% SZS status Error"; print_newline (); Printf.printf "Exception: %s\n" (Printexc.to_string e))); exit 3
