; Chad E Brown
; Jan 2010
; Controller code for Satallax.

(defvar *slave* nil)
(defvar *verbose* nil)
(defvar *print-final-proof-info* nil)
(defvar *fresh-images* t)
(defvar *explicit-minisat-binary* nil)

(defvar *strategy* '(
("mode30" 54)
("mode21" 49)
("mode16" 41)
("mode19" 38)
("mode4" 25)
("mode17" 13)
("mode39" 37)
("mode43" 18)
("mode26" 5)
("mode10" 6)
("mode9" 16)
("mode5" 14)
("mode14" 2)
("mode20" 2)
))

(defun satallax-top ()
  (sb-sys:enable-interrupt sb-unix:sigint #'(lambda (&rest args) (sb-ext:quit :unix-status 1)))
  (handler-case
   (satallax-sub-top)
   (error (x)
	  (unless *slave*
	    (format t "% SZS status Error :~%~d~%" x)
	    )
	  (sb-ext:quit :unix-status 1))
;   (serious-condition (x) (sb-ext:quit :unix-status 1))
;   (condition (x) (sb-ext:quit :unix-status 1))
   ))

(defun satallax-sub-top ()
  (sb-ext:quit
   :unix-status
   (catch 'failure
     (let ((args (cdr *posix-argv*))
	   (mode nil)
	   (modefile nil)
	   (timeout nil)
	   )
       (loop while (and args (> (length (car args)) 0) (eq (aref (car args) 0) #\-)) do
	     (let ((op (pop args)))
	       (cond ((equal op "-m")
		      (setq mode (pop args))
		      (setq modefile (format nil "~dmodes/~d" *satallax-dir* mode))
		      )
		     ((equal op "-slave") (setq *slave* t)) ; slave process - should not report timeouts or errors, only successes
		     ((equal op "-M")
		      (setq *explicit-minisat-binary* t)
		      (setq *minisat-binary* (pop args))
		      (unless (probe-file *minisat-binary*)
			(format t "% SZS status Error :~%")
			(format t "*** Could not find MiniSat binary ***~%ABORT~%")
			(throw 'failure 1)))
		     ((equal op "-t")
		      (setq timeout (read-from-string (pop args)))
		      (unless (numberp timeout)
			(format t "% SZS status Error :~%")
			(format t "*** Timeout must be a number. ***~%ABORT~%")
			(throw 'failure 1)))
		     (t
		      (dotimes (i (length op))
			(when (> i 0)
			  (case (aref op i)
				(#\V (satallax-version) (sb-ext:quit :unix-status 0))
				(#\p (setq *print-final-proof-info* t))
				(#\v (setq *verbose* t))
				(#\S (setq *fresh-images* nil))
				(#\d (setq *debug* t) (load (format nil "~dsrc/debug.lisp" *satallax-dir*)))
				(t (format t "Ignoring unknown option -~d~%" (aref op i))))))))))
       (if (car args)
	   (let ((problemfile (car args)))
	     (unless (probe-file problemfile)
	       (format t "% SZS status Error :~%")
	       (format t "*** No file ~d~%ABORT~%" problemfile)
	       (throw 'failure 1))
	     (if modefile
		 (progn
		   (if (probe-file modefile)
		       (load modefile)
		     (progn
		       (setq modefile (format nil "~d.lisp" modefile))
		       (if (probe-file modefile)
			   (load modefile)
			 (progn
			   (format t "% SZS status Error :~%")
			   (format t "*** Did not find mode file: ~d~%ABORT~%" modefile)
			   (throw 'failure 1)))))
		   (when *debug* (print-flags))
		   (if timeout
		       (handler-case
			(with-timeout timeout
			  (logic-init)
			  (hol2sat-file problemfile))
			(timeout ()
				 (when *verbose* (format t "satallax timed out.~%"))
				 (unless *slave*
				   (format t "% SZS status Timeout~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
				   )
				 (throw 'failure 5))) ; timed out
		     (progn
		       (logic-init)
		       (hol2sat-file problemfile))))
	       (if timeout ; try each of a series of modes for a given amount of time
		   (let ((succ nil)
			 (timeoutfac (max 1 (/ timeout (apply #'+ (mapcar #'cadr *strategy*))))))
		     (unless *fresh-images* (logic-init))
		     (loop while (and *strategy* (> timeout 0) (not succ)) do
			   (let ((ltimeout
				  (if (cdr *strategy*)
				      (round (* (cadar *strategy*) timeoutfac))
				    timeout)))
			     (when *verbose* (format t "Trying mode ~d for ~ds~%" (caar *strategy*) ltimeout))
			     (if *fresh-images* ; start a fresh image every time
				 (let ((largs
					(list "-slave" ; slave process - do not report timeouts or errors
					      "-t" (format nil "~d" ltimeout)
					      "-m" (caar *strategy*)
					      problemfile)))
				   (when *explicit-minisat-binary*
				     (push *minisat-binary* largs)
				     (push "-M" largs))
				   (when *verbose* (push "-v" largs))
				   (when *print-final-proof-info* (push "-p" largs))
				   (when *debug* (push "-d" largs))
				   (let (p e)
				     (handler-case
				      (with-timeout (+ 2 ltimeout) ; if it refuses to timeout after 2 extra seconds, kill it
					(progn
					  (setq timeout (- timeout ltimeout))
					  (when *verbose* (format t "** Calling mode ~d for ~d seconds~%" (caar *strategy*) ltimeout))
					  (setq p (sb-ext:run-program (car *posix-argv*) largs :output t :wait nil))
					  (sb-ext:process-wait p t)
					  (setq e (sb-ext:process-exit-code p))
					  (when *verbose* (format t "** Finished mode ~d for ~d seconds~%" (caar *strategy*) ltimeout))
					  )
					)
				      (timeout ()
					       (sb-ext:process-kill p 9)
					       (when *verbose* (format t "** Killed mode ~d for ~d seconds~%" (caar *strategy*) ltimeout))
					       (setq timeout (- timeout 2))))
				     (when (member e '(0 1 2 3 10 15 20 25) :test #'equal)
				       (setq succ e))
				     (pop *strategy*)))
			       (progn
				 (handler-case
				  (with-timeout ltimeout
				    (setq modefile (format nil "~dmodes/~d" *satallax-dir* (caar *strategy*)))
				    (if (probe-file modefile)
					(progn
					  (flag-defaults)
					  (load modefile)
					  (when *debug* (print-flags))
					  )
				      (progn
					(format t "% SZS status Error :~%")
					(format t "*** Did not find mode file: ~d~%ABORT~%" modefile)
					(throw 'failure 1)))
				    (let ((e (hol2sat-file problemfile)))
				     (when (member e '(0 1 2 3 10 15 20 25) :test #'equal)
				       (setq succ e))))
				  (timeout ()
					   (when *verbose* (format t "~d timed out.~%" (caar *strategy*)))
					   ))
				 (pop *strategy*)
				 (setq timeout (- timeout ltimeout))))))
		     (or succ 5)) ; 5 means all modes timed out
		 (progn ; or use the default mode
		   (load (format nil "~dmodes/mode38.lisp" *satallax-dir*)) ; the default mode
		   (logic-init)
		   (hol2sat-file problemfile)
		   ))))
	 (progn
	   (format t "No file given.~%Usage: satallax [-[Vvp]] [-M <MiniSat>] [-m <modefile>] [-t <timeout in seconds>] <problemfile>~%")
	   (format t "-m : Mode~%")
	   (format t "-M : MiniSat binary file~%")
	   (format t "-V : Print version number and quit~%")	   
	   (format t "-v : Verbose~%")
	   (format t "-p : Print information about the final proof (clauses and atlas of atoms).~%")
;	   (format t "-S : Run modes sequentially without starting new lisp images.~%")
	   1))))))

(defun flag-defaults ()
  (setq *sym-eq* t)
  (setq *instantiate-with-func-diseqn-sides* t) ; if this is t, then instantiate quantifiers at a type (A > B) with terms that occur on disequations of type (A > B)
  (setq *imitate-defns* t) ; if this is t, then instantiate by imitating defns
  (setq *nbcon* nil) ; unwritten
  (setq *nbdec* nil) ; unwritten
  (setq *nbmat* nil) ; unwritten
  (setq *call-period* 100) 
  (setq *minisat-timeout* 1)
  (setq *minisat-timeout-init* 1) 
  (setq *minisat-timeout-incr* 0) 
  (setq *exists-delay* 1) 
  (setq *forall-delay* 1) 
  (setq *defaultelt-delay* 30) 
  (setq *defaulteltinst-delay* 30) 
  (setq *confr-diff-delay* 100) ; I'm not sure these confrontations are really needed for completeness, so delay them a long time
  (setq *confr-same1-delay* 5)
  (setq *confr-same2-delay* 0)
  (setq *enum-start* 2) 
  (setq *enum-arrow* 10) 
  (setq *enum-o* 5) 
  (setq *enum-sort* 2) 
  (setq *enum-neg* 5) 
  (setq *enum-or* 20) 
  (setq *enum-false* 20) 
  (setq *enum-choice* 0) 
  (setq *enum-eq* 5) 
  (setq *leibeq-to-primeq* nil)
  (setq *enable-local-defns* nil) ; next 2 flags only matters if this is t
  (setq *local-defns-delay-pre* 2)
  (setq *local-defns-delay-post* 2)
  (setq *enable-pattern-rules* nil)
  (setq *pattern-rules-delay* 1)
  (setq *pattern-rules-eqn-delay* 1)
  (setq *choice-as-default* nil) ; boolean, if t, use (choose x:a . false) as the default element.  otherwise a term or fresh name may be used.
  (setq *imitate-defn-delay* 0)
  (setq *imitate-delay* 0)
  (setq *project-delay* 0)
  (setq *new-head-enum-delay* 0)
  (setq *choice-empty-delay* 0)
  (setq *choice-in-delay* 0)
  (setq *post-or-l-delay* 0)
  (setq *post-or-r-delay* 0)
  (setq *post-nor-l-delay* 0)
  (setq *post-nor-r-delay* 0)
  (setq *post-eqo-l-delay* 0)
  (setq *post-eqo-r-delay* 0)
  (setq *post-eqo-nl-delay* 0)
  (setq *post-eqo-nr-delay* 0)
  (setq *post-neqo-l-delay* 0)
  (setq *post-neqo-r-delay* 0)
  (setq *post-neqo-nl-delay* 0)
  (setq *post-neqo-nr-delay* 0)
  (setq *post-dec-delay* 0)
  (setq *post-mating-delay* 0)
  (setq *post-feq-delay* 0)
  (setq *post-nfeq-delay* 0)
  (setq *post-confront1-delay* 0)
  (setq *post-confront2-delay* 0)
  (setq *post-confront3-delay* 0)
  (setq *post-confront4-delay* 0)
  )


(defun print-flags ()
  (format t "*sym-eq*: ~d~%" *sym-eq*)
  (format t "*instantiate-with-func-diseqn-sides*: ~d~%" *instantiate-with-func-diseqn-sides*)
  (format t "*imitate-defns*: ~d~%" *imitate-defns*)
  (format t "*nbcon*: ~d~%" *nbcon*)
  (format t "*nbdec*: ~d~%" *nbdec*)
  (format t "*nbmat*: ~d~%" *nbmat*)
  (format t "*call-period*: ~d~%" *call-period*)
  (format t "*minisat-timeout*: ~d~%" *minisat-timeout*)
  (format t "*minisat-timeout-init*: ~d~%" *minisat-timeout-init*)
  (format t "*minisat-timeout-incr*: ~d~%" *minisat-timeout-incr*)
  (format t "*exists-delay*: ~d~%" *exists-delay*)
  (format t "*forall-delay*: ~d~%" *forall-delay*)
  (format t "*defaultelt-delay*: ~d~%" *defaultelt-delay*)
  (format t "*defaulteltinst-delay*: ~d~%" *defaulteltinst-delay*)
  (format t "*confr-diff-delay*: ~d~%" *confr-diff-delay*)
  (format t "*confr-same1-delay*: ~d~%" *confr-same1-delay*)
  (format t "*confr-same2-delay*: ~d~%" *confr-same2-delay*)
  (format t "*enum-start*: ~d~%" *enum-start*)
  (format t "*enum-arrow*: ~d~%" *enum-arrow*)
  (format t "*enum-o*: ~d~%" *enum-o*)
  (format t "*enum-sort*: ~d~%" *enum-sort*)
  (format t "*enum-neg*: ~d~%" *enum-neg*)
  (format t "*enum-or*: ~d~%" *enum-or*)
  (format t "*enum-false*: ~d~%" *enum-false*)
  (format t "*enum-choice*: ~d~%" *enum-choice*)
  (format t "*enum-eq*: ~d~%" *enum-eq*)
  (format t "*leibeq-to-primeq*: ~d~%" *leibeq-to-primeq*)
  (format t "*enable-local-defns*: ~d~%" *enable-local-defns*)
  (format t "*local-defns-delay-pre*: ~d~%" *local-defns-delay-pre*)
  (format t "*local-defns-delay-post*: ~d~%" *local-defns-delay-post*)
  (format t "*split-global-disjunctions*: ~d~%" *split-global-disjunctions*)
  (format t "*enable-pattern-rules*: ~d~%" *enable-pattern-rules*)
  (format t "*enable-pattern-rules*: ~d~%" *enable-pattern-rules*)
  (format t "*pattern-rules-delay*: ~d~%" *pattern-rules-delay*)
  (format t "*pattern-rules-eqn-delay*: ~d~%" *pattern-rules-eqn-delay*)
  (format t "*enable-pattern-clauses*: ~d~%" *enable-pattern-clauses*)
  (format t "*pattern-clauses-delay*: ~d~%" *pattern-clauses-delay*)
  (format t "*choice-as-default*: ~d~%" *choice-as-default*)
  (format t "*imitate-defn-delay*: ~d~%" *imitate-defn-delay*)
  (format t "*imitate-delay*: ~d~%" *imitate-delay*)
  (format t "*project-delay*: ~d~%" *project-delay*)
  (format t "*new-head-enum-delay*: ~d~%" *new-head-enum-delay*)
  (format t "*choice-empty-delay*: ~d~%" *choice-empty-delay*)
  (format t "*choice-in-delay*: ~d~%" *choice-in-delay*)
  (format t "*post-or-l-delay*: ~d~%" *post-or-l-delay*)
  (format t "*post-or-r-delay*: ~d~%" *post-or-r-delay*)
  (format t "*post-nor-l-delay*: ~d~%" *post-nor-l-delay*)
  (format t "*post-nor-r-delay*: ~d~%" *post-nor-r-delay*)
  (format t "*post-eqo-l-delay*: ~d~%" *post-eqo-l-delay*)
  (format t "*post-eqo-r-delay*: ~d~%" *post-eqo-r-delay*)
  (format t "*post-eqo-nl-delay*: ~d~%" *post-eqo-nl-delay*)
  (format t "*post-eqo-nr-delay*: ~d~%" *post-eqo-nr-delay*)
  (format t "*post-neqo-l-delay*: ~d~%" *post-neqo-l-delay*)
  (format t "*post-neqo-r-delay*: ~d~%" *post-neqo-r-delay*)
  (format t "*post-neqo-nl-delay*: ~d~%" *post-neqo-nl-delay*)
  (format t "*post-neqo-nr-delay*: ~d~%" *post-neqo-nr-delay*)
  (format t "*post-dec-delay*: ~d~%" *post-dec-delay*)
  (format t "*post-mating-delay*: ~d~%" *post-mating-delay*)
  (format t "*post-feq-delay*: ~d~%" *post-feq-delay*)
  (format t "*post-nfeq-delay*: ~d~%" *post-nfeq-delay*)
  (format t "*post-confront1-delay*: ~d~%" *post-confront1-delay*)
  (format t "*post-confront2-delay*: ~d~%" *post-confront2-delay*)
  (format t "*post-confront3-delay*: ~d~%" *post-confront3-delay*)
  (format t "*post-confront4-delay*: ~d~%" *post-confront4-delay*)
  )

