; Chad E Brown
; May 2009/Nov 2009/Jan 2010
; Main code for Satallax
; Started as a HO Tableau Prototype
; sbcl

; flags
(defvar *debug* nil)
(defvar *sym-eq* t)
(defvar *instantiate-with-func-diseqn-sides* t) ; if this is t, then instantiate quantifiers at a type (A > B) with terms that occur on disequations of type (A > B)
(defvar *imitate-defns* t) ; if this is t, then instantiate by imitating defns
(defvar *nbcon* nil) ; unwritten
(defvar *nbdec* nil) ; unwritten
(defvar *nbmat* nil) ; unwritten
(defvar *call-period* 100) 
(defvar *minisat-timeout* 1)
(defvar *minisat-timeout-init* 1) 
(defvar *minisat-timeout-incr* 0) 
(defvar *exists-delay* 1) 
(defvar *forall-delay* 1) 
(defvar *defaultelt-delay* 30) 
(defvar *defaulteltinst-delay* 30) 
(defvar *confr-diff-delay* 100) ; I'm not sure these confrontations are really needed for completeness, so delay them a long time
(defvar *confr-same1-delay* 5)
(defvar *confr-same2-delay* 0)
(defvar *enum-start* 2) 
(defvar *enum-arrow* 10) 
(defvar *enum-o* 5) 
(defvar *enum-sort* 2) 
(defvar *enum-neg* 5) 
(defvar *enum-or* 20) 
(defvar *enum-false* 20) 
(defvar *enum-choice* 0) 
(defvar *enum-eq* 5) 
(defvar *leibeq-to-primeq* nil)

(defvar *enum-special* nil) ; May 5, 2010 - list of special 'logical combinators' to do gensubs with.

; flags added Dec 28 2009
(defvar *enable-local-defns* nil) ; next 2 flags only matters if this is t
(defvar *local-defns-delay-pre* 2)
(defvar *local-defns-delay-post* 2)
(defvar *enable-pattern-rules* nil)
(defvar *pattern-rules-delay* 1)
(defvar *pattern-rules-eqn-delay* 1)
(defvar *enable-pattern-rules-eqns-as-atoms* nil) ; May 9 2010, allows pattern rules treating equations as atoms - maybe I should allow any nonatomic as well?
(defvar *enable-pattern-rules-diseqns-as-atoms* nil) ; May 11 2010, allows pattern rules treating equations as atoms - maybe I should allow any nonatomic as well?

(defvar *enable-pattern-clauses* nil) ; May 13 2010, pattern clauses
(defvar *pattern-clauses-delay* 1)

(defvar *split-global-disjunctions* nil) ; May 10 2010 - split global disjunctions (since these are essentially independent problems)

(defvar *filter-univ-usable* nil) ; May 11 2010, flag
(defvar *filter-posatm-usable* nil) ; May 11 2010, flag
(defvar *filter-negatm-usable* nil) ; May 11 2010, flag
(defvar *filter-poseq-usable* nil) ; May 11 2010, flag
(defvar *filter-negeq-usable* nil) ; May 11 2010, flag
(defvar *minisat-filter-timeout* 1) ; May 11 2010, flag

; flags added Dec 23 2009
(defvar *choice-as-default* nil) ; boolean, if t, use (choose x:a . false) as the default element.  otherwise a term or fresh name may be used.

; flags added Dec 23 2009, default to 0 not to affect prior behavior
(defvar *imitate-defn-delay* 0)
(defvar *imitate-delay* 0)
(defvar *project-delay* 0)
(defvar *new-head-enum-delay* 0)
(defvar *choice-empty-delay* 0)
(defvar *choice-in-delay* 0)
(defvar *post-or-l-delay* 0)
(defvar *post-or-r-delay* 0)
(defvar *post-nor-l-delay* 0)
(defvar *post-nor-r-delay* 0)
(defvar *post-eqo-l-delay* 0)
(defvar *post-eqo-r-delay* 0)
(defvar *post-eqo-nl-delay* 0)
(defvar *post-eqo-nr-delay* 0)
(defvar *post-neqo-l-delay* 0)
(defvar *post-neqo-r-delay* 0)
(defvar *post-neqo-nl-delay* 0)
(defvar *post-neqo-nr-delay* 0)
(defvar *post-dec-delay* 0)
(defvar *post-mating-delay* 0)
(defvar *post-feq-delay* 0)
(defvar *post-nfeq-delay* 0)
(defvar *post-confront1-delay* 0)
(defvar *post-confront2-delay* 0)
(defvar *post-confront3-delay* 0)
(defvar *post-confront4-delay* 0)

(defvar *clauses-disjs* nil) ; May 10 2010

(defvar *clauses* "")
(defvar *sorts* nil)
(defvar *consts* nil)
(defvar *dbfrees* (make-hash-table)) ; hash table indexed by terms
(defvar *arh* (make-hash-table :test #'equal))  ; indexed by (type . type)
(defvar *aph* (make-hash-table :test #'equal))  ; indexed by (term . term)
(defvar *lamh* (make-hash-table :test #'equal)) ; indexed by (type . term)
(defvar *dbh* (make-hash-table :test #'equal))  ; indexed by (num . type)
(defvar *evarh* (make-hash-table :test #'equal))  ; indexed by (name . <subst>)
(defvar *eqh* (make-hash-table))                ; indexed by type
(defvar *pih* (make-hash-table))                ; indexed by type
(defvar *sigh* (make-hash-table))               ; indexed by type
(defvar *epsilonh* (make-hash-table))               ; indexed by type
(defvar *defaulth* (make-hash-table))
(defvar *nextfresh* 0)
(defvar *handled* nil)
(defvar *posatm-usable* nil) ; indexed by head
(defvar *negatm-usable* nil) ; indexed by head
(defvar *poseq-usable* nil) ; indexed by sort
(defvar *negeq-usable* nil) ; indexed by sort
(defvar *univ-usable* nil) ; indexed by type
(defvar *choice-done* nil)
;(defvar *exists-blocked* nil) ; indexed by abstraction term
;(defvar *funcext-blocked* nil) ; indexed by abstraction term
(defvar *discriminating* nil) ; indexed by sort
(defvar *instantiations* nil) ; indexed by type
(defvar *generate-instantiations* nil) ; indexed by type
(defvar *waiting-enums* nil) ; indexed by sort
(defvar *agenda* nil)
(defvar *atomhash* (make-hash-table))
(defvar *nextatom* 0)
(defvar *searching* nil)

; hash table for local defns and pattern rules added Dec 28 2009
(defvar *local-defns* nil)
(defvar *consts-in-hash* nil)
(defvar *pattern-rules-pos* nil)
(defvar *pattern-rules-neg* nil)
(defvar *pattern-rules-eqn* nil)
(defvar *pattern-clauses* nil)
(defvar *pattern-clauses-usable* nil) ; formulas I can try to match with literals in pattern clauses as they are created

(defvar *used* nil)

(defvar *O* nil)
(defvar *TRUE* nil)
(defvar *FALSE* nil)
(defvar *NEG* nil)
(defvar *AND* nil)
(defvar *OR* nil)
(defvar *IMPLIES* nil)
(defvar *EQUIV* nil)

(defvar *conjecture-given* nil)

(defun thf-fail (msg s)
  (format t "*** WARNING: ~d Skipping ~S~%" msg s)
  (throw 'done 'thf-fail))

(defun ar (a b)
  (let ((ab (cons a b)))
    (let ((abh (gethash ab *arh*)))
      (or abh
	  (progn
	    (setf (gethash ab *arh*) ab)
	    ab)))))

(defun stp (a)
  (if (and (consp a) (equal (car a) ">") (consp (cdr a)) (consp (cddr a)))
      (ar (stp (cadr a)) (stp (caddr a)))
    (if (stringp a)
	(let ((z (intern a)))
	  (unless (or (eq z *O*) (get z 'basetp))
	    (declare-basetype z))
	  z)
      (thf-fail "Not a valid type" a))))

(defun rtp (a)
  (if (consp a) (rtp (cdr a)) a))

; just for debugging - Dec 29 2009
(defun get-sub-partial-ctx (s &optional ctx ms)
  (if s
      (get-sub-partial-ctx (cdr s)
			   (get-partial-ctx (car s) ctx ms)
			   (cons (car s) ms))
    ctx))

; just for debugging - Dec 29 2009
(defun get-partial-ctx (m &optional ctx ms)
  (if (consp (car m))
      (case (caar m)
	    (AP
	     (let ((ctx1 (get-partial-ctx (cadar m) ctx ms)))
	       (get-partial-ctx (cddar m) ctx1 (cons (cadar m) ms))))
	    (LAM
	     (cdr (get-partial-ctx (cdar m) (cons (cadr m) ctx) ms)))
	    (EVAR
	     (get-sub-partial-ctx (cddar m) ctx ms))
	    (t ctx))
    (let ((s (car m)))
      (if (numberp s)
	  (let ((a (nth s ctx)))
	    (if a
		(if (eq a (cdr m))
		    ctx
		  (thf-fail "No ctx possible" (cons m ms)))
	      (progn
		(when (<= (length ctx) s)
		  (setq ctx (append ctx (make-list (1+ (- s (length ctx)))))))
		(setf (nth s ctx) (cdr m))
		ctx)))
	ctx))))

(defun ap (a b)
  (if (and (consp (cdr a)) (eq (cadr a) (cdr b)))
      (let ((ab (cons a b)))
	(let ((abh (gethash ab *aph*)))
	  (or abh
	      (let ((apab (cons (cons 'AP ab) (cddr a))))
;		(when *debug* (get-partial-ctx apab)) ; uncomment to debug
		(setf (gethash ab *aph*) apab)
		apab))))
    (thf-fail "Not well-typed" (cons a b))))

(defun ap-p (m) (and (consp m) (consp (car m)) (eq (caar m) 'AP)))
(defun ap-f (m) (cadar m))
(defun ap-a (m) (cddar m))
(defun lam-p (m) (and (consp m) (consp (car m)) (eq (caar m) 'LAM)))
(defun lam-dom (m) (cadr m))
(defun lam-body (m) (cdar m))
(defun neg-p (m) (and (ap-p m) (eq (cadar m) *NEG*)))
(defun eq-p (m) (and (consp m) (consp (car m)) (eq (caar m) 'EQ)))
(defun eq-tp (m) (cdar m))
(defun binop (m) (and (ap-p m) (ap-p (ap-f m)) (ap-f (ap-f m))))
(defun bin-l (m) (ap-a (ap-f m)))
(defun bin-r (m) (ap-a m))
(defun and-p (m) (eq (binop m) *AND*))
(defun or-p (m) (eq (binop m) *OR*))
(defun eqn-p (m) (and (ap-p m) (ap-p (ap-f m)) (eq-p (ap-f (ap-f m)))))
(defun diseqn-p (m) (and (neg-p m) (eqn-p (ap-a m))))
(defun diseqn-l (m) (bin-l (ap-a m)))
(defun diseqn-r (m) (bin-r (ap-a m)))
(defun pi-p (m) (and (consp m) (consp (car m)) (eq (caar m) 'PI)))
(defun pi-tp (m) (cdar m))
(defun all-p (m) (and (ap-p m) (pi-p (ap-f m))))
(defun sig-p (m) (and (consp m) (consp (car m)) (eq (caar m) 'SIG)))
(defun sig-tp (m) (cdar m))
(defun choiceop-p (m) (and (consp m)
			   (or (and (consp (car m)) (eq (caar m) 'CHOOSE))
			       (and (symbolp (car m)) (get (car m) 'CHOICEFN)))))
(defun choice-p (m) (and (ap-p m) (choiceop-p (ap-f m))))

(defun lam (a b)
  (let ((ab (cons a b)))
    (let ((abh (gethash ab *lamh*)))
      (or abh
	  (let ((lamab (cons (cons 'LAM b) (ar a (cdr b)))))
	    (setf (gethash ab *lamh*) lamab)
	    lamab)))))

(defun db2 (na)
  (let ((nah (gethash na *dbh*)))
    (or nah
	(progn
	  (setf (gethash na *dbh*) na)
	  na))))

(defun db (n a)
  (db2 (cons n a)))

; corresponds to eta expanding on the fly
(defun genlam-body (m)
  (if (lam-p m)
      (lam-body m)
    (ap (shift m 1 0) (db 0 (cadr m)))))

(defun leq (a)
  (let ((e (gethash a *eqh*)))
    (or e
	(let ((etp (ar a (ar a *O*))))
	  (setf (gethash a *eqh*) (acons 'EQ a etp))))))

(defun lpi (a)
  (let ((e (gethash a *pih*)))
    (or e
	(let ((qtp (ar (ar a *O*) *O*)))
	  (setf (gethash a *pih*) (acons 'PI a qtp))))))

(defun lsig (a)
  (let ((e (gethash a *sigh*)))
    (or e
	(let ((qtp (ar (ar a *O*) *O*)))
	  (setf (gethash a *sigh*) (acons 'SIG a qtp))))))

(defun lepsilon (a)
  (let ((e (gethash a *epsilonh*)))
    (or e
	(let ((qtp (ar (ar a *O*) a)))
	  (setf (gethash a *epsilonh*) (acons 'CHOOSE a qtp))))))

(defun all (a m)
  (ap (lpi a) (lam a m)))

(defun choose (a m)
  (ap (lepsilon a) (lam a m)))

(defun set-default-elt (c)
  (setf (gethash (cdr c) *defaulth*) c))

(defun default-elt-p (a) (gethash a *defaulth*))

(defun default-elt (a)
  (let ((e (gethash a *defaulth*)))
    (or e
	(progn
	  (declare-const (intern (format nil "__~d" (incf *nextfresh*))) a) ; declare a dummy constant of the sort a
	  (gethash a *defaulth*)))))

(defun neg (m)
  (ap *NEG* m))

(defun conj (m n)
  (ap (ap *AND* m) n))

(defun disj (m n)
  (ap (ap *OR* m) n))

(defun imp (m n)
  (ap (ap *IMPLIES* m) n))

(defun equiv (m n)
  (ap (ap *EQUIV* m) n))

(defun eqn (m n)
  (ap (ap (leq (cdr m)) m) n))

(defun diseqn (m n)
  (neg (ap (ap (leq (cdr m)) m) n)))

(defun ctx-lookup (x ctx &optional (i 0))
  (if ctx
      (if (eq x (caar ctx))
	  (cons i (cdar ctx))
	(ctx-lookup x (cdr ctx) (1+ i)))
    nil))

(defun bindtrm (q xl b ctx)
  (if xl
      (if (stringp (car xl))
;	  (progn
;	    (format t "*+* Missing type for ~d -- trying 'nat'~%" (car xl))
;	    (bindtrm q (cons (list ":" (car xl) "nat") (cdr xl)) b ctx)
;	    )
	  (thf-fail (format nil "missing type on ~d-bound variable" q) (car xl))
	(let ((x (intern (cadar xl)))
	      (xtp (stp (caddar xl))))
	  (case q
		(|^| (lam xtp (bindtrm q (cdr xl) b (acons x xtp ctx))))
		(|!| (ap (lpi xtp) (lam xtp (bindtrm q (cdr xl) b (acons x xtp ctx)))))
		(|?| (ap (lsig xtp) (lam xtp (bindtrm q (cdr xl) b (acons x xtp ctx)))))
		(|@+| (ap (lepsilon xtp) (lam xtp (bindtrm q (cdr xl) b (acons x xtp ctx)))))
		(t (thf-fail "impossible binder problem, seriously, completely impossible" (list q xl b ctx))))))
    (trm b ctx)))

(defun trm (f &optional ctx)
  (if (consp f)
      (let ((ff (intern (car f))))
	(case ff
	      (|@| (ap (trm (cadr f) ctx) (trm (caddr f) ctx))) ; application
	      ((|^| |!| |?|) ; binders
	       (bindtrm ff (cadr f) (caddr f) ctx))
	      (~ (neg (trm (cadr f) ctx)))
	      (|\|| (disj (trm (cadr f) ctx) (trm (caddr f) ctx)))
	      (& (conj (trm (cadr f) ctx) (trm (caddr f) ctx)))
	      (=> (imp (trm (cadr f) ctx) (trm (caddr f) ctx)))
	      (<= (imp (trm (caddr f) ctx) (trm (cadr f) ctx)))
	      (<=> (equiv (trm (cadr f) ctx) (trm (caddr f) ctx)))
	      (<~> (neg (equiv (trm (cadr f) ctx) (trm (caddr f) ctx))))
	      (=
	       (let* ((m (trm (cadr f) ctx))
		      (n (trm (caddr f) ctx))
		      (a (cdr m)))
		 (if (eq a (cdr n))
		     (eqn m n)
		   (thf-fail "ill typed eqn" f))))
	      (!=
	       (let* ((m (trm (cadr f) ctx))
		      (n (trm (caddr f) ctx))
		      (a (cdr m)))
		 (if (eq a (cdr n))
		     (diseqn m n)
		   (thf-fail "ill typed diseqn" f))))
	      (t (thf-fail "ill formed wff" f))))
    (if (stringp f)
	(let* ((ff (intern f))
	       (p (ctx-lookup ff ctx)))
	  (if p
	      (db2 p)
	    (cond ((get ff 'const) (get ff 'consth))
		  ((get ff 'def) (get ff 'defh))
		  (t
		   (case ff
			 (|$true| *TRUE*)
			 (|$false| *FALSE*)
			 (~ *NEG*) ; connectives as names
			 (|\|| *OR*)
			 (& *AND*)
			 (=> *IMPLIES*)
			 (<=> *EQUIV*)
			 (t (thf-fail "unknown name" f)))))))
      (thf-fail "unknown trm" f))))

(defun tp-str (a)
  (if (consp a)
      (format nil "(~d > ~d)" (tp-str (car a)) (tp-str (cdr a)))
    (format nil "~d" a)))
  
(defun wff-str (m &optional names)
  (if (ap-p m)
      (cond ((neg-p m)
	     (format nil "(~~ ~d)" (wff-str (ap-a m) names))
	     )
	    ((and-p m)
	     (format nil "(~d & ~d)" (wff-str (bin-l m) names) (wff-str (bin-r m) names))
	     )
	    ((eqn-p m)
	     (format nil "(~d = ~d)" (wff-str (bin-l m) names) (wff-str (bin-r m) names))
	     )
	    ((all-p m)
	     (format nil "(! ~d)" (wff-str (ap-a m) names)))
	    (t
	     (format nil "(~d ~d)" (wff-str (ap-f m) names) (wff-str (ap-a m) names))))
    (if (lam-p m)
	(let ((x (format nil "x~d" (length names))))
	  (format nil "(\\~d:~d.~d)" x (tp-str (lam-dom m)) (wff-str (lam-body m) (cons x names))))
      (if (pi-p m)
	  "!"
	(if (sig-p m)
	    "?"
	  (if (choiceop-p m)
	      "@+"
	    (if (evar-p m)
		(format nil "~d[~d]" (cadar m)
			(esub-str (cddar m) names))
	      (if (numberp (car m))
		  (or 
		   (nth (car m) names)
		   (format nil "{~d}" (- (car m) (length names))))
		(format nil "~d" (car m))))))))))

(defun esub-str (sub names)
  (if sub
      (if (cdr sub)
	  (format nil "~d,~d"
		  (wff-str (car sub) names)
		  (esub-str (cdr sub) names))
	(wff-str (car sub) names))
    ""))

(defun logic-init ()
  (setq *searching* nil)
  (setq *sorts* nil)
  (setq *defaulth* (make-hash-table))
  (setq *nextfresh* 0)
  (setq *atomhash* (make-hash-table))
  (setq *nextatom* 0)
  (setq *O* '|$o|)
  (let* ((oo (ar *O* *O*))
	 (ooo (ar *O* oo)))
    (setq *TRUE* (cons '|$true| *O*))
    (setq *FALSE* (cons '|$false| *O*))
    (setq *NEG* (cons '_NEG oo))
    (setq *AND* (cons '_AND ooo))
    (setq *OR* (cons '_OR ooo))
    (setq *IMPLIES* (cons '_IMPLIES ooo))
    (setq *EQUIV* (cons '_EQUIV ooo))
    ))

(defun clear-sym (c)
  (setf (get c 'basetp) nil)
  (setf (get c 'const) nil)
  (setf (get c 'consth) nil)
  (setf (get c 'def) nil)
  (setf (get c 'defh) nil))

(defun declare-basetype (a)
  (when (or (get a 'basetp) (get a 'const) (get a 'def)) (thf-fail "Redeclaration of base type" a))
  (setf (get a 'basetp) t)
  (when *choice-as-default* (set-default-elt (ap (lepsilon a) (lam a *FALSE*)))) ; Use choose x . \bot as default elt
  (push a *sorts*)
  (push a *used*)
  nil)

(defun declare-const (c a)
  (when (or (get c 'basetp) (get c 'const) (get c 'def)) (thf-fail "Redeclaration of constant" (list c a)))
  (setf (get c 'const) a)
  (setf (get c 'consth) (cons c a))
  (setf (get c 'rtp) (rtp a))
  (push c *consts*)
  (push c *used*)
  (when (not (or (consp a) (eq a *O*)))
    (unless (default-elt-p a) (set-default-elt (get c 'consth))))
  (when *searching*
    (dolist (e (gethash (get c 'rtp) *waiting-enums*)) ; fire any relevant enumeration processes with this new head
      (agenda-en (list 'DELAY *new-head-enum-delay* (cons 'ENUM-AP (cons (get c 'consth) e))))))
  nil)

(defun declare-def (c m)
  (when (or (get c 'basetp) (get c 'def)) (thf-fail "Redeclaration of constant" (list c)))
  (let ((a (cdr m)))
    (when (get c 'const)
      (if (eq a (get c 'const))
	  (progn
	    (setf (get c 'const) nil)
	    (setf (get c 'consth) nil))
	(thf-fail "Definition does not match declared type" (list c a m))))
    (setf (get c 'def) m)
    (setf (get c 'defh) (cons c a))
    (push c *used*)
    nil))

(defun evar-p (m)
  (and (consp (car m))
       (eq (caar m) 'EVAR)))

(defun evar (X sub)
  (let* ((X-sub (cons X sub))
	 (Xsub (gethash X-sub *evarh*)))
    (or Xsub
	(let ((Xsub (cons (cons 'EVAR X-sub) (get X 'tp))))
;	  (when *debug* (unless (equal (mapcar #'cdr sub) (get X 'ctx)) (thf-fail "Not well-typed" Xsub)))
;	  (when *debug* (get-partial-ctx Xsub)) ; uncomment to debug
	  (setf (gethash X-sub *evarh*) Xsub)
	  Xsub))))

(defun new-evar (ctx tp)
  (if (consp tp)
      (multiple-value-bind
       (X Xsub)
       (new-evar (cons (car tp) ctx) (cdr tp))
       (values X (lam (car tp) Xsub)))
  ; intern if debugging so I can examine the symbol
    (let ((X (if *debug* (intern (format nil "~d" (gensym))) (gensym)))
	  (sub nil)
	  (i -1))
      (dolist (a ctx)
	(push (db (incf i) a) sub)) 
      (setf (get X 'evar) t)
      (setf (get X 'ctx) ctx)
      (setf (get X 'tp) tp)
      (values X (evar X (reverse sub))))))

(defun shift (m i j)
  (if (consp (car m))
      (case (caar m)
	    (AP (ap (shift (cadar m) i j)
		    (shift (cddar m) i j)))
	    (LAM (lam (cadr m) (shift (cdar m) i (1+ j))))
	    (EVAR (evar (cadar m)
			(mapcar #'(lambda (mm)
				    (shift mm i j))
				(cddar m))))
	    (t m))
    (let ((s (car m)))
      (if (numberp s)
	  (if (< s j)
	      m
	    (db (+ s i) (cdr m)))
	m))))

(defun dbsubst (m n i)
;  (format t "*~d* dbsubst in :~%m = ~d~%~S~%n = ~d~%~S~%i = ~d~%" *debuglevel* (wff-str m) m (wff-str n) n i)
;  (incf *debuglevel*)
  (let ((r (dbsubst-real m n i)))
;    (format t "*~d* dbsubst out : ~d~%~S~%" (decf *debuglevel*) (wff-str r) r)
    r))

(defun dbsubst-real (m n i)
  (if (consp (car m))
      (case (caar m)
	    (AP (ap (dbsubst (cadar m) n i)
		    (dbsubst (cddar m) n i)))
	    (LAM (lam (cadr m) (dbsubst (cdar m) n (1+ i))))
	    (EVAR (evar (cadar m)
			(mapcar #'(lambda (mm)
				    (dbsubst-real mm n i))
				(cddar m))))
	    (t m))
    (let ((s (car m)))
      (if (numberp s)
	  (if (= s i)
	      (if (= i 0)
		  n
		(shift n i 0))
	    (if (> s i)
		(db (- s 1) (cdr m))
	      m))
	m))))

(defun dbsubst-simul (m sub i)
;  (format t "*~d* dbsubst-simul in :~%m = ~d~%~S~%sub = ~S~%" *debuglevel* (wff-str m) m sub)
;  (incf *debuglevel*)
  (let ((r (dbsubst-simul-real m sub i)))
;    (format t "*~d* dbsubst-simul out : ~d~%~S~%" (decf *debuglevel*) (wff-str r) r)
    r))

(defun dbsubst-simul-real (m sub i)
  (if (consp (car m))
      (case (caar m)
	    (AP (ap (dbsubst-simul (cadar m) sub i)
		    (dbsubst-simul (cddar m) sub i)))
	    (LAM (lam (cadr m) (dbsubst-simul (cdar m) sub (1+ i))))
	    (EVAR (evar (cadar m)
			(mapcar #'(lambda (mm)
				    (dbsubst-simul-real mm n i))
				(cddar m))))
	    (t m))
    (let ((s (car m)))
      (if (numberp s)
	  (if (< s i)
	      m
	    (let ((n (nth (- s i) sub))) ; bug fixed, Jan 7 2010, (- s i) was s
	      (if n
		  (if (= i 0)
		      n
		    (shift n i 0))
		(throw 'dbsubst-simul-failed nil)
		)))
	m))))

(defun myneg (m)
  (if (neg-p m)
      (cddar m)
    (neg m)))

(defun head (m)
  (if (ap-p m)
      (head (cadar m))
    m))

(defun args (m &optional args)
  (if (ap-p m)
      (args (ap-f m) (cons (ap-a m) args))
    args))

(defun dbfrees (m)
  (multiple-value-bind
   (vl i)
   (gethash m *dbfrees*)
   (if i
       vl
     (let ((vl (dbfrees-real m)))
       (setf (gethash m *dbfrees*) vl)
       ))))

(defun dbfrees-real (m)
  (if (consp (car m))
      (case (caar m)
	    (AP (union (dbfrees (ap-f m)) (dbfrees (ap-a m))))
	    (LAM
	     (let ((vl nil))
	       (dolist (x (dbfrees (lam-body m)) vl)
		 (when (> x 0)
		   (push (- x 1) vl)))))
	    (EVAR
	     (let ((vl nil))
	       (dolist (mm (cddar m) (remove-duplicates vl))
		 (setq vl (append (dbfrees mm) vl)))))
	    (t nil))
    (if (numberp (car m))
	(list (car m))
      nil)))

; beta, eta, delta normalize, rewrite all logic into #, ~, |, !, = -- reduce double negations
; expand definitions
(defun norm-1 (m)
  (if (consp (car m))
      (if (and (neg-p m) (neg-p (cddar m)))
	  (cddar (cddar m))
	(case (caar m)
	      (AP
	       (let ((mf (norm-1 (cadar m)))
		     (ma (norm-1 (cddar m))))
		 (if (lam-p mf)
		     (dbsubst (cdar mf) ma 0)
		   (ap mf ma))))
	      (LAM
	       (let ((mb (norm-1 (cdar m))))
		 (if (and (ap-p mb) 
			  (equal (car (ap-a mb)) 0)
			  (not (member 0 (dbfrees (ap-f mb)) :test #'=))) ; eta redex
		     (shift (ap-f mb) -1 0)
		   (lam (cadr m) mb))))
	      (SIG
	       (let ((a (cdar m))
		     (ao (cadr m))) ; rewrite SIG as PI
		 (lam ao (neg (ap (lpi a) (lam a (neg (ap (db 1 ao) (db 0 a)))))))))
	      (EVAR (evar (cadar m)
			  (mapcar #'(lambda (mm)
				      (norm-1 mm))
				  (cddar m))))
	      (t m)
	      ))
    (let ((s (car m)))
      (if (and (symbolp s) (get s 'def))
	  (get s 'def)
	(cond ((eq m *AND*) (lam *O* (lam *O* (neg (disj (neg (db 1 *O*)) (neg (db 0 *O*)))))))
	      ((eq m *TRUE*) (neg *FALSE*))
	      ((eq m *IMPLIES*) (lam *O* (lam *O* (disj (neg (db 1 *O*)) (db 0 *O*)))))
	      ((eq m *EQUIV*) (lam *O* (lam *O* (eqn (db 1 *O*) (db 0 *O*)))))
	      (t m))))))

(defun norm (m)
  (let ((n (norm-1 m)))
;    (when *debug* (format t "Normalizing ~d~%" (wff-str m)))
    (loop until (eq n m) do
;	  (when *debug* (format t "Reduced to ~d~%~S~%" (wff-str n) n))
	  (setq m n)
	  (setq n (norm-1 m)))
;    (when *debug* (format t "Normal ~d~%" (wff-str n)))
    n))

(defun preprocess (m)
  (if *leibeq-to-primeq*
      (preprocess-real m)
    m))

(defun leib-or-p (l r)
  (and (neg-p l)
       (ap-p (ap-a l))
       (ap-p r)
       (equal (car (ap-f (ap-a l))) 0)
       (equal (car (ap-f r)) 0)
       (not (member 0 (dbfrees (ap-a (ap-a l))) :test #'=))
       (not (member 0 (dbfrees (ap-a r)) :test #'=))))

(defun preprocess-real (m)
  (if (and *leibeq-to-primeq*
	   (all-p m)
	   (lam-p (ap-a m))
	   (or-p (lam-body (ap-a m)))
	   (or (leib-or-p (bin-l (lam-body (ap-a m)))
			  (bin-r (lam-body (ap-a m))))
	       (leib-or-p (bin-r (lam-body (ap-a m)))
			  (bin-l (lam-body (ap-a m))))))
      (if (leib-or-p (bin-l (lam-body (ap-a m)))
		     (bin-r (lam-body (ap-a m))))
	  (eqn (shift (ap-a (ap-a (bin-l (lam-body (ap-a m))))) -1 0)
	       (shift (ap-a (bin-r (lam-body (ap-a m)))) -1 0))
	(eqn (shift (ap-a (ap-a (bin-r (lam-body (ap-a m))))) -1 0)
	     (shift (ap-a (bin-l (lam-body (ap-a m)))) -1 0)))
    (if (consp (car m))
	(case (caar m)
	      (AP (ap (preprocess-real (ap-f m)) (preprocess-real (ap-a m))))
	      (LAM (lam (cadr m) (preprocess-real (cdar m))))
	      (EVAR (evar (cadar m)
			  (mapcar #'(lambda (mm)
				      (preprocess-real mm))
				  (cddar m))))
	      (t m))
      m)))

; throws done inconsistent if the set of clauses is found to be unsatisfiable
; otherwise just returns nil
(defun call-minisat ()
  (when *verbose* (format t "Calling minisat..."))
  (when *debug* (write-string *clauses*))
  (let* ((p (sb-ext:run-program *minisat-binary* nil :wait nil :input :stream))
	 (ps (sb-ext:process-input p)))
    (handler-case
     (progn
       (write-string *clauses* ps)
       (close ps))
     (error (x) 
	    (unless *slave*
	      (format t "% SZS status Error :~%Problem writing to minisat ~d~%" *minisat-binary*)
	      )
	    (sb-ext:quit :unix-status 1)))
    (handler-case
     (with-timeout *minisat-timeout*
       (sb-ext:process-wait p t)
       (if (equal (sb-ext:process-exit-code p) 20)
	   (progn
	     (when *verbose* (format t "unsatisfiable!~%"))
	     (throw 'inconsistent 'inconsistent)) ; success!
	 (progn
	   (when *verbose* (format t "satisfiable so far.~%"))
	   (filter-usable)
	   nil)))
     (timeout ()
	      (sb-ext:process-kill p 9)
	      (when *verbose* (format t "timed out.~%"))
	      (filter-usable)
	      (setq *minisat-timeout* (+ *minisat-timeout* *minisat-timeout-incr*)) ; unclear if I should increase the timeout -- seems that more clauses might make it find unsat faster -- need to experiment to find out what works.
	      ))))

(defun call-minisat-final ()
  (when *verbose* (format t "Final call to minisat..."))
  (let* ((p (sb-ext:run-program *minisat-binary* nil :wait nil :input :stream))
	 (ps (sb-ext:process-input p)))
    (write-string *clauses* ps)
    (close ps)
    (sb-ext:process-wait p t)
    (if (equal (sb-ext:process-exit-code p) 20)
	(progn
	  (when *verbose* (format t "unsatisfiable!~%"))
	  (throw 'inconsistent 'inconsistent)) ; success!
      (if (equal (sb-ext:process-exit-code p) 10)
	   (progn
	     (when *verbose* (format t "satisfiable!~%"))
	     (throw 'done 'satisfiable)) ; satisfiable -- this is only used in the 'final' case (can only happen in EFO-like fragment)
	(progn
	  (when *verbose* (format t "terminated.~%"))
	  (throw 'done 'terminated))))))

(defun filter-usable ()
  (when *filter-univ-usable*
    (maphash #'(lambda (a mnpl)
		 (setf (gethash a *univ-usable*)
		       (remove-if #'filter-literal mnpl
				  :key #'cadr)))
	     *univ-usable*))
  (when *filter-posatm-usable*
    (maphash #'(lambda (key val)
		 (setf (gethash key *posatm-usable*)
		       (remove-if #'filter-literal val
				  :key #'cadr)))
	     *posatm-usable*))
  (when *filter-negatm-usable*
    (maphash #'(lambda (key val)
		 (setf (gethash key *negatm-usable*)
		       (remove-if #'filter-literal val
				  :key #'cadr)))
	     *negatm-usable*))
  (when *filter-poseq-usable*
    (maphash #'(lambda (key val)
		 (setf (gethash key *poseq-usable*)
		       (remove-if #'filter-literal val
				  :key #'cadr)))
	     *poseq-usable*))
  (when *filter-negeq-usable*
    (maphash #'(lambda (key val)
		 (setf (gethash key *negeq-usable*)
		       (remove-if #'filter-literal val
				  :key #'cadr)))
	     *negeq-usable*))
  )

; Call minisat to
; determine if the negation of this literal is inconsistent with the current clause set,
; in which case we drop it we may drop it from usable.
(defun filter-literal (lit)
  (when *verbose* (format t "Calling minisat to filter ~d..." lit))
  (let* ((p (sb-ext:run-program *minisat-binary* nil :wait nil :input :stream))
	 (ps (sb-ext:process-input p)))
    (handler-case
     (progn
       (write-string *clauses* ps)
       (write-string (format nil "~d 0~%" (- lit)) ps)
       (close ps))
     (error (x) 
	    (unless *slave*
	      (format t "% SZS status Error :~%Problem writing to minisat ~d~%" *minisat-binary*)
	      )
	    (sb-ext:quit :unix-status 1)))
    (handler-case
     (with-timeout *minisat-filter-timeout*
       (sb-ext:process-wait p t)
       (if (equal (sb-ext:process-exit-code p) 20)
	   (progn
	     (when *verbose* (format t "filtered~%")) 
	     (append-clause (format nil "~d" lit)) ; add its negation to the clause set
	     t)
	 (progn
	   (when *verbose* (format t "not filtered.~%"))
	   nil)))
     (timeout ()
	      (sb-ext:process-kill p 9)
	      (when *verbose* (format t "timed out.~%"))
	      (setq *minisat-timeout* (+ *minisat-timeout* *minisat-timeout-incr*)) ; unclear if I should increase the timeout -- seems that more clauses might make it find unsat faster -- need to experiment to find out what works.
	      ))))

(defun get-literal (m)
  (if (neg-p m)
      (- (get-atom (ap-a m)))
    (get-atom m)))

(defun get-atom (m)
  (let ((a (gethash m *atomhash*)))
    (if a
	a
      (if (and *sym-eq* (eqn-p m)) ; get atom up to symmetry of equality
	  (let* ((m2 (eqn (bin-r m) (bin-l m)))
		 (a (gethash m2 *atomhash*)))
	    (if a
		a
	      (new-atom m)))
	(new-atom m)))))

(defun new-atom (m)
  (let ((a (incf *nextatom*)))
    (when *debug* (format t "Atom ~d: ~d~%" a (wff-str m)))
    (setf (gethash m *atomhash*) a)))

(defun append-clause (cl)
  (when *debug* (format t "Adding clause ~d~%" cl))
  (setq *clauses* (format nil "~d~d 0~%" *clauses* cl))
  )

; eps:(a>o)>a is a choice function
; s is first argument
; add clause <!x.~s x> | <s (eps s)>
; and process ~!x.~s x and ~(s (eps s))
(defun choice-clause (eps s)
  (unless (gethash (cons eps s) *choice-done*)
    (setf (gethash (cons eps s) *choice-done*) t)
    (let* ((a (cddr eps))
	   (empty-s (norm (ap (lpi a) (lam a (neg (ap (shift s 1 0) (db 0 a)))))))
	   (s-eps-s (norm (ap s (ap eps s)))))
      (when *debug* (format t "Choice Clause:~%Empty: ~d~%In: ~d~%" (wff-str empty-s) (wff-str s-eps-s)))
      (append-clause (format nil "~d ~d"
			     (get-literal empty-s)
			     (get-literal s-eps-s)))
      (agenda-en (list 'DELAY *choice-empty-delay* (list 'PROP empty-s))) ; was a bug in <= 1.1, negated this, fixed - Chad, Jan 15, 2010
      (agenda-en (list 'DELAY *choice-in-delay* (list 'PROP s-eps-s))) ; was a bug in <= 1.1, negated this, fixed - Chad, Jan 15, 2010
      )))

(defun new-instantiation (a m)
  (unless (member m (gethash a *instantiations*)) ; if it's really new
    (when *verbose*
      (format t "Adding Instantiation of type ~d: ~d~%" (tp-str a) (wff-str m)))
    (push m (gethash a *instantiations*)) ; put it on
    ; and use it immediately
    (dolist (mnp (gethash a *univ-usable*))
      (let* ((nlit (cadr mnp))
	     (pred (caddr mnp))
	     (predm (norm (ap pred m))))
	(when *debug*
	  (format t "Applying ~d to it gives~%~d~%" (wff-str pred) (wff-str predm)))
	(append-clause (format nil "~d ~d" nlit (get-literal predm)))
	(agenda-en (list 'DELAY *forall-delay* (list 'PROP predm)))))))

(defun consider-confrontation (nnlit nplit u v l r)
  (let ((uh (head u))
	(vh (head v))
	(lh (head l))
	(rh (head r))
	(delay *confr-diff-delay*))
    (if (or (and (eq uh lh) (eq vh rh)) (and (eq uh rh) (eq vh lh)))
	(setq delay *confr-same2-delay*)
      (when (or (eq uh lh) (eq vh rh) (eq uh rh) (eq vh lh))
	(setq delay *confr-same1-delay*)))
					; delay based on similarity of heads
    (let ((conag
	   (list 'CONFRONT
		 nnlit nplit
		 u v l r)))
      (agenda-en (list 'DELAY
		       delay
		       conag)))))

(defun permute-to-front (m i j)
  (if (= i j)
      m
    (permute-to-front-real m i j)))

(defun permute-to-front-real (m i j)
  (if (consp (car m))
      (case (caar m)
	    (AP (ap (permute-to-front-real (cadar m) i j)
		    (permute-to-front-real (cddar m) i j)))
	    (LAM (lam (cadr m) (permute-to-front-real (cdar m) (1+ i) (1+ j))))
	    (EVAR (evar (cadar m)
			(mapcar #'(lambda (mm)
				    (permute-to-front-real mm i j))
				(cddar m))))
	    (t m))
    (let ((s (car m)))
      (if (numberp s)
	  (if (< s j)
	      m
	    (if (< s i)
		(db (+ s 1) (cdr m))
	      (if (= s i)
		  (db j (cdr m))
		m)))
	m))))

(defun invert-trm (args m &optional z)
  (if args
      (if (numberp (caar args))
	  (if (member (caar args) z :test #'=)
	      (throw 'invert-failed nil)
	    (invert-trm (cdr args) (lam (cdar args) (permute-to-front m (caar args) 0))))
	(throw 'invert-failed nil))
    (if (dbfrees m)
	(throw 'invert-failed nil)
      m)))

; make a partial explicit substitution (with some nils)
; using it in dbsubst-simul may result in throwing 'dbsubst-simul-failed
; ctx is a list of types an,...,a1
; sub is a substitution (list of terms) sub:ctx -> ctx' (ctx' can be extracted from sub by taking list of types of sub)
; return value r is a partial subst (list of term|nil) r:ctx' -> ctx
; in particular, r should be a list of term|nil where the types are [an|nil] ... [a1|nil]
(defun pattern-invert-sub (ctx sub &optional (i 0))
  (if ctx
      (let ((a (car ctx))
	    (p (position i sub :test #'(lambda (i m)
					 (equal i (car m))))))
	(if p
	    (cons (db p a) (pattern-invert-sub (cdr ctx) sub (1+ i)))
	  (cons nil (pattern-invert-sub (cdr ctx) sub (1+ i)))))
    nil))

(defun new-local-defn (lit m)
  (if (all-p m)
      (if (lam-p (ap-a m))
	  (new-local-defn lit (lam-body (ap-a m)))
	nil)
    (if (eqn-p m)
	(let* ((l (bin-l m))
	       (lh (head l))
	       (largs (args l))
	       (r (bin-r m))
	       (rh (head r))
	       (rargs (args r))
	       )
	  (when (and (symbolp (car l))
		     (get (car l) 'const))
	    (catch 'invert-failed
	      (let ((inv (invert-trm largs r)))
		(new-local-defn-2 lit (car l) inv))))
	  (when (and (symbolp (car r))
		     (get (car r) 'const))
	    (catch 'invert-failed
	      (let ((inv (invert-trm rargs l)))
		(new-local-defn-2 lit (car r) inv))))
	  )
      nil)))

(defun new-local-defn-2 (lit h hdef)
  (push (list lit h hdef) *local-defns*)
  (dolist (m-nmlit (gethash h *consts-in-hash*))
    (let* ((m (car m-nmlit))
	   (nmlit (cdr m-nmlit))
	   (m2 (norm (const-subst m h hdef)))
	   (m2lit (get-literal m2)))
      (format nil "~d ~d ~d" lit nmlit m2lit)
      (agenda-en (list 'DELAY *local-defns-delay-pre* (list 'PROP m2)))
      )))

; n should have no db's free.  not shifting
(defun const-subst (m c n)
  (if (consp (car m))
      (case (caar m)
	    (AP (ap (const-subst (cadar m) c n)
		    (const-subst (cddar m) c n)))
	    (LAM (lam (cadr m) (const-subst (cdar m) c n)))
	    (EVAR (evar (cadar m)
			(mapcar #'(lambda (mm)
				    (const-subst mm c n))
				(cddar m))))
	    (t m))
    (let ((s (car m)))
      (if (eq s c)
	  n
	m))))

; theta maps evars to terms in appropriate ctx
(defun evar-subst (m theta)
;  (format t "*~d* evar-subst in :~%m = ~d~%~S~%" *debuglevel* (wff-str m) m)
;  (incf *debuglevel*)
  (let ((r (evar-subst-real m theta)))
;    (format t "*~d* evar-subst out : ~d~%~S~%" (decf *debuglevel*) (wff-str r) r)
    r))

(defun evar-subst-real (m theta)
  (if (consp (car m))
      (case (caar m)
	    (AP (ap (evar-subst (cadar m) theta)
		    (evar-subst (cddar m) theta)))
	    (LAM (lam (cadr m) (evar-subst (cdar m) theta)))
	    (EVAR
	     (let ((a (assoc (cadar m) theta))
		   (sub (mapcar #'(lambda (mm)
				    (evar-subst mm theta))
				(cddar m))))
	       (if a
		   (dbsubst-simul (cdr a) sub 0)
		 (evar (cadar m) sub))))
	    (t m))
    m))

(defun hash-by-consts (m nmlit)
  (dolist (c (remove-duplicates (free-consts m)))
    (push (cons m nmlit) (gethash c *consts-in-hash*))
    (dolist (lit-h-hdef *local-defns*)
      (when (eq c (cadr lit-h-hdef))
	(let* ((lit (car lit-h-hdef))
	       (cdef (caddr lit-h-hdef))
	       (m2 (norm (const-subst m c cdef)))
	       (m2lit (get-literal m2)))
	  (format nil "~d ~d ~d" lit nmlit m2lit)
	  (agenda-en (list 'DELAY *local-defns-delay-post* (list 'PROP m2)))
	  )))))

(defun free-consts (m)
  (if (consp (car m))
      (case (caar m)
	    (AP (append (free-consts (cadar m))
			(free-consts (cddar m))))
	    (LAM (free-consts (cdar m)))
	    (EVAR
	     (let ((vl nil))
	       (dolist (mm (cddar m) vl)
		 (setq vl (append (free-consts mm) vl)))))
	    (t nil))
    (let ((s (car m)))
      (if (and (symbolp s) (get s 'const))
	  (list s)
	nil))))

(defun distinct-dbs-p (sub &optional dl)
  (if sub
      (if (numberp (caar sub))
	  (if (member (caar sub) dl :test #'equal)
	      nil
	    (distinct-dbs-p (cdr sub) (cons (caar sub) dl)))
	nil)
    t))

(defun strict-evars (m)
  (if (consp (car m))
      (case (caar m)
	    (AP (append (strict-evars (cadar m))
			(strict-evars (cddar m))))
	    (LAM (strict-evars (cdar m)))
	    (EVAR
	     (if (distinct-dbs-p (cddar m))
		 (list (cadar m))
	       nil))
	    (t nil))
    nil))

(defun strict-p (Xl m)
  (subsetp Xl (strict-evars m)))

(defun free-evars (m)
  (if (consp (car m))
      (case (caar m)
	    (AP (append (free-evars (cadar m))
			(free-evars (cddar m))))
	    (LAM (free-evars (cdar m)))
	    (EVAR (list (cadar m)))
	    (t nil))
    nil))

(defun new-pattern-rules (nmlit m)
  (catch 'pattern-rule-failed
    (new-pattern-rules-1 nmlit (list m) nil nil)))

(defun new-pattern-rules-1 (nmlit ml Xl aux)
  (if ml
      (let ((m (car ml)))
	(if (and (all-p m) (lam-p (ap-a m)))
	    (multiple-value-bind
	     (X Xsub)
	     (new-evar nil (cadr (ap-a m)))
	     (new-pattern-rules-1
	      nmlit
	      (cons (norm (dbsubst (lam-body (ap-a m)) Xsub 0))
		    (cdr ml))
	      (cons X Xl) aux))
	  (if (or-p m)
	      (new-pattern-rules-1
	       nmlit (cons (bin-l m) (cons (bin-r m) (cdr ml)))
	       Xl aux)
	    (if (neg-p m)
		(let ((n (ap-a m)))
		  (if (eq n *FALSE*) ; useless clause, stop here
		      (throw 'pattern-rule-failed nil)
		    (progn
		      (new-pattern-rules-1
		       nmlit (cdr ml) Xl
		       (cons m aux))
		      (when (and *enable-pattern-rules-diseqns-as-atoms* ; May 2010
				 (strict-p Xl m))
			(new-pattern-rules-neg
			 nmlit n Xl (append (cdr ml) aux))
			)
		      (unless (or (or-p n) (all-p n) (eqn-p n))
			; negative atom
			(when (strict-p Xl n)
			  (new-pattern-rules-neg
			   nmlit n Xl (append (cdr ml) aux)))))))
	      (if (eq m *FALSE*) ; useless lit
		  (new-pattern-rules-1
		   nmlit (cdr ml) Xl aux)
		(progn
		  (new-pattern-rules-1
		   nmlit (cdr ml) Xl (cons m aux))
		  (if (eqn-p m)
		      (let ((etp (cdr (ap-a m)))
			    (l (bin-l m))
			    (r (bin-r m)))
			(when (and *enable-pattern-rules-eqns-as-atoms* ; May 2010
				   (strict-p Xl m))
			  (new-pattern-rules-pos
			   nmlit m Xl (append (cdr ml) aux))
			  )
			(if (consp etp)
			    (multiple-value-bind
			     (X Xsub)
			     (new-evar nil (car etp))
			     (new-pattern-rules-1
			      nmlit
			      (cons (norm (eqn (ap l Xsub) (ap r Xsub)))
				    (cdr ml))
			      (cons X Xl) aux))
			  (if (eq etp *O*)
			      (progn
				(new-pattern-rules-1
				 nmlit
				 (cons l (cons (myneg r) (cdr ml)))
				 Xl aux)
				(new-pattern-rules-1
				 nmlit
				 (cons (myneg l) (cons r (cdr ml)))
				 Xl aux)
				)
			    (progn
			      (when (strict-p Xl l)
				(new-pattern-rules-eqn
				 etp nmlit l r Xl (append (cdr ml) aux)))
			      (when (strict-p Xl r)
				(new-pattern-rules-eqn
				 etp nmlit r l Xl (append (cdr ml) aux)))))))
		    (unless (or (or-p m) (all-p m))
		      ; positive atom
		      (when (strict-p Xl m)
			(new-pattern-rules-pos
			 nmlit m Xl (append (cdr ml) aux)))))))))))))

(defun new-pattern-rules-neg (lit m Xl aux)
  (when *debug*
    (format t "** New neg pattern rule for ~d:~%" lit)
    (format t "Evars: ~d~%" Xl)
    (format t "Neg: ~d~%" (wff-str m))
    (format t "Aux:~%")
    (dolist (mj aux)
      (format t "~d~%" (wff-str mj))))
  (push (list lit m Xl aux) *pattern-rules-neg*)
  (maphash #'(lambda (key val)
	       (dolist (e val)
		 (apply-pattern-rule (cadr e) (car e) lit m Xl aux)))
	   *posatm-usable*))

(defun new-pattern-rules-pos (lit m Xl aux)
  (when *debug*
    (format t "** New pos pattern rule for ~d:~%" lit)
    (format t "Evars: ~d~%" Xl)
    (format t "Pos: ~d~%" (wff-str m))
    (format t "Aux:~%")
    (dolist (mj aux)
      (format t "~d~%" (wff-str mj))))
  (push (list lit m Xl aux) *pattern-rules-pos*)
  (maphash #'(lambda (key val)
	       (dolist (e val)
		 (apply-pattern-rule (cadr e) (car e) lit m Xl aux)))
	   *negatm-usable*))

(defun new-pattern-rules-eqn (tp lit l r Xl aux)
  (when *debug*
    (format t "** New eqn pattern rule for ~d:~%" lit)
    (format t "Evars: ~d~%" Xl)
    (format t "from: ~d~%" (wff-str l))
    (format t "to: ~d~%" (wff-str r))
    (format t "Aux:~%")
    (dolist (mj aux)
      (format t "~d~%" (wff-str mj))))
  (push (list tp lit l r Xl aux) *pattern-rules-eqn*)
  (dolist (e (gethash tp *negeq-usable*))
    (apply-pattern-rule-eqn tp (cadr e) (caddr e) (cadddr e) lit l r Xl aux))
  )

(defun apply-pattern-rules-neg (nlit m)
  (dolist (pr *pattern-rules-neg*)
    (apply-pattern-rule nlit m (car pr) (cadr pr) (caddr pr) (cadddr pr))
    ))

(defun apply-pattern-rules-pos (nlit m)
  (dolist (pr *pattern-rules-pos*)
    (apply-pattern-rule nlit m (car pr) (cadr pr) (caddr pr) (cadddr pr))
    ))

(defun apply-pattern-rules-eqn (nlit tp l r)
  (dolist (pr *pattern-rules-eqn*)
    (when (eq (car pr) tp)
      (apply-pattern-rule-eqn tp nlit l r (cadr pr) (caddr pr) (cadddr pr) (cadddr (cdr pr)) (cadddr (cddr pr))))
    ))

(defun pattern-match (dpairs &optional delayed theta)
  (if dpairs
      (let ((ctx (caar dpairs))
	    (m (cadar dpairs))
	    (n (caddar dpairs))
	    (tp (cadddr (car dpairs))))
	(if (consp tp)
	    (pattern-match (cons (list (cons (car tp) ctx)
				       (genlam-body m)
				       (genlam-body n)
				       (cdr tp))
				 (cdr dpairs))
			   delayed theta)
	  (if (evar-p m)
	      (if (distinct-dbs-p (cddar m))
		  (or
		   (catch 'dbsubst-simul-failed
		     (let* ((n2 (dbsubst-simul n (pattern-invert-sub ctx (cddar m)) 0)) ; m is X[sig] where sig:ctx -> ctx', X:ctx' |- tp.  ctx |- n:tp. pattern-invert-sub(sig):ctx' -> ctx.  ctx' |- n2[pattern-invert-sub(sig)]:tp
			    (Xn2 (acons (cadar m) n2 nil)))
		       (pattern-match
			(mapcar #'(lambda (dpair)
				    (list (car dpair)
					  (norm (evar-subst (cadr dpair) Xn2))
					  (norm (evar-subst (caddr dpair) Xn2))
					  (cadddr dpair)))
				(append delayed (cdr dpairs)))
			nil
			(acons (cadar m) n2 theta)
			)))
		   (throw 'pattern-match-failed nil))
		(pattern-match (cdr dpairs) (cons (car dpairs) delayed) theta))
	    (let ((mh (head m))
		  (margs (args m))
		  (nh (head n))
		  (nargs (args n)))
	      (if (eq mh nh)
		  (pattern-match
		   (append (mapcar #'(lambda (mj nj)
				       (list ctx mj nj (cdr mj))
				       )
				   margs nargs)
			   (cdr dpairs))
		   delayed theta)
		(throw 'pattern-match-failed nil))))))
    theta))

(defun apply-pattern-rule (lit1 m1 lit2 m2 Xl aux)
  (catch 'pattern-match-failed
    (let ((theta (pattern-match (cons (list nil m2 m1 *O*) nil))))
      (when *debug*
	(format t "Pattern Match Succeeded on~%~d~%~d~%theta:" (wff-str m2) (wff-str m1))
	(dolist (Xn theta)
	  (format t "~d := ~d~%" (car Xn) (wff-str (cdr Xn)))))
      (let ((cl (format nil "~d ~d" lit2 lit1)))
	(dolist (mj aux)
	  (let ((mjth (norm (evar-subst mj theta))))
	    (setq cl (format nil "~d ~d" cl (get-literal mjth)))
	    (agenda-en
	     (list 'DELAY *pattern-rules-delay*
		   (list 'PROP mjth)))))
	(append-clause cl)
	))))

(defun apply-pattern-rule-eqn (tp lit1 l1 r1 lit2 l r Xl aux)
  (catch 'pattern-match-failed
    (let ((theta (pattern-match (cons (list nil l l1 tp) nil))))
      (when *debug*
	(format t "Pattern Match Succeeded on~%~d~%~d~%r = ~d~%theta:" (wff-str l) (wff-str l1) (wff-str r))
	(dolist (Xn theta)
	  (format t "~d := ~d~%" (car Xn) (wff-str (cdr Xn)))))
      (let* ((rthr1 (diseqn (norm (evar-subst r theta)) r1))
	     (cl (format nil "~d ~d ~d" lit2 lit1 (get-literal rthr1))))
	(agenda-en
	 (list 'DELAY *pattern-rules-eqn-delay*
	       (list 'PROP rthr1)))
	(dolist (mj aux)
	  (let ((mjth (norm (evar-subst mj theta))))
	    (setq cl (format nil "~d ~d" cl (get-literal mjth)))
	    (agenda-en
	     (list 'DELAY *pattern-rules-delay*
		   (list 'PROP mjth)))))
	(append-clause cl)
	)))
  (catch 'pattern-match-failed
    (let ((theta (pattern-match (cons (list nil l r1 tp) nil))))
      (when *debug*
	(format t "Pattern Match Succeeded on~%~d~%~d~%r = ~d~%theta:" (wff-str l) (wff-str r1) (wff-str r))
	(dolist (Xn theta)
	  (format t "~d := ~d~%" (car Xn) (wff-str (cdr Xn)))))
      (let* ((rthl1 (diseqn (norm (evar-subst r theta)) l1))
	     (cl (format nil "~d ~d ~d" lit2 lit1 (get-literal rthl1))))
	(agenda-en
	 (list 'DELAY *pattern-rules-eqn-delay*
	       (list 'PROP rthl1)))
	(dolist (mj aux)
	  (let ((mjth (norm (evar-subst mj theta))))
	    (setq cl (format nil "~d ~d" cl (get-literal mjth)))
	    (agenda-en
	     (list 'DELAY *pattern-rules-delay*
		   (list 'PROP mjth)))))
	(append-clause cl)
	))))

(defun new-pattern-clauses (nalllit m)
  (new-pattern-clauses-2 nalllit (list m) nil nil nil "")
  )

(defun new-pattern-clauses-2 (nalllit ml evars strict unmatched abslits)
  (if ml
      (let ((m (car ml)))
	(if (or-p m) ; only disjunctions are always split, everything else can be a literal of a clause
	    (new-pattern-clauses-2 nalllit
				   (cons (bin-l m) (cons (bin-r m) (cdr ml)))
				   evars strict unmatched abslits)
	  (progn
	    (if (free-evars m)
		(new-pattern-clauses-2 nalllit (cdr ml)
				       evars
				       (append (strict-evars m) strict)
				       (cons m unmatched)
				       abslits)
					; if there are no free-evars in the formula, put it on abslits, not unmatched
		(new-pattern-clauses-2 nalllit (cdr ml)
				       evars strict
				       unmatched
				       (format nil " ~d~d" (get-literal m) abslits)))
	    (when (and (all-p m) (lam-p (ap-a m))) ; or make it an evar
	      (multiple-value-bind
	       (X Xsub)
	       (new-evar nil (cadr (ap-a m)))
	       (new-pattern-clauses-2 nalllit
				      (cons (norm (dbsubst (lam-body (ap-a m)) Xsub 0)) (cdr ml))
				      (cons X evars) strict
				      unmatched abslits))))))
; check that there are evars [hence unmatched] and that strict c= evars
    (when (and evars (subsetp evars strict))
      (when *verbose*
	(format t "Creating new pattern clause for ~d: ~d ~d~%" (- nalllit) evars abslits)
	(dolist (z unmatched)
	  (format t ". ~d~%" (wff-str z))))
      (let ((c (list nalllit evars unmatched nil abslits)))
	(new-pattern-clause-3 c)))))

(defun new-pattern-clause-3 (c)
  (dolist (mlit-m *pattern-clauses-usable*) ; apply it to those I have
    (apply-pattern-clause c (car mlit-m) (cdr mlit-m))
    )
  (push c *pattern-clauses*))

(defun apply-pattern-clauses (mlit m)
  (when *debug* (format t "apply-pattern-clauses called with mlit ~d~%" mlit))
  (push (cons mlit m) *pattern-clauses-usable*) ; for later clauses
  (dolist (c *pattern-clauses*)
    (apply-pattern-clause c mlit m)))

(defun apply-pattern-clause (c mlit m)
  (when *debug*
    (format t "apply-pattern-clause called with mlit ~d and clause ~d ~d~%" mlit (car c) (cadr c))
    (dolist (z (caddr c))
      (format t ". ~d~%" (wff-str (evar-subst z (cadddr c))))))
  (let ((unmatched (caddr c))
	(theta (cadddr c)))
    (dolist (n unmatched)
      (catch 'pattern-match-failed
	(when *debug*
	  (format t "trying to pattern match~%~d~%~d~%" (wff-str n) (wff-str m)))
	(let* ((theta2 (pattern-match (list (list nil n m *O*)) nil theta))
	       (evars2 (set-difference (cadr c) (mapcar #'car theta2))))
	  (if evars2 ; new pattern clause, partially instantiated
	      (let ((c2 (list (car c) evars2 (remove n unmatched) theta2
			      (format nil " ~d~d" (- mlit) (nth 4 c)))))
		(when *verbose* (format t "Partially instantiated pattern clause with remaining evars ~d~%" evars2))
		(new-pattern-clause-3 c2))
					; otherwise, done, process the ground clause
	    (apply-pattern-clause-2 (car c) (remove n unmatched)
				    theta2
				    (format nil " ~d~d" (- mlit) (nth 4 c))
				    )))))))

(defun apply-pattern-clause-2 (nalllit unmatched theta abslits)
  (if unmatched
      (let ((g (evar-subst (car unmatched) theta)))
	(agenda-en (list 'DELAY *pattern-clauses-delay* (list 'PROP g))) ; consider unmatched proposition later
	(apply-pattern-clause-2 nalllit
				(cdr unmatched)
				theta
				(format nil " ~d~d" (get-literal g) abslits)))
    (progn
      (when *verbose* (format t "Grounded pattern clause to obtain ~d ~d~%" nalllit abslits))
      (append-clause (format nil "~d~d" nalllit abslits))
      )))

; a bit of higher-order insanity
(defun comp-param (n a m)
  (if (> n 0)
      (let ((cpn (comp-param (- n 1) a m)))
	#'(lambda (g)
	    #'(lambda (x)
		(funcall cpn (funcall a g x)))))
    #'(lambda (g)
	(funcall m g))))

(defun hol2sat-search ()
  (let ((ag nil)
	(i 0))
    (loop while (setq ag (agenda-de)) do
;	  (when *debug* (setq *ag* ag)) ; uncomment to debug
	  (incf i)
	  (when (= i *call-period*) ; call minisat periodically
	    (when *debug* (format t "Calling Minisat~%"))
	    (call-minisat)
	    (setq i 0)
	    )
	  (when *debug* (format t "Agenda Item: ~d~%" ag))
;	  (when *debug* (format t "Remaining Agenda: ~S~%" (car *agenda*)))
	  (case (car ag)
		(DELAY
		 (decf i) ; never count these as steps
		 (if (> (cadr ag) 0)
		     (agenda-en (list 'DELAY (- (cadr ag) 1) (caddr ag)))
		   (agenda-push (caddr ag)))) ; do it now
		(PROP ; handle another proposition that may come onto the branch
		 (if (gethash (cadr ag) *handled*) ; skip -- don't count this as a step
		     (decf i)
		   (let* ((m (cadr ag)))
		     (when *debug* (format t "~d~%" (wff-str m)))
		     (setf (gethash m *handled*) t) ; set it as handled
		     (if (neg-p m)
			 (let* ((n (ap-a m))
				(nlit (get-literal n))) ; nlit is <n>
			   (when *enable-pattern-clauses* (apply-pattern-clauses nlit n)) ; May 2010 ; - ah, this seems to be the bug in r 85
			   (cond ((or-p n) ; m is ~n is ~(n1 | n2) : 2 clauses: <n1 | n2> | -<n1> and <n1 | n2> | -<n2>
				  (let* ((nn1 (myneg (bin-l n)))
					 (nn2 (myneg (bin-r n)))
					 (nn1lit (get-literal nn1))
					 (nn2lit (get-literal nn2)))
				    (append-clause (format nil "~d ~d" nlit nn1lit))
				    (append-clause (format nil "~d ~d" nlit nn2lit))
				    (agenda-push (list 'DELAY *post-nor-l-delay* (list 'PROP nn1))) ; eager terminating part
				    (agenda-push (list 'DELAY *post-nor-r-delay* (list 'PROP nn2))) ; eager terminating part
				    ))
				 ((all-p n) ; m is ~n is ~(!:(a>o)>o.pred) : for a fresh x of type a, add a clause <n> | <~(pred x)>
				  (let ((a (cdar (ap-f n)))
					(pred (ap-a n))
					(x (intern (format nil "__~d" (incf *nextfresh*)))))
				    (declare-const x a)
				    (let* ((npredx (norm (neg (ap pred (get x 'consth)))))
					   (npredxlit (get-literal npredx)))
				      (append-clause (format nil "~d ~d" nlit npredxlit))
				      (agenda-push (list 'DELAY *exists-delay* (list 'PROP npredx)))))) ; delay this so that enumerating instantiations can keep up with new constants
				 ((eq n *FALSE*) nil) ; ~false...do nothing
				 ((eqn-p n)
				  (when (and *enable-pattern-rules* *enable-pattern-rules-eqns-as-atoms*) (apply-pattern-rules-pos nlit n)) ; May 2010
				  (let ((etp (cdr (ap-a n)))
					(l (bin-l n))
					(r (bin-r n)))
				    (when *enable-pattern-rules*
				      (apply-pattern-rules-eqn nlit etp l r)
				      )
				    (when (eq l r)
				      (append-clause (format nil "~d" nlit)))
				    (if (consp etp) ; funcext, m is ~n is ~(l =_(a>b) r) : for a fresh x of type a, add clause <n> | <~(lx =_b rx)>
					(let* ((a (car etp))
					       (x (intern (format nil "__~d" (incf *nextfresh*)))))
					  (declare-const x a)
					  (let* ((nlxrx (neg (eqn (norm (ap l (get x 'consth))) (norm (ap r (get x 'consth))))))
						 (nlxrxlit (get-literal nlxrx)))
					    (append-clause (format nil "~d ~d" nlit nlxrxlit))
					    (agenda-push (list 'DELAY *post-nfeq-delay* (list 'DELAY *exists-delay* (list 'PROP nlxrx))))) ; delay this so that enumerating instantiations can keep up with new constants
					  (when *instantiate-with-func-diseqn-sides*
					  ; now - include l and r as special instantiations of type etp
					    (new-instantiation etp l)
					    (new-instantiation etp r))
					  )
				      (if (eq etp *O*) ; boolext, m is ~n is ~(l =_o r) : add clauses <n> | <l> | <r> and <n> | <~l> | <~r>
					  (let* ((nl (myneg l))
						 (nr (myneg r))
						 (llit (get-literal l))
						 (rlit (get-literal r))
						 (nllit (get-literal nl))
						 (nrlit (get-literal nr)))
					    (append-clause (format nil "~d ~d ~d" nlit llit rlit))
					    (append-clause (format nil "~d ~d ~d" nlit nllit nrlit))
					    (agenda-en (list 'DELAY *post-neqo-l-delay* (list 'PROP l)))
					    (agenda-en (list 'DELAY *post-neqo-r-delay* (list 'PROP r)))
					    (agenda-en (list 'DELAY *post-neqo-nl-delay* (list 'PROP nl)))
					    (agenda-en (list 'DELAY *post-neqo-nr-delay* (list 'PROP nr)))
					    )
					(progn ; m is ~n is ~(l =_a r) : decompose and confront
					  (when *enable-local-defns* (hash-by-consts m nlit))
					  (push (list n nlit l r) (gethash etp *negeq-usable*)) ; put on usable list for later pos eqns to confront against
					  (unless (member l (gethash etp *discriminating*))
					    (when *debug* (format t "New Discriminating Term: ~d~%" (wff-str l)))
					    (push l (gethash etp *discriminating*))
					    (dolist (au (gethash etp *univ-usable*))
					      (let* ((nulit (cadr au))
						     (pred (caddr au))
						     (predtrm (norm (ap pred l))))
						(append-clause (format nil "~d ~d" nulit (get-literal predtrm)))
						(agenda-en (list 'DELAY *forall-delay* (list 'PROP predtrm))))))
					  (unless (member r (gethash etp *discriminating*))
					    (when *debug* (format t "New Discriminating Term: ~d~%" (wff-str r)))
					    (push r (gethash etp *discriminating*))
					    (dolist (au (gethash etp *univ-usable*))
					      (let* ((nulit (cadr au))
						     (pred (caddr au))
						     (predtrm (norm (ap pred r))))
						(append-clause (format nil "~d ~d" nulit (get-literal predtrm)))
						(agenda-en (list 'DELAY *forall-delay* (list 'PROP predtrm))))))
					; if decomposition applies with l1...ln and r1...rn, add clause <n> | <~(l1 = r1)> | ... | <~(ln = rn)>
					  (let ((lh (head l))
						(rh (head r)))
					    (when (eq lh rh) ; if this is a logical constant it has to be a choice operator -- so decompose in any case -- may block it if I add if-then-else operators
					      (when *debug* (format t "Decomposing ~d != ~d~%" (wff-str l) (wff-str r)))
					      (let* ((largs (args l))
						     (rargs (args r))
						     (clause (format nil "~d" nlit))
						     (nlrargs (mapcar #'(lambda (la ra) (neg (eqn la ra))) largs rargs)))
						(dolist (nlrarg nlrargs)
						  (agenda-en (list 'DELAY *post-dec-delay* (list 'PROP nlrarg)))
						  (setq clause (format nil "~d ~d" clause (get-literal nlrarg))))
						(append-clause clause)))
					    (when (choiceop-p lh) (choice-clause lh (car (args l))))
					    (when (choiceop-p rh) (choice-clause rh (car (args r)))))
					; for each p : u =_a v  in *poseq-usable*, add clauses:
					; <n> | <~p> | <~(u = l)> | <~(u = r)>
					; <n> | <~p> | <~(u = l)> | <~(v = r)>
					; <n> | <~p> | <~(v = l)> | <~(u = r)>
					; <n> | <~p> | <~(v = l)> | <~(v = r)>
					  (let ((peql (gethash etp *poseq-usable*)))
					    (dolist (peq peql)
					      (let* ((nplit (cadr peq))
						     (u (caddr peq))
						     (v (cadddr peq)))
						(consider-confrontation nlit nplit u v l r)
						))))))))
				 (t ; m is ~n is ~(h ...args...)
				  (when *enable-pattern-rules*
				    (apply-pattern-rules-pos nlit n))
				  (when *enable-local-defns* (hash-by-consts m nlit))
				  (let* ((h (head n))
					 (args (args n))
					 (patml (gethash h *posatm-usable*))
					 (clause nil))
				    (when (choiceop-p h) (choice-clause h (car args)))
				    ; for each p : h ...pargs... in patml add clause <n> | <~p> | <~(parg1 = arg1)> | ... <~(pargn = argn)>
				    ; only do this when there are args
				    (when args
				      (push (list n nlit args) (gethash h *negatm-usable*)) ; save for later positive atoms with h at the head
				      (dolist (p patml)
					(let* ((nplit (cadr p))
					       (pargs (caddr p))
					       (diseqns (mapcar #'(lambda (parg narg) (neg (eqn parg narg))) pargs args)))
					  (setq clause (format nil "~d ~d" nlit nplit))
					  (dolist (diseqn diseqns)
					    (agenda-en (list 'DELAY *post-mating-delay* (list 'PROP diseqn)))
					    (setq clause (format nil "~d ~d" clause (get-literal diseqn))))
					  (append-clause clause))))))
				 ))
		       (let ((nlit (get-literal (neg m))))
			 (when (and (not (or-p m)) *enable-pattern-clauses*) (apply-pattern-clauses (- nlit) (neg m))) ; May 2010
			 (cond ((or-p m) ; m is (m1 | m2) : 1 clause: -<m> | <m1> | <m2>
				(let* ((m1 (bin-l m))
				       (m2 (bin-r m))
				       (m1lit (get-literal m1))
				       (m2lit (get-literal m2)))
				  (append-clause (format nil "~d ~d ~d" nlit m1lit m2lit))
				  (agenda-push (list 'DELAY *post-or-l-delay* (list 'PROP m1))) ; eager terminating part
				  (agenda-push (list 'DELAY *post-or-r-delay* (list 'PROP m2))) ; eager terminating part
				  ))
			       ((all-p m) ; m is (!:(a>o)>o.pred) : for each closed instantiation t of type a add a clause <~m> | <pred t>
				(when *enable-pattern-rules*
				  (new-pattern-rules nlit m))
				(when *enable-pattern-clauses* (new-pattern-clauses nlit m)) ; May 2010
				(when *enable-local-defns*
				  (new-local-defn nlit m)
				  )
				(let ((a (cdar (ap-f m)))
				      (pred (ap-a m)))
				  (push (list m nlit pred) (gethash a *univ-usable*))
				  (if (consp a) ; function types
				      (let ((rtp (rtp a)))
					(unless (or (eq rtp '*O*) (default-elt-p rtp))
					  (agenda-en (list 'DELAY *defaultelt-delay* (list 'DEFAULTELT rtp))))
					(unless (gethash a *generate-instantiations*)
					  (setf (gethash a *generate-instantiations*) t)
					  (agenda-en (list 'DELAY *enum-start* (list 'ENUM (list (list nil a)) #'identity a))) ; eventually start a process of enumerating instances
					  ; May 5, 2010 - Special instantiations
					  (dolist (e *enum-special*)
					    (let ((tpvars (car e))
						  (tp1 (cadr e))
						  (tplist1 (caddr e))
						  (f (nth 3 e))
						  (delay1 (nth 4 e))
						  (delay2 (nth 5 e)))
					      (multiple-value-bind
					       (succ tpsubst)
					       (tp-match tp1 a)
					       (when succ
						 (when *debug* (format t "Special instantiation match for type ~d~%" a))
						 (if tplist1
						     (catch 'tp-subst-bug
						       (agenda-en (list 'DELAY delay1
									(list 'ENUM-SPECIAL
									      (mapcar #'(lambda (tpvar)
											  (or (cdr (assoc tpvar tpsubst))
											      (throw 'tp-subst-bug nil)))
										      tpvars)
									      (mapcar #'(lambda (b) (tp-subst b tpsubst)) tplist1)
									      nil
									      f
									      delay2))))
						   (agenda-en (list 'DELAY delay1
								    (list 'NEW-INST a (apply f (mapcar #'cdr tpsubst))))))))))
					  )
					(dolist (trm (gethash a *instantiations*)) ; special instantiations
					  (let ((predtrm (norm (ap pred trm))))
					    (append-clause (format nil "~d ~d" nlit (get-literal predtrm)))
					    (agenda-en (list 'DELAY *forall-delay* (list 'PROP predtrm)))))) ; come back to this later
				    (if (eq a *O*) ; boolean type - instantiate with false and ~false
					(let ((predfalse (norm (ap pred *FALSE*)))
					      (predtrue (norm (ap pred (neg *FALSE*)))))
					  (append-clause (format nil "~d ~d" nlit (get-literal predfalse)))
					  (append-clause (format nil "~d ~d" nlit (get-literal predtrue)))
					  (agenda-en (list 'DELAY *forall-delay* (list 'PROP predfalse))) ; come back to this later
					  (agenda-en (list 'DELAY *forall-delay* (list 'PROP predtrue)))) ; come back to this later
				      (let ((dl (gethash a *discriminating*))) ; for each discriminating term t:a, add a clause <~m> | <pred t>
					(if dl
					    (dolist (trm dl)
					      (let ((predtrm (norm (ap pred trm))))
						(append-clause (format nil "~d ~d" nlit (get-literal predtrm)))
						(agenda-en (list 'DELAY *forall-delay* (list 'PROP predtrm))))) ; come back to this later
					  (progn
					    (when *debug* (format t "No Discriminating Terms for ~S...may use default element~%" a))
					    (agenda-en (list 'DELAY *defaulteltinst-delay* (list 'DEFAULTELTINST a)))))))))) ; wait until next time before creating a default element of the sort...maybe a discriminant will show up.
			       ((eq m *FALSE*) (append-clause (format nil "~d" nlit))) ; false may be on the branch...put negated literal for false on the branch to take care of this.
			       ((eqn-p m) ; positive eqn
				(when (and *enable-pattern-rules* *enable-pattern-rules-diseqns-as-atoms*) (apply-pattern-rules-neg nlit m)) ; May 2010
				(when *enable-local-defns*
				  (new-local-defn nlit m)
				  )
				(let ((etp (cdr (ap-a m)))
				      (l (bin-l m))
				      (r (bin-r m)))
				  (if (consp etp) ; funceq, turn it into forall and = at lower type, this way I get the restriction on instantiation appropriate for domain type
				      (let* ((d0 (db 0 (car etp)))
					     (mall (ap (lpi (car etp)) (lam (car etp) (eqn (norm (ap (shift l 1 0) d0)) (norm (ap (shift r 1 0) d0)))))) ; !x.[lx]=[rx]
					     (malllit (get-literal mall)))
					(append-clause (format nil "~d ~d" nlit malllit))
					(agenda-push (list 'DELAY *post-feq-delay* (list 'PROP mall))))
				    (if (eq etp *O*) ; booleq, m is (l =_o r) : <~m> | <l> | <~r> and <~m> | <~l> | <r>
					(let* ((nl (myneg l))
					       (nr (myneg r))
					       (llit (get-literal l))
					       (rlit (get-literal r))
					       (nllit (get-literal nl))
					       (nrlit (get-literal nr)))
					  (append-clause (format nil "~d ~d ~d" nlit llit nrlit))
					  (append-clause (format nil "~d ~d ~d" nlit nllit rlit))
					  (agenda-en (list 'DELAY *post-eqo-l-delay* (list 'PROP l)))
					  (agenda-en (list 'DELAY *post-eqo-r-delay* (list 'PROP r)))
					  (agenda-en (list 'DELAY *post-eqo-nl-delay* (list 'PROP nl)))
					  (agenda-en (list 'DELAY *post-eqo-nr-delay* (list 'PROP nr)))
					  )
				      (progn
					(when *enable-local-defns* (hash-by-consts m nlit))
					(push (list m nlit l r) (gethash etp *poseq-usable*))
					; for each n : u =_a v  in *negeq-usable*, add clauses:
					; <n> | <~p> | <~(l = u)> | <~(r = u)>
					; <n> | <~p> | <~(l = u)> | <~(r = v)>
					; <n> | <~p> | <~(l = v)> | <~(r = u)>
					; <n> | <~p> | <~(l = v)> | <~(r = v)>
					(let ((neql (gethash etp *negeq-usable*)))
					  (dolist (neq neql)
					      (let* ((nnlit (cadr neq))
						     (u (caddr neq))
						     (v (cadddr neq)))
						(consider-confrontation nnlit nlit l r u v)))))))))
			       (t
				(when *enable-pattern-rules*
				  (apply-pattern-rules-neg nlit m))
				(when *enable-local-defns* (hash-by-consts m nlit))
				(let* ((h (head m))
				       (args (args m))
				       (natml (gethash h *negatm-usable*))
				       (clause nil))
				  (when (choiceop-p h) (choice-clause h (car args)))
				  (when args
				    (push (list m nlit args) (gethash h *posatm-usable*)) ; save for later negative atoms with h at the head
				    (dolist (n natml)
				      (let* ((nnlit (cadr n))
					     (nargs (caddr n))
					     (diseqns (mapcar #'(lambda (parg narg) (neg (eqn parg narg))) args nargs)))
					(setq clause (format nil "~d ~d" nnlit nlit))
					(dolist (diseqn diseqns)
					  (agenda-en (list 'DELAY *post-mating-delay* (list 'PROP diseqn)))
					  (setq clause (format nil "~d ~d" clause (get-literal diseqn))))
					(append-clause clause))))))
			       ))))))
		(CONFRONT
		 (let* ((nnlit (nth 1 ag))
			(nplit (nth 2 ag))
			(u (nth 3 ag))
			(v (nth 4 ag))
			(l (nth 5 ag))
			(r (nth 6 ag))
			(nul (neg (eqn u l)))
			(nur (neg (eqn u r)))
			(nvl (neg (eqn v l)))
			(nvr (neg (eqn v r)))
			(nullit (get-literal nul))
			(nurlit (get-literal nur))
			(nvllit (get-literal nvl))
			(nvrlit (get-literal nvr)))
		   (when *debug* (format t "Confronting ~d = ~d with ~d != ~d~%" (wff-str u) (wff-str v) (wff-str l) (wff-str r)))
		   (append-clause (format nil "~d ~d ~d ~d" nnlit nplit nullit nurlit))
		   (append-clause (format nil "~d ~d ~d ~d" nnlit nplit nullit nvrlit))
		   (append-clause (format nil "~d ~d ~d ~d" nnlit nplit nvllit nurlit))
		   (append-clause (format nil "~d ~d ~d ~d" nnlit nplit nvllit nvrlit))
		   (agenda-en (list 'DELAY *post-confront1-delay* (list 'PROP nul)))
		   (agenda-en (list 'DELAY *post-confront2-delay* (list 'PROP nur)))
		   (agenda-en (list 'DELAY *post-confront3-delay* (list 'PROP nvl)))
		   (agenda-en (list 'DELAY *post-confront4-delay* (list 'PROP nvr)))))
		(DEFAULTELT ; create a default element at a sort
		  (let ((a (cadr ag)))
		    (default-elt a))) ; get things started with a default elt
		(DEFAULTELTINST ; create a default element at a sort
		  (let ((a (cadr ag)))
		    (let ((m (default-elt a))) ; get instantiations started with a default elt
		      (unless (gethash a *discriminating*)
			(dolist (mnp (gethash a *univ-usable*))
			  (let* ((nlit (cadr mnp))
				 (pred (caddr mnp))
				 (predm (norm (ap pred m))))
			    (append-clause (format nil "~d ~d" nlit (get-literal predm)))
			    (agenda-en (list 'DELAY *forall-delay* (list 'PROP predm)))))))))
		(ENUM-TP ; Work on enumerating type for some polymorphic head
		 (let ((tpevars (nth 1 ag))
		       (h (nth 2 ag))
		       (ectx (nth 3 ag))
		       (evars (nth 4 ag))
		       (m (nth 5 ag))
		       (tp (nth 6 ag)))
		   (if tpevars ; (<foo> ...) where <foo> is either a base type or nil (no return base type specified)
		       (progn ; either an arrow type, o or a sort
					; arrow type, this is horrible, so don't do it often
			 (agenda-en (list 'DELAY *enum-arrow*
					  (list 'ENUM-TP
						(cons nil (cons (car tpevars) (cdr tpevars)))
						#'(lambda (a) #'(lambda (b) (funcall h (ar a b))))
						ectx evars m tp)))
			 ; otherwise base type -- specified or not
			 ; o
			 (when (or (not (car tpevars)) (eq (car tpevars) *O*))
			   (agenda-en (list 'DELAY *enum-o* (list 'ENUM-TP (cdr tpevars)
							   (funcall h *O*)
							   ectx evars m tp))))
					; a sort
			 (dolist (a *sorts*)
			   (when (or (not (car tpevars)) (eq (car tpevars) a))
			     (agenda-en (list 'DELAY *enum-sort* (list 'ENUM-TP (cdr tpevars)
							     (funcall h a)
							     ectx evars m tp))))))
		     (agenda-push (list 'ENUM-AP h ectx evars m tp))))); should call ap?
		(ENUM-AP
		 (let ((g (cadr ag))
		       (ectx (nth 2 ag))
		       (evars (nth 3 ag))
		       (newevars nil)
		       (m (nth 4 ag))
		       (tp (nth 5 ag))
		       (n 0))
		   (do ((b (cdr g) (cdr b)))
		       ((not (consp b)))
		       (push (list ectx (car b)) newevars)
		       (incf n))
		   (agenda-en (list 'ENUM (append (reverse newevars) evars) (funcall (comp-param n #'ap m) g) tp))))
		(ENUM ; Work on enumerating instantations
		 (let ((evars (nth 1 ag))
		       (m (nth 2 ag))
		       (tp (nth 3 ag)))
		   (if evars ; now evars should be a list of (<tplist[dbctx]> <tp>)
		       (let ((ectx (caar evars))
			     (etp (cadar evars)))
			 (if (consp etp) ; function type -- reduce -- no need to delay this at all
			     (let ((a (car etp)))
			       (agenda-push (list 'ENUM (cons (list (cons (car etp) ectx) (cdr etp)) (cdr evars))
						  #'(lambda (z) (funcall m (lam a z)))
						  tp)))
			   (let ((j 0))
					; projections and imitations, also put a copy to fire when new constants are declared
			     (dolist (b ectx)
			       (when (eq (rtp b) etp)
				 (agenda-push (list 'DELAY *project-delay* (list 'ENUM-AP (db j b) ectx (cdr evars) m tp))))
			       (incf j))
			     ; enumerate type for a choice function, then imitate that choice function (must return base type etp)
			     (agenda-en (list 'DELAY *enum-choice* (list 'ENUM-TP (list etp) #'lepsilon ectx (cdr evars) m tp)))
			     (when (eq etp *O*)
			       (agenda-en (list 'DELAY *enum-neg* (list 'ENUM-AP *NEG* ectx (cdr evars) m tp))) ; primsub for negation
			       (agenda-en (list 'DELAY *enum-or* (list 'ENUM-AP *OR* ectx (cdr evars) m tp))) ; primsub for or
			       (agenda-en (list 'DELAY *enum-false* (list 'ENUM-AP *FALSE* ectx (cdr evars) m tp))) ; primsub for false
			       (dolist (a *sorts*)
				 (agenda-en (list 'DELAY *enum-eq* (list 'ENUM-AP (leq a) ectx (cdr evars) m tp))))) ; primsub for = at each sort
			     (push (list ectx (cdr evars) m tp) (gethash etp *waiting-enums*))
			     (dolist (c *consts*)
			       (let ((b (get c 'const))
				     (brtp (get c 'rtp)))
				 (when (eq brtp etp)
				   (if (get c 'defh)
				       (when *imitate-defns*
					 (agenda-push (list 'DELAY *imitate-defn-delay* (list 'ENUM-AP (get c 'defh) ectx (cdr evars) m tp))))
				     (if (get c 'consth)
					 (agenda-push (list 'DELAY *imitate-delay* (list 'ENUM-AP (get c 'consth) ectx (cdr evars) m tp)))
				       (progn
;					 (setq *c* c) (break) ; uncomment to debug
					 nil
					 )
				       ))))))))
					; otherwise done...m should be a term of the right type
		     (progn
		       (when *debug*
			 (format t "Generated Instantiation of type ~d: ~d~%" (tp-str tp) (wff-str m)))
		       (agenda-push
			(list 'DELAY
			      (if (insane-instance-p m)
				  5000 ; delay it for a very long time -- probably should justify omitting it altogether
				0)
			      (list 'NEW-INST tp m)))))))
		(ENUM-SPECIAL
		 (let* ((tpargs (cadr ag))
			(tplist (caddr ag))
			(tp (car tplist))
			(args (cadddr ag))
			(f (nth 4 ag))
			(delay (nth 5 ag)))
		   (if (cdr tplist)
		       (dolist (c *consts*)
			 (when (eq (get c 'const) tp)
			   (agenda-en
			    (list 'DELAY delay
				  (list 'ENUM-SPECIAL tpargs (cdr tplist) (cons (get c 'consth) args) f delay)))))
		     (dolist (c *consts*)
		       (when (eq (get c 'const) tp)
			 (let ((m (apply f (append tpargs (reverse (cons (get c 'consth) args))))))
			   (agenda-en
			    (list 'DELAY delay
				  (list 'NEW-INST (cdr m) m)))))))))
		(NEW-INST
		 (new-instantiation (cadr ag) (caddr ag))
		 )
		(t (break))))
    (call-minisat-final)
    ))

					; if m has a double negation or a choice where body does not depend on bd var, then m is insane
(defun insane-instance-p (m)
  (if (double-negation-p m)
      t
    (if (irrelevant-choice-p m)
	t
      nil)))

(defun double-negation-p (m)
  (if (consp (car m))
      (if (and (neg-p m) (neg-p (cddar m)))
	  t
	(case (caar m)
	      (AP (or (double-negation-p (ap-f m))
		      (double-negation-p (ap-a m))))
	      (LAM (double-negation-p (lam-body m)))
	      ; EVAR can't happen here
	      (t nil)))
    nil))

(defun irrelevant-choice-p (m)
  (if (consp (car m))
      (if (and (choice-p m) (lam-p (ap-a m)) (not (member 0 (dbfrees (lam-body (ap-a m))) :test #'equal)))
	  t
	(case (caar m)
	      (AP (or (irrelevant-choice-p (ap-f m))
		      (irrelevant-choice-p (ap-a m))))
	      (LAM (irrelevant-choice-p (lam-body m)))
	      ; evar can't happen here
	      (t nil)))
    nil))

; May 5, 2010
(defun tp-match (a b &optional tpsubst)
  (if (consp a)
      (if (consp b)
	  (multiple-value-bind
	   (succ1 tpsubst1)
	   (tp-match (car a) (car b) tpsubst)
	   (if succ1
	       (tp-match (cdr a) (cdr b) tpsubst1)
	     nil))
	nil)
    (if (eq a *O*)
	(if (eq a b)
	    (values t tpsubst)
	  nil)
      (let ((a2 (assoc a tpsubst)))
	(if a2
	    (if (eq (cdr a2) b)
		(values t tpsubst)
	      nil)
	  (values t (acons a b tpsubst)))))))

(defun tp-subst (a tpsubst)
  (if (consp a)
      (ar (tp-subst (car a) tpsubst)
	  (tp-subst (cdr a) tpsubst))
    (if (eq a *O*)
	*O*
      (let ((a2 (assoc a tpsubst)))
	(if a2
	    (cdr a2)
	  (progn
	    (when *debug* (format t "Type subst problem : ~S~%~S~%" a tpsubst))
	    (throw 'tp-subst-bug nil)))))))

(defun agenda-init ()
  (setq *agenda* nil)
  )

					; put at beginning (only for terminating part)
(defun agenda-push (x)
;  (when *debug* (format t "Agenda Item Push: ~S~%" x))
  (if *agenda*
      (push x (car *agenda*))
    (let ((q (list x)))
      (setq *agenda* (cons q q)))))

					; put at the end
(defun agenda-en (x)
;  (when *debug* (format t "Agenda Item Enqueue: ~S~%" x))
  (if *agenda*
      (progn
	(rplacd (cdr *agenda*) (list x))
	(pop (cdr *agenda*)))
    (let ((q (list x)))
      (setq *agenda* (cons q q)))))

					; pop first
(defun agenda-de ()
  (if *agenda*
      (let ((r (pop (car *agenda*))))
	(unless (car *agenda*)
	  (setq *agenda* nil))
	r)
    nil))

					; return (<name> . ((<tp> . o) . <tp>)) if m is a normal formula saying <name> is a choice operator for <tp>
					; !p:(<tp> -> o).(!x.~p x) | (p (<name> p)))
					; otherwise return nil
(defun choice-fn-axiom-p (m)
  (if (and (all-p m) (lam-p (ap-a m)))
      (let ((dom (lam-dom (ap-a m))))
	(if (and (consp dom) (eq (cdr dom) *O*))
	    (let ((tp (car dom))
		  (body (lam-body (ap-a m))))
	      (if (or-p body)
		  ; I suppose I could also check for the symmetric version...but I'm not for now.
		  (let ((l (bin-l body)) ; !:<tp>.~1 0
			(r (bin-r body))) ; 0 (<name> 0)
		    (if (and (ap-p r)
			     (ap-p (ap-a r))
			     (equal (car (ap-f r)) 0)
			     (equal (car (ap-a (ap-a r))) 0)
			     (symbolp (car (ap-f (ap-a r)))) ; this is what to return
			     (all-p l)
			     (lam-p (ap-a l))
			     (eq (lam-dom (ap-a l)) tp)
			     (neg-p (lam-body (ap-a l)))
			     (ap-p (ap-a (lam-body (ap-a l))))
			     (equal (car (ap-f (ap-a (lam-body (ap-a l))))) 1)
			     (equal (car (ap-a (ap-a (lam-body (ap-a l))))) 0))
			(ap-f (ap-a r))
		      nil))
		nil))
	  nil))
    nil))

(defun hol2sat-init ()
  (setq *searching* t)
  (agenda-init)
  (setq *posatm-usable* (make-hash-table)) ; indexed by head
  (setq *negatm-usable* (make-hash-table)) ; indexed by head
  (setq *poseq-usable* (make-hash-table)) ; indexed by sort
  (setq *negeq-usable* (make-hash-table)) ; indexed by sort
  (setq *univ-usable* (make-hash-table)) ; universals on usable [list of predicates indexed by type]
  (setq *choice-done* (make-hash-table :test #'equal))
  (setq *discriminating* (make-hash-table)) ; discriminating terms on branch [list of terms indexed by base type]
  (setq *instantiations* (make-hash-table))
  (setq *generate-instantiations* (make-hash-table))
  (setq *waiting-enums* (make-hash-table)) ; index by sort
  (setq *handled* (make-hash-table)) ; start with an empty branch [t/nil indexed by wff] -- branch is sos U usable
  (setq *clauses* "")
  (setq *minisat-timeout* *minisat-timeout-init*)
  (setq *local-defns* nil)
  (setq *consts-in-hash* (make-hash-table))
  (setq *pattern-rules-pos* nil)
  (setq *pattern-rules-neg* nil)
  (setq *pattern-rules-eqn* nil)
  (setq *pattern-clauses* nil)
  (setq *pattern-clauses-usable* nil)
  )

(defun hol2sat-readfile (p &optional props)
  (let ((s (read p nil nil)))
    (if s
	(let ((c ; (catch 'thf-fail
		   (if (and (consp s) (equal (car s) "thf") (consp (cdr s)) (consp (cddr s)) (consp (cdddr s)))
		       (case (intern (caddr s))
			     (|type|
			      (let ((f (cadddr s)))
				(if (and (consp f) (equal (car f) ":") (consp (cdr f)) (stringp (cadr f)) (consp (cddr f)))
				    (if (equal (caddr f) "$tType")
					(declare-basetype (intern (cadr f)))
				      (declare-const (intern (cadr f)) (stp (caddr f))))
				  (thf-fail "" s)))
			      (hol2sat-readfile p props)
			      )
			     (|definition|
			      (let ((f (cadddr s)))
				(declare-def (intern (cadr f)) (trm (caddr f)))
				)
			      (hol2sat-readfile p props)
			      )
			     ((|hypothesis| |axiom| |lemma|)
			      (let* ((m (trm (cadddr s))))
				(if (eq (cdr m) *O*)
				    (let ((mn (preprocess (norm m)))
					  (sp nil)) ; special cases
				      (cond ((setq sp (choice-fn-axiom-p mn)) ; in the special case where we are assuming some name is a choice function, then don't add as an axiom, but mark it as a choice fn so that the choice rules will apply to it...I don't make it the same choice function for soundness reasons. - Chad, Dec 2009
					     (let ((tp (cddr sp))) ; sp : (A -> o) -> A.  tp := A
					       (when *verbose* (format t "~d is a choice function~%" sp))
					       (setf (get (car sp) 'CHOICEFN) (lepsilon tp))
					       (hol2sat-readfile p props)))
					    (t
;					     (let ((mnl (get-literal mn)))
;					       (append-clause mnl)
;					       (agenda-en (list 'PROP mn)) ; May 10 2010 potentially split disjs first
					       (hol2sat-readfile p (cons mn props))))
				      )
				  (thf-fail "axiom does not have type o" s)
				  )))
			     ((|conjecture| |theorem|)
			      (when *conjecture-given* (thf-fail "Multiple conjectures are not allowed in a single problem" s))
			      (let* ((m (trm (cadddr s))))
				(if (eq (cdr m) *O*)
				    (let* ((mn (preprocess (norm (neg m)))))
;					   (mnl (get-literal mn)))
;				      (append-clause mnl)
;				      (agenda-en (list 'PROP mn)) ; May 10 2010 potentially split disjs first
				      (setq *conjecture-given* t)
				      (hol2sat-readfile p (cons mn props)))
				  (thf-fail "conjecture does not have type o" s)
				  )))
			     (t (thf-fail "" s)))
		     (thf-fail "" s))))
	  (declare (ignore c))
					; I can only be here if I'm catching thf-fail, continue reading file
	  (hol2sat-readfile p props))
      (progn
	(close p)
	(if *split-global-disjunctions*
	    (split-global-disjunctions props)
	  (progn
	    (hol2sat-init) ; May 10 2010 - moved this to be here
	    (dolist (m (reverse props))
	      (append-clause (get-literal m))
	      (agenda-en (list 'PROP m)))
	    (hol2sat-search)))))))

(defun split-global-disjunctions (props &optional props2)
  (if props
      (let ((m (car props)))
	(cond ((or-p m)
	       (when *verbose* (format t "Splitting global disjunction...~%"))
	       (let ((foo (catch 'inconsistent
			    (split-global-disjunctions (cons (bin-l m) (cdr props)) props2))))
		 (push *clauses* *clauses-disjs*)
		 (setq *searching* nil) ; to prevent search-related activities when constants are declared in the ~! case
		 (split-global-disjunctions (cons (bin-r m) (cdr props)) props2)) ; solved this one, now solve the other
	       )
	      ((and (eqn-p m) (eq (cdr (ap-a m)) *O*)) ; eqn at type bool, convert to disj of conjs so \/ case applies
	       (let ((l (bin-l m))
		     (r (bin-r m)))
		 (split-global-disjunctions (cons (disj (neg (disj (myneg l) (myneg r))) (neg (disj l r)))
						  (cdr props)) props2)))
	      ((neg-p m)
	       (let ((n (ap-a m)))
		 (cond ((or-p n) ; conjunction, separate it (to get at possible disjuncts below)
			(split-global-disjunctions (cons (myneg (bin-l n)) (cons (myneg (bin-r n)) (cdr props))) props2))
		       ((all-p n) ; negative forall, can go ahead and put in a witness and continue
			(let ((a (cdar (ap-f n)))
			      (pred (ap-a n))
			      (x (intern (format nil "__~d" (incf *nextfresh*)))))
			  (declare-const x a)
			  (let* ((npredx (norm (neg (ap pred (get x 'consth))))))
			    (split-global-disjunctions (cons npredx (cdr props)) props2))))
		       ((and (eqn-p n) (eq (cdr (ap-a n)) *O*)) ; diseqn at type bool, convert to disj of conjs so \/ case applies
			(let ((l (bin-l n))
			      (r (bin-r n)))
			  (split-global-disjunctions (cons (disj (neg (disj l (myneg r))) (neg (disj (myneg l) r)))
							   (cdr props)) props2)))
		       ((and (eqn-p n) (consp (cdr (ap-a n)))) ; diseqn at func type, convert with FE so the ~! case applies
			(let* ((l (bin-l n))
			       (r (bin-r n))
			       (dom (cadr l)))
			  (split-global-disjunctions (cons
						      (norm (neg (all dom (eqn (ap l (db 0 dom)) ; There should never be any dangling db's in l and r here, so I'm not shifting.
									       (ap r (db 0 dom))))))
						      (cdr props)) props2)))
		       (t
			(split-global-disjunctions (cdr props) (cons m props2))))))
	      (t (split-global-disjunctions (cdr props) (cons m props2)))))
    (progn
      (when *verbose* (format t "Searching on one branch~%"))
      (hol2sat-init) ; May 10 2010 - moved this to be here (needs to be reset for each search)
      (dolist (m props2)
	(let ((sp (choice-fn-axiom-p m))) ; in the special case where we are assuming some name is a choice function, then don't add as an axiom, but mark it as a choice fn so that the choice rules will apply to it...I don't make it the same choice function for soundness reasons. - Chad, Dec 2009 -- including this in this preprocessing of subproblems - Chad, May 2010
	  (if sp
	      (let ((tp (cddr sp))) ; sp : (A -> o) -> A.  tp := A
		(when *verbose* (format t "~d is a choice function~%" sp))
		(setf (get (car sp) 'CHOICEFN) (lepsilon tp))
		)
	    (progn
	      (append-clause (get-literal m))
	      (agenda-en (list 'PROP m))))))
      (hol2sat-search))))

(defun print-atom-atlas ()
  (maphash #'(lambda (key value)
	       (format t "Atom ~d: ~d~%" value (wff-str key)))
	   *atomhash*))

(defun hol2sat-file (problemfile)
  (let ((p (open problemfile :direction :input)))
;    (hol2sat-init) ; May 10 2010 - moved this to be right before search starts
    (let ((foo
	   (catch 'inconsistent
	     (catch 'done
	       (hol2sat-readfile p)))))
      (when *verbose* (format t "Number of formulas considered: ~d~%" *nextatom*)) ; May 2010
      (if (eq foo 'inconsistent)
	  (let ((r 20))
	    (if *conjecture-given*
		(format t "% SZS status Theorem~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
	      (progn
		(format t "% SZS status Unsatisfiable~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
		(setq r 25)))
	    (when *print-final-proof-info*
	      (format t "% SZS output start Proof~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
	      (if (and *split-global-disjunctions* *clauses-disjs*)
		  (let ((i 1))
		    (format t "Solved as ~d independent problems~%" (1+ (length *clauses-disjs*)))
		    (format t "Clause Set 1:~%")
		    (write-string *clauses*)
		    (dolist (cl *clauses-disjs*)
		      (format t "Clause Set ~d:~%" (incf i))
		      (write-string cl)))
		(progn
		  (format t "Clauses:~%")
		  (write-string *clauses*)
		  ))
	      (format t "Atoms:~%")
	      (print-atom-atlas)
	      (format t "% SZS output end Proof~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
	      )
	    r)
	(if (eq foo 'satisfiable)
	    (let ((r 10))
	      (if *conjecture-given*
		  (format t "% SZS status CounterSatisfiable~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
		(progn
		  (format t "% SZS status Satisfiable~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
		  (setq r 15)
		  ))
	      (when *print-final-proof-info* ; Decided to also print the 'proof' here -- a satisfiable set of clauses.  A satisfying assignment induces an evident set, which in turn induces a (finite) model - Chad, Jan 2010
		(format t "% SZS output start Proof~%")
		(format t "Clauses:~%")
		(write-string *clauses*)
		(format t "Atoms:~%")
		(print-atom-atlas)
		(format t "% SZS output end Proof~%")
		)
	      r
	      )
	  (if (eq foo 'thf-fail)
	      (progn
		(format t "% SZS status Error : (THF Error)~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
		2)
	    (progn
	      (format t "% SZS status Error~%") ; Geoff Sutcliffe, Jan 2010 - SZS standard output
	      3)))))))

